<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\User;
use App\Models\UserActivity;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UserManagementController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasPermission('access_admin_dashboard')) {
                abort(403, 'Unauthorized access to admin panel');
            }
            return $next($request);
        });
    }

    /**
     * Display user management page
     */
    public function index(Request $request)
    {
        if (!auth()->user()->hasPermission('view_users')) {
            abort(403, 'You do not have permission to view users');
        }

        $query = User::with(['role', 'createdByAdmin']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('employee_id', 'like', "%{$search}%");
            });
        }

        // Filter by role
        if ($request->filled('role')) {
            $query->whereHas('role', function ($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by department
        if ($request->filled('department')) {
            $query->where('department', $request->department);
        }

        $users = $query->latest()->paginate(15);
        $roles = Role::active()->get();
        $departments = \App\Models\Department::active()->pluck('name');

        return view('admin.users.index', compact('users', 'roles', 'departments'));
    }

    /**
     * Store a new user
     */
    public function store(Request $request)
    {
        if (!auth()->user()->hasPermission('create_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to create users'], 403);
        }

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'department' => 'nullable|string|exists:departments,name',
            'employee_id' => 'nullable|string|unique:users,employee_id',
            'role_id' => 'required|exists:roles,id',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        // Generate employee ID if not provided
        if (empty($validated['employee_id'])) {
            $validated['employee_id'] = $this->generateEmployeeId();
        }

        // Generate temporary password
        $tempPassword = Str::random(12);
        
        $user = User::create([
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'name' => $validated['first_name'] . ' ' . $validated['last_name'],
            'email' => $validated['email'],
            'password' => Hash::make($tempPassword),
            'phone' => $validated['phone'],
            'department' => $validated['department'],
            'employee_id' => $validated['employee_id'],
            'role_id' => $validated['role_id'],
            'status' => $validated['status'],
            'first_login' => true,
            'created_by_admin_id' => auth()->id(),
            'email_verified_at' => now(),
        ]);

        // Log activity
        UserActivity::log('create_user', "Created user: {$user->full_name}", $user, [
            'user_id' => $user->id,
            'temp_password' => $tempPassword // For admin reference (consider security implications)
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User created successfully!',
            'temp_password' => $tempPassword,
            'user' => $user->load('role')
        ]);
    }

    /**
     * Update user information
     */
    public function update(Request $request, User $user)
    {
        if (!auth()->user()->hasPermission('edit_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to edit users'], 403);
        }

        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'department' => 'nullable|string|exists:departments,name',
            'employee_id' => 'nullable|string|unique:users,employee_id,' . $user->id,
            'role_id' => 'required|exists:roles,id',
            'status' => 'required|in:active,inactive,suspended',
        ]);

        $originalData = $user->toArray();
        
        $user->update([
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'name' => $validated['first_name'] . ' ' . $validated['last_name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'department' => $validated['department'],
            'employee_id' => $validated['employee_id'],
            'role_id' => $validated['role_id'],
            'status' => $validated['status'],
        ]);

        // Log activity
        UserActivity::log('update_user', "Updated user: {$user->full_name}", $user, [
            'user_id' => $user->id,
            'changes' => array_diff_assoc($validated, $originalData)
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully!',
            'user' => $user->fresh()->load('role')
        ]);
    }

    /**
     * Delete a user
     */
    public function destroy(User $user)
    {
        if (!auth()->user()->hasPermission('delete_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to delete users'], 403);
        }

        // Prevent self-deletion
        if ($user->id === auth()->id()) {
            return response()->json(['success' => false, 'message' => 'You cannot delete your own account'], 400);
        }

        $userName = $user->full_name;
        $userId = $user->id;

        // Delete avatar if exists
        if ($user->avatar && Storage::exists('public/' . $user->avatar)) {
            Storage::delete('public/' . $user->avatar);
        }

        $user->delete();

        // Log activity
        UserActivity::log('delete_user', "Deleted user: {$userName}", null, [
            'deleted_user_id' => $userId
        ]);

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully!'
        ]);
    }

    /**
     * Reset user password
     */
    public function resetPassword(User $user)
    {
        if (!auth()->user()->hasPermission('edit_users')) {
            return response()->json(['success' => false, 'message' => 'You do not have permission to reset passwords'], 403);
        }

        $tempPassword = Str::random(12);
        
        $user->update([
            'password' => Hash::make($tempPassword),
            'first_login' => true,
        ]);

        // Log activity
        UserActivity::log('reset_password', "Reset password for user: {$user->full_name}", $user);

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully!',
            'temp_password' => $tempPassword
        ]);
    }

    /**
     * Generate unique employee ID
     */
    private function generateEmployeeId(): string
    {
        do {
            $id = 'EMP' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        } while (User::where('employee_id', $id)->exists());

        return $id;
    }
}
