<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\User;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\LpoManagement;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\QualityInspectionChecklist;
use App\Models\Warehouse\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class IncomingOperationController extends Controller
{
    /**
     * Display a listing of incoming operations
     */
    public function index(Request $request)
    {
        $query = IncomingOperation::with(['supplier', 'project', 'division', 'receiver', 'inspector', 'materialRequest'])
                                 ->latest('operation_date');

        // Apply filters
        if ($request->filled('operation_type')) {
            $query->where('operation_type', $request->operation_type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('division_id')) {
            $query->where('division_id', $request->division_id);
        }

        if ($request->filled('date_from')) {
            $query->where('operation_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('operation_date', '<=', $request->date_to);
        }

        if ($request->filled('mr_reference')) {
            $query->whereHas('materialRequest', function($q) use ($request) {
                $q->where('material_request_number', 'like', '%' . $request->mr_reference . '%');
            });
        }

        $operations = $query->paginate(15);

        // Get filter options
        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();

        return view('warehouse.incoming-operations.index', compact(
            'operations', 'suppliers', 'projects', 'divisions'
        ));
    }

    /**
     * Show the form for creating a new incoming operation
     */
    public function create(Request $request)
    {
        $operationType = $request->get('type', 'supplier_delivery');

        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $inspectors = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_manager', 'warehouse_supervisor']);
        })->get();
        $lpos = LpoManagement::where('status', 'approved')->get();

        // Debug: Log the counts for troubleshooting
        \Log::info('Incoming Operations Create Data Loaded:', [
            'suppliers_count' => $suppliers->count(),
            'projects_count' => $projects->count(),
            'divisions_count' => $divisions->count(),
            'inspectors_count' => $inspectors->count(),
            'lpos_count' => $lpos->count(),
            'operation_type' => $operationType
        ]);

        // Get currency exchange rates (this could be from a service)
        $exchangeRates = [
            'AED' => 1.0000,
            'USD' => 3.6725, // Example rate
            'EUR' => 4.0250  // Example rate
        ];

        return view('warehouse.incoming-operations.create', compact(
            'operationType', 'suppliers', 'projects', 'divisions',
            'inspectors', 'lpos', 'exchangeRates'
        ));
    }

    /**
     * Store a newly created incoming operation
     */
    public function store(Request $request)
    {
        // Debug logging
        \Log::info('Supplier Delivery Store Method Called', [
            'all_data' => $request->all(),
            'operation_type' => $request->operation_type,
            'items_count' => count($request->items ?? []),
            'method' => $request->method()
        ]);

        // Set default values for currency and exchange_rate if not provided
        if (!$request->has('currency') || empty($request->currency)) {
            $request->merge(['currency' => 'AED']);
        }
        if (!$request->has('exchange_rate') || empty($request->exchange_rate)) {
            $request->merge(['exchange_rate' => 1.0]);
        }

        // Different validation rules for different operation types
        $validationRules = [
            'operation_type' => 'required|in:supplier_delivery,site_return,emergency_procurement',
            'operation_date' => 'required|date',
            'received_by' => 'required|exists:users,id',

            // Files
            'delivery_note_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'photos.*' => 'nullable|file|mimes:jpg,jpeg,png|max:5120'
        ];

        // Site return specific validation
        if ($request->operation_type === 'site_return') {
            $validationRules = array_merge($validationRules, [
                'material_request_id' => 'required_without_all:direct_delivery_id,material_transfer_id|exists:outgoing_transactions,id',
                'direct_delivery_id' => 'required_without_all:material_request_id,material_transfer_id|exists:direct_deliveries,id',
                'material_transfer_id' => 'required_without_all:material_request_id,direct_delivery_id|exists:material_transfer_requests,id',
                'project_id' => 'required|exists:projects,id',
                'division' => 'nullable|string|max:255',
                'currency' => 'required|in:AED,USD,EUR',
                'exchange_rate' => 'required|numeric|min:0.0001',

                // Items for site return
                'items' => 'required|array|min:1',
                'items.*.item_id' => 'required|exists:items,id',
                'items.*.outgoing_item_id' => 'nullable|exists:outgoing_items,id',
                'items.*.direct_delivery_item_id' => 'nullable|exists:direct_delivery_items,id',
                'items.*.material_transfer_item_id' => 'nullable|exists:material_transfer_request_items,id',
                'items.*.quantity_returned' => 'required|numeric|min:0.01',
                'items.*.condition' => 'required|string|in:good,fair,damaged',
                'items.*.return_reason' => 'nullable|string',
                'items.*.notes' => 'nullable|string',

                // Force close validation
                'force_close_return' => 'nullable|boolean',
                'force_close_reason' => 'nullable|required_if:force_close_return,1|string'
            ]);
        } else {
            // Standard operation validation (supplier delivery, etc.)
            $validationRules = array_merge($validationRules, [
                'division' => 'required|string|max:255',
                'supplier_id' => 'required_if:operation_type,supplier_delivery|exists:suppliers,id',
                'lpo_number' => 'nullable|string',
                'delivery_note_number' => 'nullable|string',
                'currency' => 'required|in:AED,USD,EUR',
                'exchange_rate' => 'required|numeric|min:0.0001',

                // Items for standard operations
                'items' => 'required|array|min:1',
                'items.*.item_id' => 'required|exists:items,id',
                'items.*.quantity_delivered' => 'required|numeric|min:0.01',
                'items.*.unit_price' => 'required|numeric|min:0',
                'items.*.batch_number' => 'nullable|string',
                'items.*.production_date' => 'nullable|date',
                'items.*.expiry_date' => 'nullable|date'
            ]);
        }

        try {
            $request->validate($validationRules);
            \Log::info('Validation passed for operation type: ' . $request->operation_type);
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation failed for supplier delivery', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            throw $e;
        }

        // Debug logging for site returns
        if ($request->operation_type === 'site_return') {
            \Log::info('Site Return Submission Debug', [
                'project_id' => $request->project_id,
                'material_request_id' => $request->material_request_id,
                'direct_delivery_id' => $request->direct_delivery_id,
                'material_transfer_id' => $request->material_transfer_id,
                'items_count' => count($request->items ?? []),
                'all_request_data' => $request->all()
            ]);
        }

        // Additional validation for site returns to ensure data integrity
        if ($request->operation_type === 'site_return') {
            if (empty($request->material_request_id) && empty($request->direct_delivery_id) && empty($request->material_transfer_id)) {
                return back()->withErrors(['reference' => 'Either Material Request, Direct Delivery, or Material Transfer must be selected.']);
            }

            if (empty($request->project_id)) {
                return back()->withErrors(['project_id' => 'Project must be selected for site returns.']);
            }

            if (empty($request->items) || count($request->items) == 0) {
                return back()->withErrors(['items' => 'At least one item must be returned.']);
            }
        }

        DB::beginTransaction();

        try {
            // Create the operation

            $operation = IncomingOperation::create([
                'operation_type' => $request->operation_type,
                'operation_date' => $request->operation_date,
                'status' => 'draft',
                'supplier_id' => $request->supplier_id,
                'material_request_id' => $request->material_request_id,
                'direct_delivery_id' => $request->direct_delivery_id,
                'material_transfer_id' => $request->material_transfer_id,
                'project_id' => $request->project_id,
                'division' => $request->division,
                'lpo_number' => $request->lpo_number,
                'currency' => $request->currency,
                'exchange_rate' => $request->exchange_rate,
                'delivery_note_number' => $request->delivery_note_number,
                'force_close_return' => $request->boolean('force_close_return', false),
                'force_close_reason' => $request->force_close_reason,
                'inspector_id' => $request->inspector_id,
                'vat_rate' => 5.00, // UAE VAT rate
                'total_amount' => 0, // Initial value, will be updated after items are added
                'received_by' => $request->received_by,
                'received_by_name' => $request->received_by_name,
                'notes' => $request->notes
            ]);


            // Handle delivery note file upload
            if ($request->hasFile('delivery_note_file')) {
                $path = $request->file('delivery_note_file')->store(
                    'delivery-notes/' . $operation->id, 'public'
                );
                $operation->delivery_note_path = $path;
            }

            // Handle photo documentation
            $photoPaths = [];
            if ($request->hasFile('photos')) {
                foreach ($request->file('photos') as $photo) {
                    $photoPaths[] = $photo->store(
                        'incoming-operations/' . $operation->id . '/photos', 'public'
                    );
                }
                $operation->photo_documentation = $photoPaths;
            }

            // Create operation items
            $totalAmount = 0;

            foreach ($request->items as $index => $itemData) {

                // For site returns, use quantity_returned field and set default pricing
                if ($operation->operation_type === 'site_return') {
                    $quantity = $itemData['quantity_returned'] ?? 0;
                    $unitPrice = 0; // Returns don't have pricing
                    $unitPriceAed = 0;
                } else {
                    $quantity = $itemData['quantity_delivered'] ?? 0;
                    $unitPrice = $itemData['unit_price'] ?? 0;
                    $unitPriceAed = $unitPrice;
                    if ($request->currency !== 'AED') {
                        $unitPriceAed = $unitPrice * $request->exchange_rate;
                    }
                }

                $itemCreateData = [
                    'incoming_operation_id' => $operation->id,
                    'item_id' => $itemData['item_id'],
                    'quantity_delivered' => $quantity,
                    'unit_price' => $unitPrice,
                    'currency' => $request->currency,
                    'unit_price_aed' => $unitPriceAed,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'production_date' => $itemData['production_date'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ];

                // For site returns, link to the original outgoing item, direct delivery item, or material transfer item
                if ($operation->operation_type === 'site_return') {
                    if (isset($itemData['outgoing_item_id'])) {
                        $itemCreateData['outgoing_item_id'] = $itemData['outgoing_item_id'];
                    }
                    if (isset($itemData['direct_delivery_item_id'])) {
                        $itemCreateData['direct_delivery_item_id'] = $itemData['direct_delivery_item_id'];
                    }
                    if (isset($itemData['material_transfer_item_id'])) {
                        $itemCreateData['material_transfer_item_id'] = $itemData['material_transfer_item_id'];
                    }
                }

                // Skip items with zero quantity
                if ($quantity <= 0) {
                    continue;
                }

                $item = IncomingOperationItem::create($itemCreateData);

                // The model automatically calculates total_price_aed in the boot method
                $item->refresh(); // Reload to get calculated values
                $totalAmount += $item->total_price_aed;
            }

            // Update operation total amount
            $operation->total_amount = $totalAmount / $request->exchange_rate;
            $operation->save();

            // Handle site return - always auto-complete and update inventory immediately
            if ($operation->operation_type === 'site_return') {
                $returnStatus = $this->calculateSiteReturnStatus($operation, $request);
                $operation->status = 'completed'; // Always complete immediately
                $operation->approved_by = $request->received_by;
                $operation->approved_at = now();

                if ($operation->shouldForceClose()) {
                    $operation->approval_notes = 'Force closed: ' . $request->force_close_reason;
                } else {
                    $operation->approval_notes = 'Auto-completed site return (' . $returnStatus . ')';
                }

                $operation->save();

                // Always update inventory immediately for any site return
                $operation->processToInventory();
            }
            // For supplier delivery, bypass approvals and go directly to inventory
            elseif ($operation->operation_type === 'supplier_delivery') {
                $operation->status = 'completed';
                $operation->approved_by = $request->received_by;
                $operation->approved_at = now();
                $operation->approval_notes = 'Auto-approved: Direct processing for supplier delivery';
                $operation->save();

                // Update inventory immediately
                $operation->processToInventory();
            } else {
                // For other operation types, check if force close is enabled
                if ($operation->shouldForceClose()) {
                    $operation->status = 'completed';
                    $operation->approved_by = $request->received_by;
                    $operation->approved_at = now();
                    $operation->approval_notes = 'Force closed by user request';
                    $operation->save();

                    // Update inventory immediately
                    $operation->processToInventory();
                } else {
                    // Normal workflow - pending processing
                    $operation->status = 'pending_processing';
                    $operation->save();
                }
            }

            DB::commit();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Incoming operation created successfully.');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withInput()
                        ->with('error', 'Failed to create incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified incoming operation
     */
    public function show(IncomingOperation $operation)
    {
        $operation->load([
            'supplier', 'project', 'division', 'receiver',
            'inspector', 'approver', 'items.item', 'lpo', 'materialRequest', 'materialTransfer'
        ]);

        return view('warehouse.incoming-operations.show', compact('operation'));
    }

    /**
     * Show the form for editing the specified incoming operation
     */
    public function edit(IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot edit completed operations. Items have already been processed to inventory.');
        }

        $operation->load(['items.item']);

        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $inspectors = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_manager', 'warehouse_supervisor']);
        })->get();
        $lpos = LpoManagement::where('status', 'approved')->get();

        $exchangeRates = [
            'AED' => 1.0000,
            'USD' => 3.6725,
            'EUR' => 4.0250
        ];

        return view('warehouse.incoming-operations.edit', compact(
            'operation', 'suppliers', 'projects', 'divisions',
            'inspectors', 'lpos', 'exchangeRates'
        ));
    }

    /**
     * Update the specified incoming operation
     */
    public function update(Request $request, IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot update completed operations. Items have already been processed to inventory.');
        }

        $request->validate([
            'operation_date' => 'required|date',
            'division_id' => 'required|exists:departments,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'project_id' => 'nullable|exists:projects,id',
            'currency' => 'required|in:AED,USD,EUR',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'items' => 'required|array|min:1',
            'delivery_note_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240'
        ]);

        DB::beginTransaction();

        try {
            // Update operation
            $operation->update($request->only([
                'operation_date', 'division_id', 'supplier_id', 'project_id',
                'project_phase', 'site_location', 'lpo_number', 'currency',
                'exchange_rate', 'delivery_note_number', 'vehicle_number',
                'driver_name', 'driver_contact', 'delivery_datetime',
                'delivery_instructions', 'inspector_id', 'notes'
            ]));

            // Handle file upload
            if ($request->hasFile('delivery_note_file')) {
                // Delete old file
                if ($operation->delivery_note_path) {
                    Storage::disk('public')->delete($operation->delivery_note_path);
                }

                $path = $request->file('delivery_note_file')->store(
                    'delivery-notes/' . $operation->id, 'public'
                );
                $operation->delivery_note_path = $path;
                $operation->save();
            }

            // Update items
            $operation->items()->delete(); // Remove existing items
            $totalAmount = 0;

            foreach ($request->items as $itemData) {
                $unitPriceAed = $itemData['unit_price'];
                if ($request->currency !== 'AED') {
                    $unitPriceAed = $itemData['unit_price'] * $request->exchange_rate;
                }

                $item = IncomingOperationItem::create([
                    'incoming_operation_id' => $operation->id,
                    'item_id' => $itemData['item_id'],
                    'quantity_delivered' => $itemData['quantity_delivered'],
                    'unit_price' => $itemData['unit_price'],
                    'currency' => $request->currency,
                    'unit_price_aed' => $unitPriceAed,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                    'allocated_project_id' => $itemData['allocated_project_id'] ?? $request->project_id
                ]);

                $totalAmount += $item->total_price_aed;
            }

            // Update total amount
            $operation->total_amount = $totalAmount / $request->exchange_rate;
            $operation->save();

            DB::commit();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Incoming operation updated successfully.');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                        ->with('error', 'Failed to update incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified incoming operation
     */
    public function destroy(IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot delete completed operations. Items have already been processed to inventory.');
        }

        try {
            // Delete associated files
            if ($operation->delivery_note_path) {
                Storage::disk('public')->delete($operation->delivery_note_path);
            }

            if ($operation->photo_documentation) {
                foreach ($operation->photo_documentation as $photoPath) {
                    Storage::disk('public')->delete($photoPath);
                }
            }

            $operation->delete();

            return redirect()->route('warehouse.incoming-operations.index')
                           ->with('success', 'Incoming operation deleted successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Perform quality inspection
     */
    public function inspect(Request $request, IncomingOperation $operation)
    {
        if ($operation->status !== 'pending_inspection') {
            return back()->with('error', 'Operation is not pending inspection.');
        }

        $request->validate([
            'inspection_result' => 'required|in:passed,failed,conditional',
            'inspection_notes' => 'nullable|string',
            'items' => 'required|array',
            'items.*.id' => 'required|exists:incoming_operation_items,id',
            'items.*.quality_status' => 'required|in:passed,failed,conditional',
            'items.*.quality_notes' => 'nullable|string'
        ]);

        DB::beginTransaction();

        try {
            // Update operation inspection
            $operation->update([
                'status' => $request->inspection_result === 'passed' ? 'approved' : 'rejected',
                'inspection_datetime' => now(),
                'inspection_result' => $request->inspection_result,
                'inspection_notes' => $request->inspection_notes
            ]);

            // Update item quality status
            foreach ($request->items as $itemData) {
                $item = IncomingOperationItem::find($itemData['id']);
                $item->update([
                    'quality_status' => $itemData['quality_status'],
                    'quality_notes' => $itemData['quality_notes'] ?? null
                ]);
            }

            DB::commit();

            $message = $request->inspection_result === 'passed' ?
                      'Inspection completed. Operation approved.' :
                      'Inspection completed. Operation rejected.';

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Failed to complete inspection: ' . $e->getMessage());
        }
    }

    /**
     * Complete the operation and update inventory
     */
    public function complete(IncomingOperation $operation)
    {
        // Check if operation is already completed (for direct processing operations)
        if ($operation->status === 'completed') {
            return back()->with('info', 'Operation is already completed.');
        }

        if ($operation->status !== 'approved') {
            return back()->with('error', 'Operation must be approved before completion.');
        }

        try {
            $operation->complete();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Operation completed successfully. Inventory updated.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to complete operation: ' . $e->getMessage());
        }
    }

    /**
     * Print goods received note
     */
    public function print(IncomingOperation $operation)
    {
        $operation->load(['supplier', 'project', 'division', 'receiver', 'items.item']);

        return view('warehouse.incoming-operations.print', compact('operation'));
    }

    /**
     * Get items for AJAX requests
     */
    public function getItems(Request $request)
    {
        $search = $request->get('search');

        $items = Item::with('category')
                    ->where('status', 'active')
                    ->when($search, function ($query, $search) {
                        return $query->where('item_code', 'like', "%{$search}%")
                                   ->orWhere('item_description', 'like', "%{$search}%");
                    })
                    ->limit(20)
                    ->get();

        return response()->json($items);
    }

    /**
     * Get LPO details for AJAX requests
     */
    public function getLpoDetails(Request $request)
    {
        $lpoNumber = $request->get('lpo_number');

        $lpo = LpoManagement::with(['items.item', 'supplier', 'project'])
                           ->where('lpo_number', $lpoNumber)
                           ->first();

        if (!$lpo) {
            return response()->json(['error' => 'LPO not found'], 404);
        }

        return response()->json($lpo);
    }

    /**
     * Get quality checklists for material category
     */
    public function getQualityChecklists(Request $request)
    {
        $category = $request->get('category');

        $checklists = QualityInspectionChecklist::getForCategory($category);

        return response()->json($checklists);
    }

    /**
     * Display supplier delivery operations page
     */
    public function supplierDeliveryIndex(Request $request)
    {
        $allDeliveries = collect();

        // Get standard supplier deliveries (IncomingOperations)
        $operationsQuery = IncomingOperation::with(['supplier', 'items', 'receiver'])
                                 ->where('operation_type', 'supplier_delivery')
                                 ->latest('operation_date');

        // Apply filters for operations
        if ($request->filled('status')) {
            $operationsQuery->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $operationsQuery->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('division')) {
            $operationsQuery->where('division', $request->division);
        }

        if ($request->filled('lpo_status')) {
            $operationsQuery->where('lpo_status', $request->lpo_status);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $operationsQuery->whereBetween('operation_date', [$request->date_from, $request->date_to]);
        }

        if ($request->filled('amount_min') && $request->filled('amount_max')) {
            $operationsQuery->whereBetween('total_amount_aed', [$request->amount_min, $request->amount_max]);
        }

        $standardDeliveries = $operationsQuery->get();

        // Get direct delivery transactions (IncomingTransactions)
        $transactionsQuery = \App\Models\Warehouse\IncomingTransaction::with(['supplier', 'items', 'receiver', 'project'])
                                 ->where('transaction_type', 'direct_delivery')
                                 ->latest('transaction_date');

        // Apply filters for transactions
        if ($request->filled('status')) {
            $transactionsQuery->where('status', $request->status);
        }

        if ($request->filled('supplier_id')) {
            $transactionsQuery->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $transactionsQuery->whereBetween('transaction_date', [$request->date_from, $request->date_to]);
        }

        if ($request->filled('amount_min') && $request->filled('amount_max')) {
            $transactionsQuery->whereBetween('total_amount', [$request->amount_min, $request->amount_max]);
        }

        $directDeliveries = $transactionsQuery->get();

        // Format standard deliveries
        $formattedStandardDeliveries = $standardDeliveries->map(function ($delivery) {
            return (object) [
                'id' => $delivery->id,
                'type' => 'standard',
                'operation_number' => $delivery->operation_number,
                'operation_date' => $delivery->operation_date,
                'supplier' => $delivery->supplier,
                'status' => $delivery->status,
                'total_amount' => $delivery->total_amount_aed ?? 0,
                'items_count' => $delivery->items->count(),
                'receiver' => $delivery->receiver,
                'received_by_name' => $delivery->received_by_name,
                'lpo_number' => $delivery->lpo_number,
                'delivery_note_number' => $delivery->delivery_note_number,
                'division' => $delivery->division,
                'currency' => $delivery->currency,
                'created_at' => $delivery->created_at,
                'updated_at' => $delivery->updated_at
            ];
        });

        // Format direct deliveries
        $formattedDirectDeliveries = $directDeliveries->map(function ($delivery) {
            return (object) [
                'id' => 'DD_' . $delivery->id, // Prefix to differentiate
                'type' => 'direct_delivery',
                'operation_number' => $delivery->delivery_note_number ?: 'DD-' . $delivery->id,
                'operation_date' => $delivery->transaction_date,
                'supplier' => $delivery->supplier,
                'status' => $delivery->status,
                'total_amount' => $delivery->total_amount ?? 0,
                'items_count' => $delivery->items->count(),
                'receiver' => $delivery->receiver,
                'received_by_name' => $delivery->received_by_name,
                'lpo_number' => $delivery->lpo_number,
                'delivery_note_number' => $delivery->delivery_note_number,
                'division' => 'Direct to ' . ($delivery->project->project_name ?? 'Project'),
                'currency' => 'AED',
                'created_at' => $delivery->created_at,
                'updated_at' => $delivery->updated_at
            ];
        });

        // Combine and sort by operation_date
        $allDeliveries = collect($formattedStandardDeliveries)->merge(collect($formattedDirectDeliveries))
                                                   ->sortByDesc(function ($item) {
                                                       return $item->operation_date;
                                                   });

        // Paginate manually
        $perPage = 25;
        $currentPage = request()->input('page', 1);
        $paginatedItems = $allDeliveries->slice(($currentPage - 1) * $perPage, $perPage)->values();

        $deliveries = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedItems,
            $allDeliveries->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url()]
        );

        // Append query parameters to pagination links
        $deliveries->appends($request->all());

        // Get suppliers for filter
        $suppliers = Supplier::where('status', 'active')->get(['id', 'name']);

        // Calculate statistics (include both standard and direct deliveries)
        $standardDeliveryCount = IncomingOperation::where('operation_type', 'supplier_delivery')->count();
        $directDeliveryCount = \App\Models\Warehouse\IncomingTransaction::where('transaction_type', 'direct_delivery')->count();

        $standardPendingCount = IncomingOperation::where('operation_type', 'supplier_delivery')
                                                ->where('status', 'pending_delivery')->count();
        $directPendingCount = \App\Models\Warehouse\IncomingTransaction::where('transaction_type', 'direct_delivery')
                                                   ->where('status', 'pending')->count();

        $standardCompletedToday = IncomingOperation::where('operation_type', 'supplier_delivery')
                                                 ->where('status', 'completed')
                                                 ->whereDate('operation_date', today())->count();
        $directCompletedToday = \App\Models\Warehouse\IncomingTransaction::where('transaction_type', 'direct_delivery')
                                                     ->where('status', 'completed')
                                                     ->whereDate('transaction_date', today())->count();

        $standardTotalValue = IncomingOperation::where('operation_type', 'supplier_delivery')
                                             ->where('status', 'completed')
                                             ->sum('total_amount_aed') ?? 0;
        $directTotalValue = \App\Models\Warehouse\IncomingTransaction::where('transaction_type', 'direct_delivery')
                                                   ->where('status', 'completed')
                                                   ->sum('total_amount') ?? 0;

        $statistics = [
            'total_deliveries' => $standardDeliveryCount + $directDeliveryCount,
            'pending_delivery' => $standardPendingCount + $directPendingCount,
            'completed_today' => $standardCompletedToday + $directCompletedToday,
            'total_value_aed' => $standardTotalValue + $directTotalValue,
        ];

        // Get top suppliers (include both standard and direct deliveries)
        $topSuppliers = Supplier::withCount([
            'incomingOperations as standard_deliveries_count' => function ($query) {
                $query->where('operation_type', 'supplier_delivery');
            }
        ])->get()->map(function ($supplier) {
            // Count direct deliveries for this supplier
            $directDeliveryCount = \App\Models\Warehouse\IncomingTransaction::where('supplier_id', $supplier->id)
                                                         ->where('transaction_type', 'direct_delivery')
                                                         ->count();

            $supplier->deliveries_count = $supplier->standard_deliveries_count + $directDeliveryCount;
            return $supplier;
        })->sortByDesc(function ($supplier) {
            return $supplier->deliveries_count;
        })->take(5);

        return view('warehouse.supplier-delivery.index', compact(
            'deliveries',
            'suppliers',
            'statistics',
            'topSuppliers'
        ));
    }

    /**
     * Show the form for creating a new supplier delivery
     */
    public function supplierDeliveryCreate()
    {
        // Get suppliers for dropdown
        $suppliers = Supplier::where('status', 'active')
                            ->orderBy('name')
                            ->get(['id', 'name', 'supplier_code']);

        // Get users for received by dropdown
        $users = User::orderBy('name')
                    ->get(['id', 'name']);

        // Get items for dropdown
        $items = \App\Models\Warehouse\Item::where('status', 'active')
                            ->orderBy('item_description')
                            ->get(['id', 'item_code', 'item_description', 'unit_of_measure']);

        return view('warehouse.supplier-delivery.create', compact(
            'suppliers',
            'users',
            'items'
        ));
    }

    /**
     * Get supplier delivery data for AJAX
     */
    public function supplierDeliveryData(Request $request)
    {
        $query = IncomingOperation::with(['supplier', 'items', 'receiver'])
                                 ->where('operation_type', 'supplier_delivery');

        // Apply filters if provided
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Additional filters can be added here

        $deliveries = $query->latest('operation_date')->get();

        return response()->json([
            'data' => $deliveries,
            'recordsTotal' => $deliveries->count(),
            'recordsFiltered' => $deliveries->count()
        ]);
    }

    /**
     * Export supplier deliveries
     */
    public function exportSupplierDeliveries(Request $request)
    {
        // This is a placeholder for export functionality
        // You would implement actual export logic here (Excel, CSV, etc.)

        return response()->json([
            'success' => false,
            'message' => 'Export functionality will be implemented soon'
        ]);
    }

    /**
     * Display site return operations page
     */
    public function siteReturnIndex(Request $request)
    {
        // Get projects that have site returns with aggregated data
        $projectReturns = IncomingOperation::with(['project', 'project.projectDivision'])
            ->leftJoin('incoming_operation_items', 'incoming_operations.id', '=', 'incoming_operation_items.incoming_operation_id')
            ->where('incoming_operations.operation_type', 'site_return')
            ->select(
                \DB::raw('COALESCE(incoming_operations.project_id, 0) as project_id'),
                \DB::raw('COUNT(DISTINCT incoming_operations.id) as total_returns'),
                \DB::raw('COUNT(DISTINCT incoming_operations.material_request_id) as material_requests_count'),
                \DB::raw('MAX(incoming_operations.operation_date) as latest_return_date'),
                \DB::raw('COALESCE(SUM(incoming_operation_items.quantity_delivered), 0) as total_items_returned'),
                \DB::raw('SUM(incoming_operations.total_amount_aed) as total_value_returned')
            )
            ->groupBy(\DB::raw('COALESCE(incoming_operations.project_id, 0)'))
            ->having('total_returns', '>', 0);

        // Apply project filter if provided
        if ($request->filled('project_id')) {
            $projectReturns->where('incoming_operations.project_id', $request->project_id);
        }

        $projectReturns = $projectReturns->get();

        // Load project details and add status information
        $projects = $projectReturns->map(function ($returnData) {
            if ($returnData->project_id == 0) {
                // Handle unassigned returns (no project)
                $statuses = IncomingOperation::whereNull('project_id')
                    ->where('operation_type', 'site_return')
                    ->select('status', \DB::raw('COUNT(*) as count'))
                    ->groupBy('status')
                    ->pluck('count', 'status')
                    ->toArray();

                return (object) [
                    'project_id' => 0,
                    'project_name' => 'Unassigned Returns',
                    'project_number' => 'N/A',
                    'project_division' => 'N/A',
                    'total_returns' => $returnData->total_returns,
                    'material_requests_count' => $returnData->material_requests_count,
                    'latest_return_date' => $returnData->latest_return_date,
                    'total_items_returned' => $returnData->total_items_returned,
                    'total_value_returned' => $returnData->total_value_returned,
                    'status_breakdown' => $statuses,
                    'primary_status' => $this->getPrimaryProjectStatus($statuses),
                ];
            } else {
                $project = \App\Models\Warehouse\Project::with(['projectDivision'])->find($returnData->project_id);

                if ($project) {
                    // Get status breakdown for this project
                    $statuses = IncomingOperation::where('project_id', $returnData->project_id)
                        ->where('operation_type', 'site_return')
                        ->select('status', \DB::raw('COUNT(*) as count'))
                        ->groupBy('status')
                        ->pluck('count', 'status')
                        ->toArray();

                    return (object) [
                        'project_id' => $project->id,
                        'project_name' => $project->project_name,
                        'project_number' => $project->project_number,
                        'project_division' => $project->projectDivision ? $project->projectDivision->division_name : 'N/A',
                        'total_returns' => $returnData->total_returns,
                        'material_requests_count' => $returnData->material_requests_count,
                        'latest_return_date' => $returnData->latest_return_date,
                        'total_items_returned' => $returnData->total_items_returned,
                        'total_value_returned' => $returnData->total_value_returned,
                        'status_breakdown' => $statuses,
                        'primary_status' => $this->getPrimaryProjectStatus($statuses),
                    ];
                }
            }
            return null;
        })->filter()->sortByDesc(function ($item) {
            return $item->latest_return_date;
        });

        // Get all projects for filter dropdown
        $allProjects = \App\Models\Warehouse\Project::orderBy('project_name')->get();

        // Calculate statistics
        $statistics = [
            'total_projects_with_returns' => $projects->count(),
            'total_returns' => $projects->sum('total_returns'),
            'total_material_requests' => $projects->sum('material_requests_count'),
            'total_items_returned' => $projects->sum('total_items_returned'),
        ];

        return view('warehouse.site-return.index', compact('projects', 'allProjects', 'statistics'));
    }

    /**
     * Get primary status for a project based on status breakdown
     */
    private function getPrimaryProjectStatus($statuses)
    {
        if (empty($statuses)) {
            return 'no_returns';
        }

        // Priority order for statuses
        $statusPriority = [
            'pending_processing' => 4,
            'processing' => 3,
            'processed' => 2,
            'completed' => 1,
            'rejected' => 5
        ];

        $primaryStatus = 'unknown';
        $highestPriority = 999;

        foreach ($statuses as $status => $count) {
            $priority = $statusPriority[$status] ?? 6;
            if ($priority < $highestPriority) {
                $highestPriority = $priority;
                $primaryStatus = $status;
            }
        }

        return $primaryStatus;
    }

    /**
     * Show detailed material returns for a specific project
     */
    public function showProjectReturns($projectId)
    {
        // Get the project
        $project = \App\Models\Warehouse\Project::with(['projectDivision'])
            ->findOrFail($projectId);

        // Get all site returns for this project grouped by material request
        $query = IncomingOperation::with(['items.item', 'receiver', 'materialRequest', 'materialRequest.requester'])
            ->where('operation_type', 'site_return')
            ->where('project_id', $projectId);

        $allReturns = $query->orderBy('operation_date', 'desc')->get();

        // Group by material request ID
        $groupedReturns = $allReturns->groupBy('material_request_id');

        // Get consolidated return data
        $returns = collect();
        foreach ($groupedReturns as $mrId => $mrReturns) {
            $consolidatedReturn = $mrReturns->first();
            $consolidatedReturn->return_count = $mrReturns->count();
            $consolidatedReturn->all_returns = $mrReturns;
            $consolidatedReturn->total_returned_items = $mrReturns->sum(function($return) {
                return $return->items->sum('quantity_delivered');
            });
            $consolidatedReturn->latest_return_date = $mrReturns->max('operation_date');
            $consolidatedReturn->total_amount = $mrReturns->sum('total_amount_aed');

            // Calculate consolidated status
            if ($consolidatedReturn->material_request_id) {
                $materialRequest = OutgoingTransaction::with('items')->find($consolidatedReturn->material_request_id);
                $consolidatedReturn->consolidated_status = $this->calculateMRReturnStatus($materialRequest, $mrReturns);
            } else {
                $consolidatedReturn->consolidated_status = $this->calculateConsolidatedStatus($mrReturns);
            }

            $returns->push($consolidatedReturn);
        }

        // Project statistics
        $statistics = [
            'total_returns' => $allReturns->count(),
            'total_material_requests' => $groupedReturns->count(),
            'total_items_returned' => $allReturns->sum(function($return) {
                return $return->items->sum('quantity_delivered');
            }),
            'total_value_returned' => $allReturns->sum('total_amount_aed'),
            'latest_return_date' => $allReturns->max('operation_date'),
        ];

        return view('warehouse.site-return.project-details', compact('project', 'returns', 'statistics'));
    }

    /**
     * Calculate consolidated status for multiple returns against same material request
     */
    private function calculateConsolidatedStatus($returns)
    {
        $statuses = $returns->pluck('status')->unique();

        if ($statuses->contains('pending_processing')) {
            return 'pending_processing';
        } elseif ($statuses->count() === 1) {
            return $statuses->first();
        } else {
            return 'mixed'; // Multiple different statuses
        }
    }


    /**
     * Show all returns for a specific Material Request
     */
    public function showMaterialRequestReturns($materialRequestId)
    {
        // Get the material request (outgoing transaction)
        $materialRequest = OutgoingTransaction::with(['project', 'division', 'requester', 'items'])
                                             ->findOrFail($materialRequestId);

        // Get all site returns for this material request
        $returns = IncomingOperation::with(['items.item', 'receiver', 'materialRequest'])
                                  ->where('operation_type', 'site_return')
                                  ->where('material_request_id', $materialRequestId)
                                  ->orderBy('operation_date', 'desc')
                                  ->get();

        // Calculate summary statistics
        $totalIssued = $materialRequest->items->sum('quantity_released');
        $totalReturned = $returns->sum(function($return) {
            return $return->items->sum('quantity_delivered');
        });

        $summary = [
            'total_returns' => $returns->count(),
            'total_items_returned' => $totalReturned,
            'pending_quantity' => $totalIssued - $totalReturned,
            'latest_return_date' => $returns->first()?->operation_date,
            'completion_status' => $this->calculateMRReturnStatus($materialRequest, $returns)
        ];

        return view('warehouse.site-return.mr-returns', compact(
            'materialRequest', 'returns', 'summary'
        ));
    }

    /**
     * Calculate return completion status for Material Request
     */
    private function calculateMRReturnStatus($materialRequest, $returns)
    {
        // Handle case where material request doesn't exist
        if (!$materialRequest) {
            return 'unknown';
        }

        $totalIssued = $materialRequest->items->sum('quantity_released');
        $totalReturned = $returns->sum(function($return) {
            return $return->items->sum('quantity_delivered');
        });

        if ($totalReturned == 0) {
            return 'not_returned';
        } elseif ($totalReturned < $totalIssued) {
            return 'partially_returned';
        } elseif ($totalReturned >= $totalIssued) {
            return 'fully_returned';
        }

        return 'unknown';
    }

    /**
     * Show the form for creating a new site return
     */
    public function siteReturnCreate()
    {
        // Get projects for dropdown
        $projects = Project::orderBy('project_name')
                          ->get(['id', 'project_name']);

        // Get users for received by dropdown
        $users = User::orderBy('name')
                    ->get(['id', 'name']);

        // Get items for dropdown
        $items = Item::where('status', 'active')
                    ->orderBy('item_description')
                    ->get(['id', 'item_code', 'item_description', 'unit_of_measure']);

        // Don't load all material requests by default - will be loaded via AJAX based on project selection
        $materialRequests = collect(); // Empty collection

        // Don't load all direct deliveries by default - will be loaded via AJAX based on project selection
        $directDeliveries = collect(); // Empty collection

        return view('warehouse.site-return.create', compact(
            'projects',
            'users',
            'items',
            'materialRequests',
            'directDeliveries'
        ));
    }

    /**
     * Get material request details for site return
     */
    public function getMaterialRequest($id)
    {
        \Log::info('getMaterialRequest called with ID: ' . $id);

        try {
            $materialRequest = OutgoingTransaction::with([
                'project',
                'requester',
                'division',
                'items.item'
            ])->findOrFail($id);

            \Log::info('Material request found:', ['id' => $materialRequest->id, 'status' => $materialRequest->status]);

            // Only allow successful material requests to be returned
            if ($materialRequest->status !== 'successful') {
                return response()->json([
                    'success' => false,
                    'message' => 'Only successfully dispatched material requests can be returned.'
                ]);
            }

            $data = [
                'id' => $materialRequest->id,
                'reference_number' => $materialRequest->material_request_number ?: 'OUT-' . str_pad($materialRequest->id, 6, '0', STR_PAD_LEFT),
                'transaction_date' => $materialRequest->transaction_date->format('M d, Y'),
                'project_id' => $materialRequest->project_id,
                'project_name' => $materialRequest->project ? $materialRequest->project->project_name : 'Unknown Project',
                'division_name' => $materialRequest->division ? $materialRequest->division->division_name : null,
                'requested_by' => $materialRequest->requester ? $materialRequest->requester->name : 'Unknown User',
                'status' => ucfirst($materialRequest->status),
                'items' => $materialRequest->items->map(function($item) {
                    // Calculate total previously returned quantity for this outgoing item
                    $previouslyReturned = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($query) {
                        $query->where('operation_type', 'site_return')
                              ->where('status', 'completed');
                    })
                    ->where('outgoing_item_id', $item->id)
                    ->sum('quantity_delivered'); // quantity_delivered is the returned quantity in site return

                    // Calculate quantity transferred to other projects via Material Transfers
                    $transferredToOtherProjects = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($query) {
                        $query->whereIn('status', ['approved', 'partially_approved']);
                    })
                    ->where('outgoing_item_id', $item->id)
                    ->sum('quantity_approved');

                    $quantityReleased = $item->quantity_released ?? 0;
                    $remainingReturnable = max(0, $quantityReleased - $previouslyReturned - $transferredToOtherProjects);


                    return [
                        'id' => $item->id,
                        'item_id' => $item->item_id,
                        'item_code' => $item->item ? $item->item->item_code : 'N/A',
                        'item_description' => $item->item ? $item->item->item_description : 'Unknown Item',
                        'unit' => $item->item ? $item->item->unit_of_measure : 'N/A',
                        'quantity_requested' => $item->quantity_requested,
                        'quantity_released' => $quantityReleased,
                        'previously_returned' => $previouslyReturned,
                        'transferred_to_projects' => $transferredToOtherProjects,
                        'remaining_returnable' => $remainingReturnable,
                    ];
                })
            ];

            \Log::info('Returning successful response', $data);

            return response()->json([
                'success' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getMaterialRequest', [
                'id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Material request not found or error loading details.'
            ]);
        }
    }

    /**
     * Get direct delivery details for site return
     */
    public function getDirectDelivery($id)
    {
        \Log::info('getDirectDelivery called with ID: ' . $id);

        try {
            $directDelivery = \App\Models\DirectDelivery::with([
                'project',
                'supplier',
                'items.item'
            ])->findOrFail($id);

            \Log::info('Direct delivery found:', ['id' => $directDelivery->id, 'status' => $directDelivery->status]);

            // Allow pending, completed, and delivered direct deliveries to be returned
            if (!in_array($directDelivery->status, ['delivered', 'completed', 'pending'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'This direct delivery status does not allow returns.'
                ]);
            }

            // Check if it can be returned
            if (!$directDelivery->can_be_returned) {
                return response()->json([
                    'success' => false,
                    'message' => 'This direct delivery is not eligible for returns.'
                ]);
            }

            $data = [
                'id' => $directDelivery->id,
                'reference_number' => $directDelivery->delivery_number,
                'delivery_date' => $directDelivery->delivery_date->format('M d, Y'),
                'project_id' => $directDelivery->project_id,
                'project_name' => $directDelivery->project ? $directDelivery->project->project_name : 'Unknown Project',
                'supplier_name' => $directDelivery->supplier ? $directDelivery->supplier->name : 'No Supplier',
                'status' => ucfirst(str_replace('_', ' ', $directDelivery->status)),
                'items' => $directDelivery->items->map(function($item) {
                    // Calculate total previously returned quantity for this direct delivery item
                    $previouslyReturned = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($query) {
                        $query->where('operation_type', 'site_return')
                              ->where('status', 'completed');
                    })
                    ->where('direct_delivery_item_id', $item->id)
                    ->sum('quantity_delivered'); // quantity_delivered is the returned quantity in site return

                    // Calculate quantity transferred to other projects via Material Transfers
                    $transferredToOtherProjects = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($query) {
                        $query->whereIn('status', ['approved', 'partially_approved']);
                    })
                    ->where('direct_delivery_item_id', $item->id)
                    ->sum('quantity_approved');

                    // Use delivered_quantity if available, otherwise use ordered_quantity for pending deliveries
                    $deliveredQuantity = $item->delivered_quantity ?? 0;
                    $baseQuantity = ($deliveredQuantity > 0) ? $deliveredQuantity : $item->ordered_quantity;
                    $remainingReturnable = max(0, $baseQuantity - $previouslyReturned - $transferredToOtherProjects);

                    return [
                        'id' => $item->id,
                        'item_id' => $item->item_id,
                        'item_code' => $item->item ? $item->item->item_code : 'N/A',
                        'item_description' => $item->item ? $item->item->item_description : 'Unknown Item',
                        'unit' => $item->unit_of_measure ?: ($item->item ? $item->item->unit_of_measure : 'N/A'),
                        'quantity_ordered' => $item->ordered_quantity,
                        'quantity_delivered' => $baseQuantity,
                        'previously_returned' => $previouslyReturned,
                        'transferred_to_projects' => $transferredToOtherProjects,
                        'remaining_returnable' => $remainingReturnable,
                        'batch_number' => $item->batch_number,
                        'unit_price' => $item->unit_price,
                    ];
                })
            ];

            \Log::info('Returning successful response', $data);

            return response()->json([
                'success' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getDirectDelivery', [
                'id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Direct delivery not found or error loading details.'
            ]);
        }
    }

    /**
     * Get material requests by project for site return
     */
    public function getMaterialRequestsByProject($projectId)
    {
        try {
            $materialRequests = OutgoingTransaction::with(['project', 'requester'])
                ->where('project_id', $projectId)
                ->where('status', 'successful')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function($request) {
                    return [
                        'id' => $request->id,
                        'text' => ($request->material_request_number ?: 'OUT-' . str_pad($request->id, 6, '0', STR_PAD_LEFT)) .
                               ' - ' . ($request->requester->name ?? 'Unknown') .
                               ' (' . $request->transaction_date->format('M d, Y') . ')',
                        'material_request_number' => $request->material_request_number ?: 'OUT-' . str_pad($request->id, 6, '0', STR_PAD_LEFT),
                        'requester_name' => $request->requester->name ?? 'Unknown',
                        'transaction_date' => $request->transaction_date->format('M d, Y')
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $materialRequests
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getMaterialRequestsByProject', [
                'project_id' => $projectId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error loading material requests for this project.'
            ]);
        }
    }

    /**
     * Get direct deliveries by project for site return dropdown
     */
    public function getDirectDeliveriesByProject($projectId)
    {
        try {
            $directDeliveries = \App\Models\DirectDelivery::with(['project', 'supplier'])
                ->where('project_id', $projectId)
                ->where('status', 'delivered')
                ->where('can_be_returned', true)
                ->orderBy('delivery_date', 'desc')
                ->get()
                ->map(function($delivery) {
                    return [
                        'id' => $delivery->id,
                        'text' => $delivery->delivery_number .
                               ' - ' . $delivery->project->project_name .
                               ($delivery->supplier ? ' (' . $delivery->supplier->name . ')' : '') .
                               ' - ' . $delivery->delivery_date->format('M d, Y'),
                        'delivery_number' => $delivery->delivery_number,
                        'supplier_name' => $delivery->supplier->name ?? 'No Supplier',
                        'delivery_date' => $delivery->delivery_date->format('M d, Y')
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $directDeliveries
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getDirectDeliveriesByProject', [
                'project_id' => $projectId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error loading direct deliveries for this project.'
            ]);
        }
    }

    /**
     * Get combined references (Material Requests + Direct Deliveries) by project for site return
     * Sorted by date in descending order (newest first)
     */
    public function getCombinedReferencesByProject($projectId)
    {
        try {
            \Log::info('getCombinedReferencesByProject called', ['project_id' => $projectId]);

            $combinedReferences = [];

            // Get Material Requests (excluding those with closed returns)
            $materialRequests = OutgoingTransaction::with(['project', 'requester'])
                ->where('project_id', $projectId)
                ->where('status', 'successful')
                ->get(); // Simplified - removed complex whereDoesntHave for now

            \Log::info('Found material requests', ['count' => $materialRequests->count()]);

            $materialRequests = $materialRequests->map(function($request) {
                    // Use the same logic as in other methods to get the reference number
                    $materialRequestNumber = $request->material_request_number ?: 'OUT-' . str_pad($request->id, 6, '0', STR_PAD_LEFT);

                    return [
                        'id' => 'MR_' . $request->id,
                        'reference_id' => $request->id,
                        'reference_type' => 'material_request',
                        'display_text' => 'MR: ' . $materialRequestNumber . ' (' . $request->transaction_date->format('M d, Y') . ')',
                        'sort_date' => $request->transaction_date,
                        'details' => [
                            'number' => $materialRequestNumber,
                            'date' => $request->transaction_date->format('M d, Y'),
                            'requester' => $request->requester ? $request->requester->name : 'Unknown',
                            'project' => $request->project ? $request->project->project_name : 'Unknown'
                        ]
                    ];
                });

            // Get Direct Deliveries (excluding those with closed returns)
            $directDeliveries = \App\Models\DirectDelivery::with(['project', 'supplier'])
                ->where('project_id', $projectId)
                ->whereIn('status', ['delivered', 'completed', 'pending'])
                ->get(); // Simplified - removed complex conditions for now

            \Log::info('Found direct deliveries', ['count' => $directDeliveries->count()]);

            $directDeliveries = $directDeliveries->map(function($delivery) {
                    return [
                        'id' => 'DD_' . $delivery->id,
                        'reference_id' => $delivery->id,
                        'reference_type' => 'direct_delivery',
                        'display_text' => 'DD: ' . $delivery->delivery_number . ' (' . $delivery->delivery_date->format('M d, Y') . ')',
                        'sort_date' => $delivery->delivery_date,
                        'details' => [
                            'number' => $delivery->delivery_number,
                            'date' => $delivery->delivery_date->format('M d, Y'),
                            'supplier' => $delivery->supplier ? $delivery->supplier->name : 'No Supplier',
                            'project' => $delivery->project ? $delivery->project->project_name : 'Unknown'
                        ]
                    ];
                });

            // Get Material Transfer Requests (where this project is the receiver, excluding those with closed returns)
            $materialTransfers = \App\Models\Warehouse\MaterialTransferRequest::with(['transferrerProject', 'receiverProject'])
                ->where('receiver_project_id', $projectId)
                ->whereIn('status', ['approved', 'partially_approved'])
                ->get(); // Simplified - removed complex conditions for now

            \Log::info('Found material transfers', ['count' => $materialTransfers->count()]);

            $materialTransfers = $materialTransfers->map(function($transfer) {
                    return [
                        'id' => 'MT_' . $transfer->id,
                        'reference_id' => $transfer->id,
                        'reference_type' => 'material_transfer',
                        'display_text' => 'MT: ' . $transfer->transfer_request_number . ' (' . $transfer->request_date->format('M d, Y') . ') from ' . ($transfer->transferrerProject->project_name ?? 'Unknown'),
                        'sort_date' => $transfer->request_date,
                        'details' => [
                            'number' => $transfer->transfer_request_number,
                            'date' => $transfer->request_date->format('M d, Y'),
                            'from_project' => $transfer->transferrerProject ? $transfer->transferrerProject->project_name : 'Unknown',
                            'to_project' => $transfer->receiverProject ? $transfer->receiverProject->project_name : 'Unknown'
                        ]
                    ];
                });

            // Combine all collections - ensure they are collections
            $combinedReferences = collect($materialRequests)->merge(collect($directDeliveries))->merge(collect($materialTransfers));

            // Sort by date in descending order (newest first)
            $combinedReferences = $combinedReferences->sortByDesc(function ($item) {
                return $item['sort_date'];
            })->values();

            \Log::info('Combined references result', ['total_count' => $combinedReferences->count()]);

            return response()->json([
                'success' => true,
                'data' => $combinedReferences
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in getCombinedReferencesByProject', [
                'project_id' => $projectId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error loading references for this project: ' . $e->getMessage(),
                'data' => []
            ]);
        }
    }

    /**
     * Get material transfer details for site return
     */
    public function getMaterialTransfer($transferId)
    {
        try {
            $materialTransfer = \App\Models\Warehouse\MaterialTransferRequest::with([
                'transferrerProject',
                'receiverProject',
                'items.item'
            ])->findOrFail($transferId);

            // Only allow returns for approved/partially approved transfers
            if (!in_array($materialTransfer->status, ['approved', 'partially_approved'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Material transfer must be approved to be available for return.'
                ]);
            }

            $data = [
                'id' => $materialTransfer->id,
                'transfer_request_number' => $materialTransfer->transfer_request_number,
                'request_date' => $materialTransfer->request_date->format('M d, Y'),
                'transferrer_project_id' => $materialTransfer->transferrer_project_id,
                'transferrer_project_name' => $materialTransfer->transferrerProject ? $materialTransfer->transferrerProject->project_name : 'Unknown Project',
                'receiver_project_id' => $materialTransfer->receiver_project_id,
                'receiver_project_name' => $materialTransfer->receiverProject ? $materialTransfer->receiverProject->project_name : 'Unknown Project',
                'status' => ucwords(str_replace('_', ' ', $materialTransfer->status)),
                'priority' => ucfirst($materialTransfer->priority),
                'items' => $materialTransfer->items->map(function($item) use ($materialTransfer) {
                    // For material transfers, we need to check what was actually approved/transferred
                    $transferredQuantity = $item->quantity_approved ?? 0;

                    // Calculate already returned quantity for this transfer item
                    $returnedQuantity = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($query) {
                        $query->where('operation_type', 'site_return')
                              ->where('status', 'completed');
                    })
                    ->where('material_transfer_item_id', $item->id)
                    ->sum('quantity_delivered');

                    // Calculate quantity transferred to other projects
                    // Check for subsequent transfers from this Material Transfer item using the new tracking field
                    $transferredToOtherProjects = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($query) {
                        $query->whereIn('status', ['approved', 'partially_approved']);
                    })
                    ->where('material_transfer_item_id', $item->id)
                    ->sum('quantity_approved');

                    // Enhanced calculation: if no specific tracking found, try to calculate based on recent transfer requests
                    if ($transferredToOtherProjects == 0) {
                        // Look for Material Transfer Requests where items came from this receiver project
                        // and check if they reference items that might have originated from this transfer
                        $subsequentTransfers = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($query) use ($materialTransfer) {
                            $query->where('transferrer_project_id', $materialTransfer->receiver_project_id)
                                  ->whereIn('status', ['approved', 'partially_approved']);
                        })
                        ->where('item_id', $item->item_id)
                        ->whereDate('created_at', '>=', $materialTransfer->approved_at ?? $materialTransfer->created_at)
                        ->sum('quantity_approved');

                        $transferredToOtherProjects = $subsequentTransfers;
                    }

                    \Log::info('Material Transfer calculation', [
                        'transfer_id' => $materialTransfer->id,
                        'item_id' => $item->item_id,
                        'receiver_project_id' => $materialTransfer->receiver_project_id,
                        'transferred_quantity' => $transferredQuantity,
                        'returned_quantity' => $returnedQuantity,
                        'transferred_to_projects' => $transferredToOtherProjects,
                        'calculation_method' => $transferredToOtherProjects > 0 ? 'direct_tracking' : 'subsequent_transfers'
                    ]);

                    // Calculate remaining returnable
                    $remainingReturnable = max(0, $transferredQuantity - $returnedQuantity - $transferredToOtherProjects);

                    return [
                        'id' => $item->id,
                        'item_id' => $item->item_id,
                        'item_code' => $item->item ? $item->item->item_code : 'N/A',
                        'item_description' => $item->item ? $item->item->item_description : 'Unknown Item',
                        'unit' => $item->item ? $item->item->unit_of_measure : 'PCS',
                        'quantity_requested' => $item->quantity_requested,
                        'quantity_approved' => $transferredQuantity,
                        'previously_returned' => $returnedQuantity,
                        'transferred_to_projects' => $transferredToOtherProjects,
                        'remaining_returnable' => $remainingReturnable,
                        'estimated_unit_price' => $item->estimated_unit_price ?? 0,
                        'approval_status' => $item->approval_status,
                    ];
                })->filter(function($item) {
                    // Include items that have been transferred (have approved quantity > 0)
                    // For partially transferred materials, we need any item that was actually transferred
                    return $item['quantity_approved'] > 0;
                })->values()
            ];

            return response()->json([
                'success' => true,
                'data' => $data
            ]);

        } catch (\Exception $e) {
            \Log::error('Error getting material transfer for site return: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading material transfer details: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get site return data for AJAX
     */
    public function siteReturnData(Request $request)
    {
        $query = IncomingOperation::with(['project', 'items', 'receiver'])
                                 ->where('operation_type', 'site_return');

        // Apply filters if provided
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Additional filters can be added here

        $returns = $query->latest('operation_date')->get();

        return response()->json([
            'data' => $returns,
            'recordsTotal' => $returns->count(),
            'recordsFiltered' => $returns->count()
        ]);
    }

    /**
     * Export site returns
     */
    public function exportSiteReturns(Request $request)
    {
        // This is a placeholder for export functionality
        // You would implement actual export logic here (Excel, CSV, etc.)

        return response()->json([
            'success' => false,
            'message' => 'Export functionality will be implemented soon'
        ]);
    }

    /**
     * Calculate site return status based on return quantities vs issued quantities
     */
    private function calculateSiteReturnStatus($operation, $request)
    {
        // If force close is enabled, return force_closed status
        if ($operation->shouldForceClose()) {
            return 'force_closed';
        }

        $totalIssued = 0;
        $totalReturned = 0;

        // Get the material request to compare against issued quantities
        $materialRequestId = $request->material_request_id;
        $materialRequest = \App\Models\Warehouse\OutgoingTransaction::with('items')->find($materialRequestId);

        if (!$materialRequest) {
            return 'not_returned';
        }

        // Calculate totals
        foreach ($materialRequest->items as $issuedItem) {
            $totalIssued += (float)$issuedItem->quantity_released;

            // Find the corresponding returned item
            foreach ($request->items as $returnItem) {
                if ($returnItem['outgoing_item_id'] == $issuedItem->id) {
                    $totalReturned += (float)($returnItem['quantity_returned'] ?? 0);
                    break;
                }
            }
        }

        // Determine status based on return ratio
        if ($totalReturned == 0) {
            return 'not_returned';
        } elseif ($totalReturned >= $totalIssued) {
            return 'closed_return';  // All or more returned
        } else {
            return 'partially_returned';  // Some returned
        }
    }

    /**
     * Print material request returns
     */
    public function printMaterialRequestReturns($materialRequestId)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the material request (outgoing transaction)
        $materialRequest = OutgoingTransaction::with(['project', 'requester', 'items.item'])
                                             ->findOrFail($materialRequestId);

        // Get all site returns for this material request
        $returns = IncomingOperation::with(['items.item', 'receiver'])
                                  ->where('operation_type', 'site_return')
                                  ->where('material_request_id', $materialRequestId)
                                  ->orderBy('operation_date', 'desc')
                                  ->get();

        // Prepare data for print template
        $siteReturn = [
            'id' => $materialRequest->id,
            'return_number' => 'SR-' . $materialRequest->material_request_number,
            'return_date' => $returns->isNotEmpty() ? $returns->first()->operation_date->format('d.m.Y') : now()->format('d.m.Y'),
            'project' => $materialRequest->project ? $materialRequest->project->project_name : 'N/A',
            'returned_to' => 'Warehouse',
            'returned_by' => $returns->isNotEmpty() ? ($returns->first()->receiver->name ?? 'N/A') : 'N/A',
            'items' => $returns->flatMap(function($return) {
                return $return->items->map(function($item) {
                    return [
                        'description' => $item->item->item_description,
                        'unit' => $item->item->unit_of_measure ?: 'PCS',
                        'quantity_returned' => $item->quantity_delivered
                    ];
                });
            })->toArray()
        ];

        return view('warehouse.site-return.print', compact('siteReturn'));
    }

    /**
     * Print individual site return
     */
    public function printSiteReturn($returnId)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the site return operation
        $return = IncomingOperation::with(['project', 'receiver', 'items.item', 'materialRequest'])
                                  ->where('operation_type', 'site_return')
                                  ->findOrFail($returnId);

        // Prepare data for print template
        $siteReturn = [
            'id' => $return->id,
            'return_number' => $return->operation_number ?: 'SR-' . $return->id,
            'return_date' => $return->operation_date->format('d.m.Y'),
            'project' => $return->project ? $return->project->project_name : 'N/A',
            'returned_to' => 'Warehouse',
            'returned_by' => $return->receiver ? $return->receiver->name : 'N/A',
            'items' => $return->items->map(function($item) {
                return [
                    'description' => $item->item->item_description,
                    'unit' => $item->item->unit_of_measure ?: 'PCS',
                    'quantity_returned' => $item->quantity_delivered
                ];
            })->toArray()
        ];

        return view('warehouse.site-return.print', compact('siteReturn'));
    }

    /**
     * Print consolidated returns for a project.
     */
    public function printProjectConsolidatedReturns($projectId)
    {
        // Handle unassigned returns (projectId = 0)
        if ($projectId == 0) {
            $project = (object) [
                'id' => 0,
                'project_name' => 'Unassigned Returns',
                'project_number' => 'N/A',
                'projectDivision' => null
            ];
        } else {
            // Get the project
            $project = Project::findOrFail($projectId);
        }

        // Get all site returns for this project with related data
        $query = IncomingOperation::with([
            'items.item',
            'materialRequest',
            'directDelivery',
            'receiver'  // Changed from receivedByUser
        ])
        ->where('operation_type', 'site_return');

        // Handle unassigned returns
        if ($projectId == 0) {
            $query->whereNull('project_id');
        } else {
            $query->where('project_id', $projectId);
        }

        $returns = $query->orderBy('operation_date', 'desc')->get();

        // Group returns by material request for better organization
        $groupedReturns = $returns->groupBy('material_request_id');

        // Calculate totals
        $totalItems = 0;
        $totalValue = 0;
        $itemsSummary = [];

        foreach ($returns as $return) {
            foreach ($return->items as $item) {
                $totalItems += $item->quantity_delivered;
                $totalValue += ($item->quantity_delivered * $item->unit_price);

                // Build items summary
                $itemCode = $item->item->item_code ?? 'N/A';
                if (!isset($itemsSummary[$itemCode])) {
                    $itemsSummary[$itemCode] = [
                        'code' => $itemCode,
                        'description' => $item->item->item_description ?? 'N/A',
                        'unit' => $item->item->unit_of_measure ?? 'pcs',
                        'total_quantity' => 0,
                        'returns' => []
                    ];
                }
                $itemsSummary[$itemCode]['total_quantity'] += $item->quantity_delivered;
                $itemsSummary[$itemCode]['returns'][] = [
                    'date' => $return->operation_date->format('d/m/Y'),
                    'mr_number' => $return->materialRequest->mr_number ?? 'Direct',
                    'quantity' => $item->quantity_delivered,
                    'condition' => $item->return_condition ?? 'Good'
                ];
            }
        }

        // Prepare consolidated data
        $consolidatedData = [
            'project' => $project,
            'returns' => $returns,
            'groupedReturns' => $groupedReturns,
            'itemsSummary' => $itemsSummary,
            'totalReturns' => $returns->count(),
            'totalItems' => $totalItems,
            'totalValue' => $totalValue,
            'printDate' => now()->format('d/m/Y H:i'),
            'printedBy' => auth()->user()->name ?? 'System'
        ];

        return view('warehouse.site-return.print-consolidated', compact('consolidatedData'));
    }
}