<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Services\Warehouse\InventoryService;
use App\Services\Auth\RoleBasedAccessService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class InventoryController extends Controller
{
    protected $inventoryService;
    protected $accessService;

    public function __construct(InventoryService $inventoryService, RoleBasedAccessService $accessService)
    {
        $this->inventoryService = $inventoryService;
        $this->accessService = $accessService;

        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.inventory.view')->only(['index', 'show', 'getAdjustmentDetails']);
        $this->middleware('warehouse.permission:warehouse.inventory.adjust')->only(['adjust', 'performAdjustment', 'storeAdjustment', 'clearAdjustments']);
        $this->middleware('warehouse.permission:warehouse.inventory.transfer')->only(['transfer', 'performTransfer']);
        $this->middleware('warehouse.permission:warehouse.inventory.count')->only(['physicalCount', 'performPhysicalCount']);
    }

    /**
     * Display a listing of the inventory.
     */
    public function index(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Handle dashboard chart data requests
        if ($request->has('chart_data')) {
            return $this->getChartData($request);
        }

        // Handle dashboard metrics refresh
        if ($request->has('refresh') && $request->refresh === 'metrics') {
            return $this->getDashboardMetrics();
        }

        // Handle DataTable requests
        if (($request->expectsJson() && $request->has('draw')) || $request->has('ajax_request')) {
            try {
                return $this->getDataForTable($request);
            } catch (\Exception $e) {
                \Log::error('DataTable Error in index method', [
                    'error' => $e->getMessage(),
                    'request_data' => $request->all()
                ]);

                return response()->json([
                    'error' => 'DataTable Error: ' . $e->getMessage(),
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ], 500);
            }
        }

        // Handle regular JSON requests
        if ($request->expectsJson()) {
            $filters = $request->only([
                'location', 'item_id', 'low_stock', 'zero_stock', 'has_reserved',
                'updated_from', 'updated_to', 'search'
            ]);

            $inventory = $this->inventoryService->getAll($filters, ['item', 'item.category']);

            return response()->json([
                'success' => true,
                'data' => $inventory,
                'message' => 'Inventory retrieved successfully'
            ]);
        }

        // Return dashboard view
        return view('warehouse.inventory.index');
    }

    /**
     * Get data for DataTable.
     */
    private function getDataForTable(Request $request)
    {
        try {
            $start = $request->get('start', 0);
            $length = $request->get('length', 25);
            $search = $request->get('search.value');

            // Debug log
            \Log::info('DataTable Request', [
                'start' => $start,
                'length' => $length,
                'search' => $search,
                'draw' => $request->get('draw')
            ]);

            // Query inventory with items and categories
            $query = \App\Models\Warehouse\Inventory::with(['item.category']);

            // Apply search
            if (!empty($search)) {
                $query->whereHas('item', function ($q) use ($search) {
                    $q->where('item_description', 'LIKE', "%{$search}%")
                      ->orWhere('item_code', 'LIKE', "%{$search}%")
                      ->orWhere('notes', 'LIKE', "%{$search}%");
                });
            }

            // Get total count
            $totalRecords = \App\Models\Warehouse\Inventory::count();
            $filteredRecords = $query->count();

            // Apply pagination
            $inventoryRecords = $query->skip($start)
                                    ->take($length)
                                    ->orderBy('updated_at', 'desc')
                                    ->get();

            // Format data for DataTable
            $data = $inventoryRecords->map(function ($inventory) {
                $item = $inventory->item;

                // Calculate actual values from inventory
                $currentStock = $inventory->quantity_available ?: 0;
                $reservedStock = 0; // Default - could be calculated from outgoing transactions
                $minLevel = 10; // Default - this could be added to items table later
                $stockValue = $currentStock * ($inventory->unit_price ?: 0);

                // Get last stock movement date
                $lastMovement = \App\Models\Warehouse\StockMovement::where('item_id', $inventory->item_id)
                                                                  ->latest('created_at')
                                                                  ->first();

                return [
                    'id' => $inventory->id,
                    'item_name' => $item ? $item->item_description : 'Unknown Item',
                    'item_code' => $item ? $item->item_code : 'NO-CODE',
                    'category' => $item && $item->category ? $item->category->category_name : 'Uncategorized',
                    'current_stock' => number_format($currentStock, 2),
                    'min_level' => $minLevel,
                    'reserved_stock' => $reservedStock,
                    'stock_value' => number_format($stockValue, 2),
                    'unit' => $item ? ($item->unit_of_measure ?: 'units') : 'units',
                    'last_movement' => $lastMovement ? $lastMovement->created_at->format('M d, Y') : 'Never',
                    'batch_number' => $inventory->batch_number ?: '-',
                    'location' => $inventory->location ?: 'Main Warehouse',
                    'expiry_date' => $inventory->expiry_date ? $inventory->expiry_date->format('M d, Y') : '-'
                ];
            });

            $response = [
                'draw' => intval($request->get('draw')),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data->toArray()
            ];

            \Log::info('DataTable Response', ['response_count' => count($data)]);

            return response()->json($response);

        } catch (\Exception $e) {
            \Log::error('DataTable Error', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            throw $e;
        }
    }

    /**
     * Get dashboard metrics.
     */
    private function getDashboardMetrics()
    {
        // Calculate real metrics from inventory
        $totalItems = \App\Models\Warehouse\Inventory::count();
        $totalValue = \App\Models\Warehouse\Inventory::sum('total_value') ?: 0;
        $lowStockCount = \App\Models\Warehouse\Inventory::where('quantity_available', '<=', 10)
                                                       ->where('quantity_available', '>', 0)
                                                       ->count();
        $recentMovements = \App\Models\Warehouse\StockMovement::where('created_at', '>=', now()->subDays(7))
                                                              ->count();

        return response()->json([
            'metrics' => [
                'total_items' => $totalItems,
                'total_value' => number_format($totalValue, 2),
                'low_stock_count' => $lowStockCount,
                'recent_movements' => $recentMovements
            ]
        ]);
    }

    /**
     * Get chart data for dashboard.
     */
    private function getChartData(Request $request)
    {
        $chartType = $request->get('chart_data');
        $period = $request->get('period', '7d');

        if ($chartType === 'movements') {
            return $this->getMovementChartData($period);
        }

        if ($chartType === 'categories') {
            return $this->getCategoryChartData();
        }

        return response()->json(['error' => 'Invalid chart type'], 400);
    }

    /**
     * Get stock movement chart data.
     */
    private function getMovementChartData($period)
    {
        $days = match($period) {
            '7d' => 7,
            '30d' => 30,
            '90d' => 90,
            default => 7
        };

        $labels = [];
        $stockInData = [];
        $stockOutData = [];

        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $labels[] = $date->format('M j');

            // Generate sample data for demonstration
            $stockInData[] = rand(0, 50);
            $stockOutData[] = rand(0, 30);
        }

        return response()->json([
            'movements' => [
                'labels' => $labels,
                'stock_in' => $stockInData,
                'stock_out' => $stockOutData
            ]
        ]);
    }

    /**
     * Get category distribution chart data.
     */
    private function getCategoryChartData()
    {
        $categories = \App\Models\Warehouse\ItemCategory::with('items')->get();
        $labels = [];
        $values = [];

        foreach ($categories as $category) {
            $labels[] = $category->category_name;
            $values[] = $category->items->count();
        }

        // Add uncategorized items
        $uncategorizedCount = \App\Models\Warehouse\Item::whereNull('category_id')->count();
        if ($uncategorizedCount > 0) {
            $labels[] = 'Uncategorized';
            $values[] = $uncategorizedCount;
        }

        return response()->json([
            'categories' => [
                'labels' => $labels,
                'values' => $values
            ]
        ]);
    }

    /**
     * Display inventory by location.
     */
    public function byLocation($location)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        $inventory = $this->inventoryService->getInventoryByLocation($location);

        return response()->json([
            'success' => true,
            'data' => $inventory,
            'message' => 'Inventory for location retrieved successfully'
        ]);
    }

    /**
     * Show the form for adjusting inventory.
     */
    public function adjust($itemId)
    {
        Gate::authorize('adjust', \App\Models\Warehouse\Inventory::class);

        $item = \App\Models\Warehouse\Item::findOrFail($itemId);
        $inventory = $this->inventoryService->findByItemId($itemId);

        return view('warehouse.inventory.adjust', compact('item', 'inventory'));
    }

    /**
     * Perform inventory adjustment.
     */
    public function performAdjustment(Request $request, $itemId)
    {
        Gate::authorize('adjust', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'quantity' => 'required|integer|min:1',
            'type' => 'required|in:increase,decrease,set',
            'reason' => 'required|string|max:255',
            'metadata' => 'nullable|array'
        ]);

        try {
            $inventory = $this->inventoryService->adjustInventory(
                $itemId,
                $request->quantity,
                $request->type,
                $request->reason,
                $request->metadata ?? []
            );

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $inventory,
                    'message' => 'Inventory adjusted successfully'
                ]);
            }

            return redirect()->route('warehouse.inventory.index')
                           ->with('success', 'Inventory adjusted successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Show the form for transferring inventory.
     */
    public function transfer($itemId)
    {
        Gate::authorize('transfer', \App\Models\Warehouse\Inventory::class);

        $item = \App\Models\Warehouse\Item::findOrFail($itemId);
        $locations = \App\Models\Warehouse\Inventory::distinct('location')->pluck('location');

        return view('warehouse.inventory.transfer', compact('item', 'locations'));
    }

    /**
     * Perform inventory transfer.
     */
    public function performTransfer(Request $request, $itemId)
    {
        Gate::authorize('transfer', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'from_location' => 'required|string|max:255',
            'to_location' => 'required|string|max:255|different:from_location',
            'quantity' => 'required|integer|min:1',
            'reason' => 'nullable|string|max:255'
        ]);

        try {
            $result = $this->inventoryService->transferInventory(
                $itemId,
                $request->from_location,
                $request->to_location,
                $request->quantity,
                $request->reason ?? 'Location transfer'
            );

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $result,
                    'message' => 'Inventory transferred successfully'
                ]);
            }

            return redirect()->route('warehouse.inventory.index')
                           ->with('success', 'Inventory transferred successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Reserve inventory.
     */
    public function reserve(Request $request, $itemId)
    {
        Gate::authorize('reserve', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'metadata' => 'nullable|array'
        ]);

        try {
            $inventory = $this->inventoryService->reserveInventory(
                $itemId,
                $request->quantity,
                $request->reason,
                $request->metadata ?? []
            );

            return response()->json([
                'success' => true,
                'data' => $inventory,
                'message' => 'Inventory reserved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Release reserved inventory.
     */
    public function releaseReserved(Request $request, $itemId)
    {
        Gate::authorize('release', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'quantity' => 'required|integer|min:1',
            'reason' => 'required|string|max:255',
            'metadata' => 'nullable|array'
        ]);

        try {
            $inventory = $this->inventoryService->releaseReservedInventory(
                $itemId,
                $request->quantity,
                $request->reason,
                $request->metadata ?? []
            );

            return response()->json([
                'success' => true,
                'data' => $inventory,
                'message' => 'Reserved inventory released successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Show the form for physical count.
     */
    public function physicalCount()
    {
        Gate::authorize('physicalCount', \App\Models\Warehouse\Inventory::class);

        $items = \App\Models\Warehouse\Item::with('inventories')->where('status', 'active')->get();

        return view('warehouse.inventory.physical-count', compact('items'));
    }

    /**
     * Perform physical count.
     */
    public function performPhysicalCount(Request $request)
    {
        Gate::authorize('physicalCount', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'count_data' => 'required|array|min:1',
            'count_data.*.item_id' => 'required|exists:items,id',
            'count_data.*.counted_quantity' => 'required|integer|min:0',
            'count_data.*.location' => 'nullable|string|max:255'
        ]);

        try {
            $results = $this->inventoryService->performPhysicalCount($request->count_data);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $results,
                    'message' => 'Physical count completed successfully'
                ]);
            }

            $message = "Physical count completed. {$results['total_items']} items counted, {$results['adjustments']} adjustments made.";

            return redirect()->route('warehouse.inventory.index')
                           ->with('success', $message);

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Get low stock items.
     */
    public function lowStock(Request $request)
    {
        // Route already has middleware protection, skip additional auth checks for now

        // Handle DataTable requests
        if ($request->expectsJson() && $request->has('draw')) {
            \Log::info('Processing DataTable request for low stock', [
                'url' => $request->url(),
                'method' => $request->method(),
                'headers' => $request->headers->all(),
                'parameters' => $request->all()
            ]);

            return $this->getLowStockDataForTable($request);
        }

        // Handle JSON requests (non-DataTable)
        if ($request->expectsJson() && !$request->has('draw')) {
            // Generate sample low stock items for demo
            $items = \App\Models\Warehouse\Item::with(['category'])->take(5)->get();

            $lowStockItems = $items->map(function ($item) {
                $minLevel = rand(20, 50);
                $currentStock = rand(0, $minLevel); // Ensure it's low stock

                return [
                    'id' => $item->id,
                    'item_name' => $item->item_description,
                    'item_code' => $item->item_code,
                    'category' => $item->category ? $item->category->category_name : null,
                    'current_stock' => $currentStock,
                    'min_level' => $minLevel,
                    'shortage' => max(0, $minLevel - $currentStock),
                    'stock_value' => $currentStock * rand(10, 50), // Demo pricing since no price field exists
                    'days_left' => $currentStock > 0 ? rand(1, 7) : 0,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $lowStockItems,
                'message' => 'Low stock items retrieved successfully'
            ]);
        }

        // Return low stock alerts view
        return view('warehouse.inventory.low-stock');
    }

    /**
     * Get low stock data for DataTable.
     */
    private function getLowStockDataForTable(Request $request)
    {
        try {
            $start = $request->get('start', 0);
            $length = $request->get('length', 25);
            $search = $request->get('search.value');

            // Query inventory records that are low stock
            $lowStockThreshold = 10; // Define low stock threshold
            $query = \App\Models\Warehouse\Inventory::with(['item.category'])
                                                    ->where('quantity_available', '<=', $lowStockThreshold)
                                                    ->where('quantity_available', '>', 0); // Exclude out of stock

            // Apply search
            if (!empty($search)) {
                $query->whereHas('item', function ($q) use ($search) {
                    $q->where('item_description', 'LIKE', "%{$search}%")
                      ->orWhere('item_code', 'LIKE', "%{$search}%");
                });
            }

            // Get total count
            $totalRecords = \App\Models\Warehouse\Inventory::where('quantity_available', '<=', $lowStockThreshold)
                                                          ->where('quantity_available', '>', 0)
                                                          ->count();

            if ($totalRecords == 0) {
                return response()->json([
                    'draw' => intval($request->get('draw')),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ]);
            }

            $filteredRecords = $query->count();

            // Apply pagination
            $inventoryRecords = $query->skip($start)
                                    ->take($length)
                                    ->orderBy('quantity_available', 'asc') // Show lowest stock first
                                    ->get();

            // Format data for DataTable
            $data = $inventoryRecords->map(function ($inventory) {
                $item = $inventory->item;
                $minLevel = 10; // Default minimum level
                $currentStock = $inventory->quantity_available;
                $unitPrice = $inventory->unit_price ?: 0;
                $stockValue = $currentStock * $unitPrice;

                // Calculate estimated days left (simple calculation)
                $daysLeft = $currentStock > 0 ? ceil($currentStock / 2) : 0; // Assuming 2 units consumed per day

                return [
                    'id' => $inventory->id,
                    'item_name' => $item ? $item->item_description : 'Unknown Item',
                    'item_code' => $item ? $item->item_code : 'NO-CODE',
                    'category' => $item && $item->category ? $item->category->category_name : 'Uncategorized',
                    'current_stock' => number_format($currentStock, 2),
                    'min_level' => $minLevel,
                    'shortage' => max(0, $minLevel - $currentStock),
                    'stock_value' => number_format($stockValue, 2),
                    'days_left' => $daysLeft,
                    'unit' => $item ? ($item->unit_of_measure ?: 'units') : 'units',
                    'last_updated' => $inventory->updated_at ? $inventory->updated_at->format('M d, Y H:i') : 'Never',
                    'batch_number' => $inventory->batch_number ?: '-',
                    'location' => $inventory->location ?: 'Main Warehouse'
                ];
            });

            return response()->json([
                'draw' => intval($request->get('draw')),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data->toArray()
            ]);

        } catch (\Exception $e) {
            \Log::error('Low Stock DataTable Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'draw' => intval($request->get('draw', 1)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading low stock data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get inventory aging report.
     */
    public function agingReport(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        $days = $request->input('days', 90);

        $agingInventory = $this->inventoryService->getInventoryAging($days);

        // If it's an AJAX request, return JSON
        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $agingInventory,
                'message' => 'Inventory aging report generated successfully'
            ]);
        }

        // Otherwise return the view
        return view('warehouse.inventory.aging-report', compact('agingInventory', 'days'));
    }

    /**
     * Display inventory valuation report.
     */
    public function valuation(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Handle DataTable AJAX requests
        if ($request->expectsJson() && $request->has('draw')) {
            return $this->getValuationDataForTable($request);
        }

        // Get real inventory data with proper calculations
        $inventoryQuery = \App\Models\Warehouse\Inventory::with(['item.category'])
                          ->whereHas('item', function($q) {
                              $q->where('status', 'active');
                          });

        $inventories = $inventoryQuery->get();

        // Calculate real valuation data
        $valuationData = $inventories->map(function ($inventory) {
            $item = $inventory->item;
            $currentStock = $inventory->quantity_available ?? 0;

            // Use actual prices from inventory, with fallback to reasonable defaults
            $unitPrice = $inventory->unit_price ?? $inventory->purchase_price ?? 0;

            $totalValue = $currentStock * $unitPrice;

            return [
                'id' => $item->id,
                'item_name' => $item->item_description ?? $item->item_code ?? 'Unknown Item',
                'item_code' => $item->item_code ?? 'N/A',
                'category' => $item->category ? $item->category->category_name : 'Uncategorized',
                'current_stock' => $currentStock,
                'unit_price' => $unitPrice,
                'total_value' => $totalValue,
                'unit' => $item->unit_of_measure ?? 'pcs',
                'last_updated' => $inventory->last_updated ?? $inventory->updated_at,
                'location' => $inventory->location ?? 'Main Warehouse',
                'cost_method' => 'FIFO', // Default costing method
            ];
        })->filter(function($item) {
            // Only include items with actual stock or value
            return $item['current_stock'] > 0 || $item['total_value'] > 0;
        })->sortByDesc('total_value');

        // Calculate accurate summary statistics
        $statistics = [
            'total_items' => $valuationData->count(),
            'total_stock_value' => $valuationData->sum('total_value'),
            'total_units' => $valuationData->sum('current_stock'),
            'average_unit_value' => $valuationData->count() > 0 ? $valuationData->avg('unit_price') : 0,
            'highest_value_item' => $valuationData->first()['total_value'] ?? 0,
            'categories_count' => $valuationData->pluck('category')->unique()->count(),
        ];

        // Category breakdown for charts
        $categoryBreakdown = $valuationData->groupBy('category')->map(function ($items, $category) {
            return [
                'category' => $category,
                'total_value' => $items->sum('total_value'),
                'item_count' => $items->count(),
                'average_value' => $items->avg('total_value'),
            ];
        })->sortByDesc('total_value');

        return view('warehouse.inventory.valuation', compact('valuationData', 'statistics', 'categoryBreakdown'));
    }

    /**
     * Get valuation data for DataTable.
     */
    private function getValuationDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        // Get real inventory data
        $query = \App\Models\Warehouse\Inventory::with(['item.category'])
                ->whereHas('item', function($q) {
                    $q->where('status', 'active');
                });

        // Apply search filter
        if (!empty($search)) {
            $query->whereHas('item', function ($q) use ($search) {
                $q->where('item_description', 'LIKE', "%{$search}%")
                  ->orWhere('item_code', 'LIKE', "%{$search}%");
            });
        }

        // Apply category filter
        if ($request->filled('category')) {
            $query->whereHas('item.category', function ($q) use ($request) {
                $q->where('category_name', $request->get('category'));
            });
        }

        // Get total counts
        $totalRecords = \App\Models\Warehouse\Inventory::whereHas('item', function($q) {
            $q->where('status', 'active');
        })->count();
        $filteredRecords = $query->count();

        // Apply pagination
        $inventories = $query->skip($start)->take($length)->get();

        // Format data for DataTable
        $data = $inventories->map(function ($inventory) {
            $item = $inventory->item;
            $currentStock = $inventory->quantity_available ?? 0;
            $unitPrice = $inventory->unit_price ?? $inventory->purchase_price ?? 0;
            $totalValue = $currentStock * $unitPrice;

            return [
                'id' => $item->id,
                'item_name' => $item->item_description ?? 'Unknown Item',
                'item_code' => $item->item_code ?? 'N/A',
                'category' => $item->category ? $item->category->category_name : 'Uncategorized',
                'current_stock' => number_format($currentStock, 2),
                'unit_price' => number_format($unitPrice, 2),
                'total_value' => number_format($totalValue, 2),
                'unit' => $item->unit_of_measure ?? 'pcs',
                'last_updated' => ($inventory->last_updated ?? $inventory->updated_at)->format('Y-m-d H:i:s'),
                'location' => $inventory->location ?? 'Main Warehouse',
                'cost_method' => 'FIFO'
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data->toArray()
        ]);
    }

    /**
     * Check bulk availability.
     */
    public function checkBulkAvailability(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.quantity' => 'required|integer|min:1'
        ]);

        try {
            $results = $this->inventoryService->checkBulkAvailability($request->items);

            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => 'Bulk availability check completed successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Display stock adjustments listing.
     */
    public function adjustments(Request $request)
    {
        Gate::authorize('adjust', \App\Models\Warehouse\Inventory::class);

        // Handle DataTable requests
        if ($request->expectsJson() && $request->has('draw')) {
            try {
                return $this->getAdjustmentsDataForTable($request);
            } catch (\Exception $e) {
                return response()->json([
                    'error' => 'DataTable Error: ' . $e->getMessage(),
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ], 500);
            }
        }

        // Calculate real statistics for the cards
        $today = now()->startOfDay();
        $thisMonth = now()->startOfMonth();

        $todaysAdjustments = \App\Models\Warehouse\StockMovement::where('movement_type', 'adjustment')
            ->whereDate('created_at', $today)
            ->count();

        $thisMonthsAdjustments = \App\Models\Warehouse\StockMovement::where('movement_type', 'adjustment')
            ->whereDate('created_at', '>=', $thisMonth)
            ->count();

        $negativeAdjustments = \App\Models\Warehouse\StockMovement::where('movement_type', 'adjustment')
            ->whereDate('created_at', '>=', $thisMonth)
            ->whereRaw('quantity_after < quantity_before')
            ->count();

        // Calculate value impact (approximate based on average unit prices)
        $valueImpact = \App\Models\Warehouse\StockMovement::join('inventory', 'stock_movements.item_id', '=', 'inventory.item_id')
            ->where('stock_movements.movement_type', 'adjustment')
            ->whereDate('stock_movements.created_at', '>=', $thisMonth)
            ->selectRaw('SUM(ABS(stock_movements.quantity_after - stock_movements.quantity_before) * COALESCE(inventory.unit_price, 0)) as total_value')
            ->value('total_value') ?? 0;

        $statistics = [
            'todays_adjustments' => $todaysAdjustments,
            'monthly_adjustments' => $thisMonthsAdjustments,
            'negative_adjustments' => $negativeAdjustments,
            'value_impact' => round($valueImpact, 2)
        ];

        // Return stock adjustments view with real statistics
        return view('warehouse.inventory.adjustments', compact('statistics'));
    }

    /**
     * Get stock adjustments data for DataTable.
     */
    private function getAdjustmentsDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        // Get real stock adjustments from database
        $query = \App\Models\Warehouse\StockMovement::with(['item', 'user'])
            ->where('movement_type', 'adjustment')
            ->orderBy('created_at', 'desc');

        // Apply search filter
        if (!empty($search)) {
            $query->whereHas('item', function ($q) use ($search) {
                $q->where('item_description', 'LIKE', "%{$search}%")
                  ->orWhere('item_code', 'LIKE', "%{$search}%");
            })->orWhere('notes', 'LIKE', "%{$search}%");
        }

        // Apply date filters if provided
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->get('start_date'));
        }
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->get('end_date'));
        }

        // Get total counts
        $totalRecords = \App\Models\Warehouse\StockMovement::where('movement_type', 'adjustment')->count();
        $filteredRecords = $query->count();

        // Apply pagination
        $stockMovements = $query->skip($start)->take($length)->get();

        // Format data for DataTable
        $adjustments = $stockMovements->map(function ($movement) {
            $adjustmentAmount = $movement->quantity_after - $movement->quantity_before;

            // Determine adjustment display
            $adjustmentDisplay = '';
            $adjustmentType = '';

            if ($adjustmentAmount > 0) {
                $adjustmentDisplay = '+' . number_format(abs($adjustmentAmount), 2);
                $adjustmentType = 'Increase';
            } elseif ($adjustmentAmount < 0) {
                $adjustmentDisplay = '-' . number_format(abs($adjustmentAmount), 2);
                $adjustmentType = 'Decrease';
            } else {
                $adjustmentDisplay = '→ ' . number_format($movement->quantity_after, 2);
                $adjustmentType = 'Set';
            }

            return [
                'id' => $movement->id,
                'date' => $movement->created_at->format('Y-m-d H:i:s'),
                'item_name' => $movement->item->item_description ?? 'Unknown Item',
                'item_code' => $movement->item->item_code ?? 'N/A',
                'type' => $adjustmentType,
                'old_quantity' => number_format($movement->quantity_before, 2),
                'adjustment' => $adjustmentDisplay,
                'new_quantity' => number_format($movement->quantity_after, 2),
                'reason' => $movement->notes ?? 'No reason provided',
                'user' => $movement->user->name ?? 'Unknown User',
                'location' => 'Main Warehouse'
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $adjustments->toArray()
        ]);
    }

    /**
     * Store a new stock adjustment.
     */
    public function storeAdjustment(Request $request)
    {
        Gate::authorize('adjust', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'item_id' => 'required|exists:items,id',
            'type' => 'required|in:increase,decrease,set',
            'quantity' => 'required|numeric|min:0',
            'reason' => 'required|string|max:1000',
            'location' => 'nullable|string|max:255'
        ]);

        try {
            \DB::beginTransaction();

            $item = \App\Models\Warehouse\Item::findOrFail($request->item_id);

            // Get current inventory for this item
            $inventory = \App\Models\Warehouse\Inventory::firstOrCreate([
                'item_id' => $request->item_id,
                'location' => $request->location ?? 'Main Warehouse'
            ], [
                'division_id' => null,
                'supplier_id' => null,
                'quantity_available' => 0,
                'unit_price' => 0,
                'purchase_price' => 0,
                'vat_amount' => 0,
                'currency' => 'AED',
                'total_value' => 0,
                'batch_number' => null,
                'supplier_batch_number' => null,
                'purchase_order_number' => null,
                'production_date' => null,
                'expiry_date' => null,
                'status' => 'out_of_stock',
                'delivery_status' => 'received',
                'delivery_type' => null,
                'delivery_reference' => null,
                'delivered_at' => now(),
                'dispatched_at' => null,
                'quality_status' => 'approved',
                'quality_notes' => null,
                'compliance_verified' => true,
                'last_updated' => now()
            ]);

            $oldQuantity = $inventory->quantity_available;
            $adjustmentAmount = $request->quantity;
            $newQuantity = $oldQuantity;

            // Calculate new quantity based on adjustment type
            switch ($request->type) {
                case 'increase':
                    $newQuantity = $oldQuantity + $adjustmentAmount;
                    break;
                case 'decrease':
                    $newQuantity = max(0, $oldQuantity - $adjustmentAmount); // Prevent negative inventory
                    $adjustmentAmount = $oldQuantity - $newQuantity; // Actual decrease amount
                    break;
                case 'set':
                    $newQuantity = $adjustmentAmount;
                    $adjustmentAmount = $newQuantity - $oldQuantity;
                    break;
            }

            // Update inventory
            $inventory->update([
                'quantity_available' => $newQuantity,
                'last_updated' => now()
            ]);

            // Create stock movement record
            \App\Models\Warehouse\StockMovement::create([
                'item_id' => $request->item_id,
                'movement_type' => 'adjustment',
                'reference_type' => 'stock_adjustment',
                'reference_id' => 0, // Use 0 instead of null for non-nullable reference_id
                'quantity_before' => $oldQuantity,
                'quantity_moved' => abs($adjustmentAmount),
                'quantity_after' => $newQuantity,
                'division_id' => null,
                'user_id' => auth()->id(),
                'notes' => $request->reason
            ]);

            \DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Stock adjustment created successfully',
                'data' => [
                    'item' => $item->item_description . ' (' . $item->item_code . ')',
                    'old_quantity' => $oldQuantity,
                    'new_quantity' => $newQuantity,
                    'adjustment' => ($adjustmentAmount >= 0 ? '+' : '') . $adjustmentAmount,
                    'reason' => $request->reason
                ]
            ]);

        } catch (\Exception $e) {
            \DB::rollback();
            \Log::error('Stock adjustment error: ' . $e->getMessage(), [
                'request' => $request->all(),
                'user_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create stock adjustment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get adjustment details for view modal.
     */
    public function getAdjustmentDetails($id, Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        try {
            $adjustment = \App\Models\Warehouse\StockMovement::with(['item', 'user'])
                ->where('movement_type', 'adjustment')
                ->findOrFail($id);

            $adjustmentAmount = $adjustment->quantity_after - $adjustment->quantity_before;
            $adjustmentDisplay = $adjustmentAmount >= 0 ?
                '+' . number_format($adjustmentAmount, 2) :
                number_format($adjustmentAmount, 2);

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $adjustment->id,
                    'item_name' => $adjustment->item->item_description ?? 'Unknown Item',
                    'item_code' => $adjustment->item->item_code ?? 'N/A',
                    'quantity_before' => number_format($adjustment->quantity_before, 2),
                    'quantity_after' => number_format($adjustment->quantity_after, 2),
                    'adjustment_amount' => $adjustmentAmount,
                    'adjustment_display' => $adjustmentDisplay,
                    'reason' => $adjustment->notes ?? 'No reason provided',
                    'user_name' => $adjustment->user->name ?? 'Unknown User',
                    'created_at' => $adjustment->created_at->format('Y-m-d H:i:s'),
                    'location' => 'Main Warehouse'
                ]
            ]);

        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Adjustment record not found.'
            ], 404);

        } catch (\Exception $e) {
            \Log::error('Error fetching adjustment details: ' . $e->getMessage(), [
                'adjustment_id' => $id,
                'user_id' => auth()->id()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to load adjustment details.'
            ], 500);
        }
    }

    /**
     * Clear all stock adjustment data.
     */
    public function clearAdjustments(Request $request)
    {
        Gate::authorize('adjust', \App\Models\Warehouse\Inventory::class);

        try {
            \DB::beginTransaction();

            // Delete all adjustment stock movements
            $deletedCount = \App\Models\Warehouse\StockMovement::where('movement_type', 'adjustment')->delete();

            \DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Successfully cleared {$deletedCount} stock adjustment records.",
                'deleted_count' => $deletedCount
            ]);

        } catch (\Exception $e) {
            \DB::rollback();
            \Log::error('Error clearing stock adjustments: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to clear stock adjustment data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display inter-division transfers listing.
     */
    public function transfers(Request $request)
    {
        Gate::authorize('transfer', \App\Models\Warehouse\Inventory::class);

        // Handle DataTable requests
        if ($request->expectsJson() && $request->has('draw')) {
            try {
                return $this->getTransfersDataForTable($request);
            } catch (\Exception $e) {
                return response()->json([
                    'error' => 'DataTable Error: ' . $e->getMessage(),
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ], 500);
            }
        }

        // Return transfers view
        return view('warehouse.inventory.transfers');
    }

    /**
     * Get transfers data for DataTable.
     */
    private function getTransfersDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        // For demo, generate sample transfer data
        $totalRecords = 35; // Total transfers
        $filteredRecords = $totalRecords;

        // Generate sample data
        $transfers = collect();
        $statuses = ['pending', 'in_transit', 'completed', 'cancelled'];
        $locations = ['Warehouse-A', 'Warehouse-B', 'Warehouse-C', 'Distribution Center', 'Retail Store 1'];

        for ($i = 0; $i < min($length, $totalRecords - $start); $i++) {
            $status = $statuses[array_rand($statuses)];
            $quantity = rand(1, 50);
            $fromLocation = $locations[array_rand($locations)];
            $toLocation = $locations[array_rand($locations)];
            while ($toLocation === $fromLocation) {
                $toLocation = $locations[array_rand($locations)];
            }

            $transfers->push([
                'id' => $start + $i + 1,
                'transfer_id' => 'TRF-' . str_pad($start + $i + 1, 5, '0', STR_PAD_LEFT),
                'date' => now()->subDays(rand(0, 14))->format('Y-m-d H:i:s'),
                'item_name' => 'Sample Item ' . rand(1, 20),
                'item_code' => 'ITM-' . str_pad(rand(1, 20), 4, '0', STR_PAD_LEFT),
                'quantity' => $quantity,
                'from_location' => $fromLocation,
                'to_location' => $toLocation,
                'status' => ucfirst($status),
                'requested_by' => 'User ' . rand(1, 5),
                'approved_by' => $status !== 'pending' ? 'Manager ' . rand(1, 3) : null,
                'completed_at' => $status === 'completed' ? now()->subDays(rand(0, 7))->format('Y-m-d H:i:s') : null
            ]);
        }

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $transfers->toArray()
        ]);
    }

    /**
     * Get stock movements for an item.
     */
    public function stockMovements($itemId, Request $request)
    {
        Gate::authorize('viewStockMovements', \App\Models\Warehouse\Inventory::class);

        $filters = $request->only(['date_from', 'date_to', 'type']);

        $movements = \App\Models\Warehouse\StockMovement::where('item_id', $itemId)
                    ->when($filters['date_from'] ?? null, function ($query, $date) {
                        return $query->where('created_at', '>=', $date);
                    })
                    ->when($filters['date_to'] ?? null, function ($query, $date) {
                        return $query->where('created_at', '<=', $date);
                    })
                    ->when($filters['type'] ?? null, function ($query, $type) {
                        return $query->where('type', $type);
                    })
                    ->with(['item', 'user'])
                    ->orderBy('created_at', 'desc')
                    ->get();

        return response()->json([
            'success' => true,
            'data' => $movements,
            'message' => 'Stock movements retrieved successfully'
        ]);
    }

    /**
     * Display material requests from outgoing transactions.
     */
    public function materialRequests(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Handle DataTable requests
        if ($request->expectsJson() && $request->has('draw')) {
            try {
                return $this->getMaterialRequestsDataForTable($request);
            } catch (\Exception $e) {
                return response()->json([
                    'error' => 'DataTable Error: ' . $e->getMessage(),
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ], 500);
            }
        }

        // Get real statistics from outgoing transactions (exclude internal inventory transactions)
        $totalRequests = \App\Models\Warehouse\OutgoingTransaction::where('material_request_number', 'not like', '%-INV')->count();
        $pendingRequests = \App\Models\Warehouse\OutgoingTransaction::where('status', 'pending')->where('material_request_number', 'not like', '%-INV')->count();
        $successfulRequests = \App\Models\Warehouse\OutgoingTransaction::where('status', 'successful')->where('material_request_number', 'not like', '%-INV')->count();
        $canceledRequests = \App\Models\Warehouse\OutgoingTransaction::where('status', 'canceled')->where('material_request_number', 'not like', '%-INV')->count();

        $stats = [
            'total_requests' => $totalRequests,
            'pending' => $pendingRequests,
            'dispatched' => $successfulRequests,
            'cancelled' => $canceledRequests
        ];

        return view('warehouse.inventory.material-requests', compact('stats'));
    }

    /**
     * Get material requests data for DataTable.
     */
    private function getMaterialRequestsDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        // Get outgoing transactions (material requests) with relationships
        // Exclude internal inventory transactions (those with -INV suffix)
        $query = \App\Models\Warehouse\OutgoingTransaction::with(['project', 'requester', 'approver', 'items.item'])
            ->where('material_request_number', 'not like', '%-INV')
            ->latest('created_at');

        // Apply search filter
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('material_request_number', 'like', "%{$search}%")
                  ->orWhere('type_of_goods', 'like', "%{$search}%")
                  ->orWhereHas('project', function($subQ) use ($search) {
                      $subQ->where('project_name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('requester', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $totalRecords = \App\Models\Warehouse\OutgoingTransaction::where('material_request_number', 'not like', '%-INV')->count();
        $filteredRecords = $query->count();

        $requests = $query->offset($start)->limit($length)->get();

        // Format data for DataTable
        $data = $requests->map(function($request) {
            return [
                'id' => $request->id,
                'request_number' => $request->reference_number,
                'date' => $request->transaction_date->format('M d, Y'),
                'project' => $request->project ? $request->project->project_name : null,
                'division' => $request->division ? $request->division->name : null,
                'requester' => $request->requester ? $request->requester->name : null,
                'items_count' => $request->items->count(),
                'total_value' => '<span class="dirham-symbol-inline">' . number_format($request->items->sum('total_price'), 2) . '</span>',
                'status' => [
                    'label' => ucfirst($request->status),
                    'class' => $this->getMaterialRequestStatusClass($request->status)
                ],
                'actions' => $this->generateMaterialRequestActions($request)
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data->toArray()
        ]);
    }

    /**
     * Get status CSS class for badge styling.
     */
    private function getStatusClass($status)
    {
        return match($status) {
            'pending' => 'warning',
            'dispatched' => 'success',
            'cancelled' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Get status CSS class for material request badge styling.
     */
    private function getMaterialRequestStatusClass($status)
    {
        return match($status) {
            'pending' => 'warning',
            'successful' => 'success',
            'canceled' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Generate action buttons for material requests.
     */
    private function generateMaterialRequestActions($request)
    {
        $buttons = [];

        // View button - always available
        $buttons[] = '<a href="' . route('warehouse.inventory.material-requests.view', $request->id) . '" class="btn btn-sm btn-outline-primary" title="View Details">
                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                      </a>';

        // Print button - always available
        $buttons[] = '<button class="btn btn-sm btn-outline-info" title="Print Request" onclick="printMaterialRequest(' . $request->id . ')">
                        <i class="material-icons" style="font-size: 16px;">print</i>
                      </button>';

        return implode(' ', $buttons);
    }

    /**
     * View material request details.
     */
    public function viewMaterialRequest($id, Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the actual material request from database
        $transaction = \App\Models\Warehouse\OutgoingTransaction::with([
            'project',
            'requester',
            'approver',
            'items.item',
            'items' => function($query) {
                $query->with(['item.inventories' => function($subQuery) {
                    $subQuery->where('quantity_available', '>', 0);
                }]);
            }
        ])->findOrFail($id);

        // Get fulfillment data for this transaction
        $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $id)
            ->get()
            ->groupBy('item_id');

        // Calculate available quantities for each item
        $items = $transaction->items->map(function($item) use ($transaction, $fulfillments) {
            $query = \App\Models\Warehouse\Inventory::where('item_id', $item->item_id);

            // Only filter by division if it exists
            if ($transaction->division_id) {
                $query->where('division_id', $transaction->division_id);
            }

            $availableQty = $query->sum('quantity_available');

            // Calculate released quantity from all fulfillment records
            $releasedQty = 0;
            if (isset($fulfillments[$item->item_id])) {
                $releasedQty = $fulfillments[$item->item_id]->sum('quantity_fulfilled');
            }

            // Calculate remaining quantity
            $remainingQty = max(0, $item->quantity_requested - $releasedQty);

            return [
                'id' => $item->id,
                'item_id' => $item->item_id,
                'item_code' => $item->item->item_code,
                'description' => $item->item->item_description,
                'unit' => $item->item->unit_of_measure,
                'quantity_requested' => $item->quantity_requested,
                'quantity_released' => $releasedQty, // Updated to show actual released from all sources
                'quantity_released_old' => $item->quantity_released, // Keep original for reference
                'quantity_remaining' => $remainingQty, // Add remaining quantity
                'quantity_available' => $availableQty,
                'unit_price' => $item->unit_price,
                'can_fulfill' => $availableQty >= $remainingQty, // Update to check if remaining can be fulfilled
                'release_details' => $fulfillments[$item->item_id] ?? collect() // Add release breakdown
            ];
        });

        // Calculate overall release status
        $totalRequested = $items->sum('quantity_requested');
        $totalReleased = $items->sum('quantity_released');
        $totalRemaining = $items->sum('quantity_remaining');

        // Determine display status based on transaction status and release progress
        $displayStatus = $transaction->status;

        switch ($transaction->status) {
            case 'pending':
                if ($totalReleased > 0) {
                    if ($totalRemaining > 0) {
                        $displayStatus = "Partially Released ({$totalReleased}/{$totalRequested})";
                    } else {
                        $displayStatus = "Fully Released";
                    }
                } else {
                    $displayStatus = "Pending";
                }
                break;

            case 'successful':
                $displayStatus = "Released";
                break;

            case 'processed_multi_source':
                $displayStatus = "Released (Multi-Source)";
                break;

            case 'canceled':
            case 'cancelled':
                $displayStatus = "Cancelled";
                break;

            case 'rejected':
                $displayStatus = "Rejected";
                break;

            case 'approved':
                if ($totalReleased > 0) {
                    if ($totalRemaining > 0) {
                        $displayStatus = "Partially Released ({$totalReleased}/{$totalRequested})";
                    } else {
                        $displayStatus = "Fully Released";
                    }
                } else {
                    $displayStatus = "Approved";
                }
                break;

            default:
                // For any other status, show it as-is but check for release progress
                if ($totalReleased > 0) {
                    if ($totalRemaining > 0) {
                        $displayStatus = ucfirst($transaction->status) . " - Partially Released ({$totalReleased}/{$totalRequested})";
                    } else {
                        $displayStatus = ucfirst($transaction->status) . " - Fully Released";
                    }
                } else {
                    $displayStatus = ucfirst($transaction->status);
                }
                break;
        }

        $materialRequest = [
            'id' => $transaction->id,
            'request_number' => $transaction->reference_number,
            'material_request_number' => $transaction->material_request_number,
            'request_date' => $transaction->transaction_date->format('M d, Y'),
            'project_id' => $transaction->project_id,
            'project' => $transaction->project ? $transaction->project->project_name : 'N/A',
            'requested_by' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'status' => $displayStatus,
            'original_status' => $transaction->status,
            'total_requested' => $totalRequested,
            'total_released' => $totalReleased,
            'total_remaining' => $totalRemaining,
            'release_percentage' => $totalRequested > 0 ? round(($totalReleased / $totalRequested) * 100, 1) : 0,
            'notes' => $transaction->notes,
            'items' => $items
        ];

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $materialRequest
            ]);
        }

        return view('warehouse.inventory.material-requests.view', compact('materialRequest'));
    }

    /**
     * Show dispatch form for material request.
     */
    public function dispatchMaterialRequest($id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the actual material request from database
        $transaction = \App\Models\Warehouse\OutgoingTransaction::with([
            'project',
            'requester',
            'approver',
            'items.item',
            'items' => function($query) {
                $query->with(['item.inventories' => function($subQuery) {
                    $subQuery->where('quantity_available', '>', 0);
                }]);
            }
        ])->findOrFail($id);

        // Calculate available quantities for each item
        $items = $transaction->items->map(function($item) use ($transaction) {
            $query = \App\Models\Warehouse\Inventory::where('item_id', $item->item_id);

            // Only filter by division if it exists
            if ($transaction->division_id) {
                $query->where('division_id', $transaction->division_id);
            }

            $availableQty = $query->sum('quantity_available');

            // Calculate remaining quantity that needs to be dispatched
            $quantityReleased = $item->quantity_released ?: 0;
            $quantityRemaining = $item->quantity_requested - $quantityReleased;

            return [
                'id' => $item->id,
                'item_id' => $item->item_id,
                'item_code' => $item->item->item_code,
                'description' => $item->item->item_description,
                'unit' => $item->item->unit_of_measure,
                'quantity_requested' => $item->quantity_requested,
                'quantity_released' => $quantityReleased,
                'quantity_remaining' => $quantityRemaining,
                'quantity_available' => $availableQty,
                'unit_price' => $item->unit_price,
                'can_fulfill' => $availableQty >= $quantityRemaining
            ];
        });

        $materialRequest = [
            'id' => $transaction->id,
            'request_number' => $transaction->reference_number,
            'material_request_number' => $transaction->material_request_number,
            'request_date' => $transaction->transaction_date->format('Y-m-d'),
            'project_id' => $transaction->project_id,
            'project' => $transaction->project ? $transaction->project->project_name : 'N/A',
            'requested_by' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'status' => $transaction->status,
            'notes' => $transaction->notes,
            'items' => $items
        ];

        // Get projects and users for form dropdowns
        $projects = \App\Models\Warehouse\Project::select('id', 'project_number', 'project_name')->get();
        $users = \App\Models\User::select('id', 'name')->get();

        return view('warehouse.inventory.material-requests.dispatch', compact('materialRequest', 'projects', 'users'));
    }

    /**
     * Process material request dispatch.
     */
    public function processDispatch(Request $request, $id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        $request->validate([
            'action' => 'required|in:approve,cancel,force_complete',
            'items' => 'required_if:action,approve|array|min:1',
            'items.*.item_id' => 'required_if:action,approve|exists:outgoing_items,id',
            'items.*.quantity_approved' => 'required_if:action,approve|numeric|min:0'
        ]);

        try {
            $transaction = \App\Models\Warehouse\OutgoingTransaction::findOrFail($id);

            if ($request->action === 'approve') {
                // Update only the released quantities, keep original requested amounts
                foreach ($request->items as $itemData) {
                    $outgoingItem = \App\Models\Warehouse\OutgoingItem::findOrFail($itemData['item_id']);
                    // Keep original quantity_requested, only update quantity_released
                    $outgoingItem->quantity_released = $itemData['quantity_approved'];
                    $outgoingItem->save();
                }

                // Approve and dispatch in one step
                $transaction->approveAndDispatch(auth()->id());

                return redirect()->route('warehouse.inventory.material-issued')
                               ->with('success', 'Material request approved and dispatched successfully. Inventory has been updated.');

            } elseif ($request->action === 'force_complete') {
                // Force complete: Set quantity_requested to quantity_released for all items
                foreach ($transaction->items as $outgoingItem) {
                    // Update quantity_requested to match quantity_released (effectively completing with current dispatch)
                    $outgoingItem->quantity_requested = $outgoingItem->quantity_released ?: 0;
                    $outgoingItem->save();
                }

                // Mark transaction as successful/completed
                $transaction->status = 'successful';
                $transaction->save();

                return redirect()->route('warehouse.inventory.material-issued')
                               ->with('success', 'Material request force completed. Request marked as fulfilled with dispatched quantities.');

            } elseif ($request->action === 'cancel') {
                // Cancel the transaction
                $transaction->cancel();

                return redirect()->route('warehouse.inventory.material-requests')
                               ->with('success', 'Material request has been canceled.');
            }

            return back()->withErrors(['error' => 'Invalid action specified.'])->withInput();

        } catch (\Exception $e) {
            \Log::error('Material request dispatch error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            return back()->withErrors(['error' => 'Failed to process request: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Delete material request.
     */
    public function deleteMaterialRequest($id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        try {
            // Here you would implement the actual deletion logic
            // For demo purposes, we'll just return success

            return response()->json([
                'success' => true,
                'message' => 'Material request deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Print material request note.
     */
    public function printMaterialRequest($id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the material request transaction from database
        $transaction = \App\Models\Warehouse\OutgoingTransaction::with([
            'project',
            'requester',
            'approver',
            'items.item'
        ])->findOrFail($id);

        // Prepare data for the material request print template
        $materialRequest = [
            'id' => $transaction->id,
            'request_number' => $transaction->reference_number ?: 'REQ-' . $transaction->id,
            'material_request_number' => $transaction->material_request_number,
            'request_date' => $transaction->transaction_date->format('M d, Y'),
            'project' => $transaction->project ? $transaction->project->project_name : 'N/A',
            'requested_by' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'approved_by' => $transaction->approver ? $transaction->approver->name : 'Pending',
            'status' => $transaction->status,
            'total_items' => $transaction->items->count(),
            'notes' => $transaction->notes ?: '',
            'items' => $transaction->items->map(function($item) {
                return [
                    'item_code' => $item->item->item_code,
                    'description' => $item->item->item_description,
                    'unit' => $item->item->unit_of_measure ?: 'pcs',
                    'quantity_requested' => $item->quantity_requested,
                    'quantity_released' => $item->quantity_released,
                    'unit_price' => $item->unit_price ?: 0,
                    'total_price' => ($item->quantity_requested * ($item->unit_price ?: 0))
                ];
            })->toArray(),
            'total_value' => $transaction->items->sum(function($item) {
                return $item->quantity_requested * ($item->unit_price ?: 0);
            })
        ];

        return view('warehouse.inventory.material-requests.print', compact('materialRequest'));
    }

    /**
     * Display material issued/dispatched items.
     */
    public function materialIssued(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Handle DataTable requests
        if ($request->expectsJson() && $request->has('draw')) {
            try {
                return $this->getMaterialIssuedDataForTable($request);
            } catch (\Exception $e) {
                return response()->json([
                    'error' => 'DataTable Error: ' . $e->getMessage(),
                    'draw' => intval($request->get('draw', 1)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => []
                ], 500);
            }
        }

        // Get statistics from dispatched/issued outgoing transactions
        $totalIssued = \App\Models\Warehouse\OutgoingTransaction::whereIn('status', ['partially_dispatched', 'successful'])->count();
        $recentlyIssued = \App\Models\Warehouse\OutgoingTransaction::whereIn('status', ['partially_dispatched', 'successful'])
                          ->where('updated_at', '>=', now()->subDays(7))->count();
        $pendingIssues = \App\Models\Warehouse\OutgoingTransaction::where('status', 'pending')->count();
        $totalValue = \App\Models\Warehouse\OutgoingItem::whereHas('transaction', function($q) {
            $q->whereIn('status', ['partially_dispatched', 'successful']);
        })->sum('total_price');

        $stats = [
            'total_issued' => $totalIssued,
            'recently_issued' => $recentlyIssued,
            'pending_issues' => $pendingIssues,
            'total_value' => $totalValue
        ];

        return view('warehouse.inventory.material-issued', compact('stats'));
    }

    /**
     * Get material issued data for DataTable.
     */
    private function getMaterialIssuedDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        // Get outgoing transactions (pending for dispatch + partially/fully issued for tracking)
        $query = \App\Models\Warehouse\OutgoingTransaction::with(['project', 'requester', 'approver', 'items.item'])
            ->whereIn('status', ['pending', 'partially_dispatched', 'successful']) // Show pending, partially dispatched, and fully issued
            ->latest('updated_at');

        // Apply search filter
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('material_request_number', 'like', "%{$search}%")
                  ->orWhere('type_of_goods', 'like', "%{$search}%")
                  ->orWhereHas('project', function($subQ) use ($search) {
                      $subQ->where('project_name', 'like', "%{$search}%");
                  })
                  ->orWhereHas('requester', function($subQ) use ($search) {
                      $subQ->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $totalRecords = \App\Models\Warehouse\OutgoingTransaction::whereIn('status', ['pending', 'partially_dispatched', 'successful'])->count();
        $filteredRecords = $query->count();

        $issuedItems = $query->offset($start)->limit($length)->get();

        // Format data for DataTable
        $data = $issuedItems->map(function($transaction) {
            // Calculate actual status based on quantities
            $actualStatus = $this->calculateActualStatus($transaction);

            return [
                'id' => $transaction->id,
                'issue_number' => $transaction->material_request_number ?: 'REQ-' . $transaction->id,
                'date_issued' => $transaction->updated_at->format('M d, Y'),
                'project' => $transaction->project ? $transaction->project->project_name : null,
                'division' => $transaction->division ? $transaction->division->name : null,
                'issued_to' => $transaction->requester ? $transaction->requester->name : null,
                'items_count' => $transaction->items->count(),
                'total_value' => '<span class="dirham-symbol-inline">' . number_format($transaction->items->sum('total_price'), 2) . '</span>',
                'status' => [
                    'label' => match($actualStatus) {
                        'pending' => 'Pending Dispatch',
                        'partially_issued' => 'Partially Issued',
                        'fully_issued' => 'Fully Issued',
                        default => 'Unknown'
                    },
                    'class' => match($actualStatus) {
                        'pending' => 'warning',
                        'partially_issued' => 'info',
                        'fully_issued' => 'success',
                        default => 'secondary'
                    }
                ],
                'actions' => $this->generateMaterialIssuedActions($transaction, $actualStatus)
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data->toArray()
        ]);
    }

    /**
     * Calculate actual status based on requested vs issued quantities.
     */
    private function calculateActualStatus($transaction)
    {
        // If transaction status is still pending, it's pending
        if ($transaction->status === 'pending') {
            return 'pending';
        }

        // Check if any items have been issued
        $totalRequested = $transaction->items->sum('quantity_requested');
        $totalIssued = $transaction->items->sum('quantity_released');

        if ($totalIssued <= 0) {
            return 'pending';
        } elseif ($totalIssued < $totalRequested) {
            return 'partially_issued';
        } else {
            return 'fully_issued';
        }
    }

    /**
     * Generate action buttons for material issued items.
     */
    private function generateMaterialIssuedActions($transaction, $actualStatus = null)
    {
        $buttons = [];

        // View button - always available
        $buttons[] = '<a href="' . route('warehouse.inventory.material-issued.view', $transaction->id) . '" class="btn btn-sm btn-outline-primary" title="View Details">
                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                      </a>';

        // Use actual status if provided, otherwise fall back to transaction status
        $status = $actualStatus ?: $this->calculateActualStatus($transaction);

        // Actions based on calculated status
        if ($status === 'pending' || $status === 'partially_issued') {
            // Dispatch button for pending/partially issued items - links to material-requests dispatch page
            $buttonTitle = $status === 'pending' ? 'Dispatch' : 'Complete Dispatch';
            $buttons[] = '<a href="' . route('warehouse.inventory.material-requests.dispatch', $transaction->id) . '" class="btn btn-sm btn-outline-success" title="' . $buttonTitle . '">
                            <i class="material-icons" style="font-size: 16px;">local_shipping</i>
                          </a>';
        }

        // Print/Download button for all materials
        $buttons[] = '<button class="btn btn-sm btn-outline-info" title="Print Issue Note" onclick="printIssueNote(' . $transaction->id . ')">
                        <i class="material-icons" style="font-size: 16px;">print</i>
                      </button>';

        return implode(' ', $buttons);
    }

    /**
     * View material issued details.
     */
    public function viewMaterialIssued($id, Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the actual issued material transaction from database
        $transaction = \App\Models\Warehouse\OutgoingTransaction::with([
            'project',
            'requester',
            'approver',
            'items.item'
        ])->findOrFail($id);

        // Format items data
        $items = $transaction->items->map(function($item) {
            return [
                'id' => $item->id,
                'item_id' => $item->item_id,
                'item_code' => $item->item->item_code,
                'description' => $item->item->item_description,
                'unit' => $item->item->unit_of_measure,
                'quantity_requested' => $item->quantity_requested,
                'quantity_released' => $item->quantity_released,
                'unit_price' => $item->unit_price,
                'total_price' => $item->total_price
            ];
        });

        $materialIssued = [
            'id' => $transaction->id,
            'issue_number' => $transaction->material_request_number ?: 'REQ-' . $transaction->id,
            'material_request_number' => $transaction->material_request_number,
            'issue_date' => $transaction->updated_at->format('M d, Y'),
            'project_id' => $transaction->project_id,
            'project' => $transaction->project ? $transaction->project->project_name : 'N/A',
            'issued_to' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'approved_by' => $transaction->approver ? $transaction->approver->name : 'N/A',
            'status' => $transaction->status,
            'notes' => $transaction->notes,
            'items' => $items
        ];

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $materialIssued
            ]);
        }

        return view('warehouse.inventory.material-issued.view', compact('materialIssued'));
    }

    /**
     * Print material issued note.
     */
    public function printMaterialIssued($id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get the actual issued material transaction from database
        $transaction = \App\Models\Warehouse\OutgoingTransaction::with([
            'project',
            'requester',
            'approver',
            'items.item'
        ])->findOrFail($id);

        // Create a delivery note object that matches our template structure
        $deliveryNote = (object) [
            'outgoingTransaction' => (object) [
                'mr_number' => $transaction->material_request_number ?: 'REQ-' . $transaction->id,
            ],
            'delivered_to' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'generated_date' => $transaction->updated_at,
            'project' => $transaction->project ? (object) [
                'project_number' => $transaction->project->project_number ?: $transaction->project->project_name
            ] : null,
            'items' => $transaction->items->map(function($item) {
                return (object) [
                    'item' => (object) [
                        'item_name' => $item->item->item_description,
                        'unit' => $item->item->unit_of_measure ?: 'pcs'
                    ],
                    'quantity_released' => $item->quantity_released,
                    'remarks' => ''
                ];
            })
        ];

        // Prepare data for the material issued print template
        $materialIssued = [
            'id' => $transaction->id,
            'issue_number' => $transaction->material_request_number ?: 'REQ-' . $transaction->id,
            'issue_date' => $transaction->updated_at->format('M d, Y'),
            'project' => $transaction->project ? $transaction->project->project_name : 'N/A',
            'issued_to' => $transaction->requester ? $transaction->requester->name : 'N/A',
            'approved_by' => $transaction->approver ? $transaction->approver->name : 'System',
            'status' => $transaction->status,
            'total_items' => $transaction->items->count(),
            'notes' => $transaction->notes ?: '',
            'items' => $transaction->items->map(function($item) {
                return [
                    'item_code' => $item->item->item_code,
                    'description' => $item->item->item_description,
                    'unit' => $item->item->unit_of_measure ?: 'pcs',
                    'quantity_requested' => $item->quantity_requested,
                    'quantity_released' => $item->quantity_released,
                    'unit_price' => $item->unit_price ?: 0,
                    'total_price' => ($item->quantity_released * ($item->unit_price ?: 0))
                ];
            })->toArray(),
            'total_value' => $transaction->items->sum(function($item) {
                return $item->quantity_released * ($item->unit_price ?: 0);
            })
        ];

        return view('warehouse.inventory.material-issued.print', compact('materialIssued'));
    }

    /**
     * Show dispatch form for material issued.
     */
    public function dispatchMaterialIssued($id)
    {
        // This method would be used for re-dispatch or transfer scenarios
        // For now, redirect to view since materials are already issued
        return redirect()->route('warehouse.inventory.material-issued.view', $id);
    }

    /**
     * Process issued material operations.
     */
    public function processIssuedDispatch(Request $request, $id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        try {
            // This would handle any post-issue operations
            // For now, just return success
            return response()->json([
                'success' => true,
                'message' => 'Operation completed successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Delete material issued record.
     */
    public function deleteMaterialIssued($id)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        try {
            // Here you would implement the actual deletion/reversal logic
            // For demo purposes, we'll just return success
            return response()->json([
                'success' => true,
                'message' => 'Material issued record deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}