<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\Project;
use App\Models\Department;
use App\Models\User;
use Illuminate\Http\Request;

class OutgoingTransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = OutgoingTransaction::with(['project', 'division', 'requester', 'approver', 'items'])
                                   ->latest('transaction_date');

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('division_id')) {
            $query->where('division_id', $request->division_id);
        }

        if ($request->filled('date_from')) {
            $query->where('transaction_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('transaction_date', '<=', $request->date_to);
        }

        $transactions = $query->paginate(15);

        // Get filter options
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();

        return view('warehouse.outgoing.index', compact(
            'transactions', 'projects', 'divisions'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $users = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_manager', 'warehouse_supervisor']);
        })->get();

        // Get items with their current stock levels
        $items = \App\Models\Warehouse\Item::active()
            ->with(['inventories' => function($query) {
                $query->where('quantity_available', '>', 0);
            }])
            ->get()
            ->map(function($item) {
                $item->current_stock = $item->inventories->sum('quantity_available');
                return $item;
            });

        return view('warehouse.outgoing.create', compact(
            'projects', 'divisions', 'users', 'items'
        ));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'request_date' => 'required|date',
            'project_id' => 'required|exists:projects,id',
            'material_request_number' => 'required|string|max:255',
            'requested_by' => 'required|exists:users,id',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.quantity_requested' => 'required|numeric|min:0.01',
        ]);

        // Get project to determine division
        $project = \App\Models\Warehouse\Project::findOrFail($request->project_id);

        // Create outgoing transaction (pending decision)
        $transaction = \App\Models\Warehouse\OutgoingTransaction::create([
            'transaction_date' => $request->request_date,
            'project_id' => $request->project_id,
            'division_id' => $project->division_id,
            'material_request_number' => $request->material_request_number,
            'type_of_goods' => 'Material Request',
            'requested_by' => $request->requested_by,
            'status' => 'pending', // Will be processed in Material Issue Decision Center
            'notes' => $request->notes ?? 'Pending fulfillment decision',
        ]);

        // Create transaction items
        foreach ($request->items as $itemData) {
            $item = \App\Models\Warehouse\Item::findOrFail($itemData['item_id']);

            // Get average unit price from inventory
            $averagePrice = \App\Models\Warehouse\Inventory::where('item_id', $item->id)
                ->where('quantity_available', '>', 0)
                ->avg('unit_price') ?: 0;

            \App\Models\Warehouse\OutgoingItem::create([
                'outgoing_transaction_id' => $transaction->id,
                'item_id' => $item->id,
                'quantity_requested' => $itemData['quantity_requested'],
                'quantity_released' => 0, // Will be filled during decision processing
                'unit_price' => $averagePrice,
            ]);
        }

        return redirect()->route('warehouse.material-issue.index')
            ->with('success', 'Material request created successfully. Please proceed to Material Issue Decision Center to process this request.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $transaction = OutgoingTransaction::with(['project', 'division', 'requester', 'approver', 'items.item'])
                                         ->findOrFail($id);

        return view('warehouse.outgoing.show', compact('transaction'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $transaction = OutgoingTransaction::with(['items.item'])->findOrFail($id);

        if ($transaction->status === 'completed') {
            return back()->with('error', 'Cannot edit completed transactions.');
        }

        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $users = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_manager', 'warehouse_supervisor']);
        })->get();

        return view('warehouse.outgoing.edit', compact(
            'transaction', 'projects', 'divisions', 'users'
        ));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $transaction = OutgoingTransaction::findOrFail($id);

        if ($transaction->status === 'completed') {
            return back()->with('error', 'Cannot update completed transactions.');
        }

        $request->validate([
            'transaction_date' => 'required|date',
            'project_id' => 'required|exists:projects,id',
            'division_id' => 'required|exists:departments,id',
            'material_request_number' => 'nullable|string|max:255',
            'type_of_goods' => 'required|string|max:255',
            'requested_by' => 'required|exists:users,id',
            'receiver_name' => 'required|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $transaction->update($request->validated());

        return redirect()->route('warehouse.outgoing.show', $transaction)
                        ->with('success', 'Outgoing transaction updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $transaction = OutgoingTransaction::findOrFail($id);

        if ($transaction->status === 'completed') {
            return back()->with('error', 'Cannot delete completed transactions.');
        }

        $transaction->delete();

        return redirect()->route('warehouse.outgoing.index')
                        ->with('success', 'Outgoing transaction deleted successfully.');
    }

    /**
     * Show pending approval transactions.
     */
    public function pendingApproval()
    {
        $transactions = OutgoingTransaction::with(['project', 'division', 'requester', 'items'])
                                          ->where('status', 'pending')
                                          ->latest('transaction_date')
                                          ->paginate(15);

        return view('warehouse.outgoing.pending-approval', compact('transactions'));
    }

    /**
     * Approve a transaction.
     */
    public function approve(Request $request, string $id)
    {
        $transaction = OutgoingTransaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return back()->with('error', 'Only pending transactions can be approved.');
        }

        $transaction->approve(auth()->id());

        return back()->with('success', 'Transaction approved successfully.');
    }

    /**
     * Dispatch a transaction.
     */
    public function dispatch(string $id)
    {
        $transaction = OutgoingTransaction::findOrFail($id);

        if ($transaction->status !== 'approved') {
            return back()->with('error', 'Only approved transactions can be dispatched.');
        }

        if (!$transaction->can_be_fulfilled) {
            return back()->with('error', 'Insufficient inventory to fulfill this transaction.');
        }

        $transaction->complete();

        return back()->with('success', 'Transaction dispatched and inventory updated successfully.');
    }
}