<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\ItemCategory;
use App\Models\Warehouse\Inventory;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use App\Models\Warehouse\OutgoingItem;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Models\Warehouse\ReturnItem;
use App\Models\Warehouse\ReturnTransaction;
use App\Models\Warehouse\Supplier;
use App\Models\Department;
use App\Models\DirectDelivery;
use App\Models\Warehouse\MaterialTransferRequest;
use App\Models\Warehouse\MaterialTransferRequestItem;
use App\Models\Warehouse\OutgoingFulfillment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        return view('warehouse.reports.index');
    }

    public function inventory(Request $request)
    {
        $query = Item::with(['category', 'inventories.division'])
            ->select('items.*')
            ->leftJoin('inventory', 'items.id', '=', 'inventory.item_id')
            ->groupBy('items.id');

        // Apply filters
        if ($request->filled('category_id')) {
            $query->where('items.category_id', $request->category_id);
        }

        if ($request->filled('division_id')) {
            $query->where('inventory.division_id', $request->division_id);
        }

        if ($request->filled('status')) {
            $query->where('inventory.status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('items.item_code', 'like', "%{$search}%")
                  ->orWhere('items.item_description', 'like', "%{$search}%");
            });
        }

        // Get the items with calculated values
        $items = $query->get()->map(function ($item) {
            // Get all inventory records for this item
            $inventories = $item->inventories;

            // Calculate aggregated data
            $totalQuantity = $inventories->sum('quantity_available');
            $totalValue = $inventories->sum('total_value');

            // Calculate average price (weighted by quantity)
            $averagePrice = $totalQuantity > 0 ? $totalValue / $totalQuantity : 0;

            // Get pricing range from incoming operations
            $pricingData = IncomingOperationItem::where('item_id', $item->id)
                ->where('quantity_delivered', '>', 0)
                ->selectRaw('MIN(unit_price_aed) as min_price, MAX(unit_price_aed) as max_price, AVG(unit_price_aed) as avg_price')
                ->first();

            // Get stock status counts
            $stockStatusCounts = $inventories->groupBy('status')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            // Get latest production and expiry dates
            $latestProduction = $inventories->whereNotNull('production_date')->max('production_date');
            $latestExpiry = $inventories->whereNotNull('expiry_date')->max('expiry_date');

            // Get divisions where this item is stored
            $divisions = $inventories->groupBy('division_id')->map(function ($group) {
                $division = $group->first()->division;
                return [
                    'name' => $division ? $division->name : 'Unknown',
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            return (object) [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category' => $item->category ? $item->category->category_name : 'No Category',
                'category_id' => $item->category_id,
                'unit_of_measure' => $item->unit_of_measure,
                'total_quantity' => $totalQuantity,
                'average_unit_price' => $averagePrice,
                'total_value' => $totalValue,
                'min_price' => $pricingData->min_price ?? 0,
                'max_price' => $pricingData->max_price ?? 0,
                'avg_purchase_price' => $pricingData->avg_price ?? 0,
                'latest_production_date' => $latestProduction,
                'latest_expiry_date' => $latestExpiry,
                'divisions' => $divisions,
                'stock_status_counts' => $stockStatusCounts,
                'overall_status' => $this->determineOverallStatus($stockStatusCounts, $totalQuantity)
            ];
        });

        // Filter out items with no inventory if requested
        if ($request->has('exclude_zero_stock') && $request->exclude_zero_stock) {
            $items = $items->filter(function ($item) {
                return $item->total_quantity > 0;
            });
        }

        // Sort items
        $sortBy = $request->get('sort', 'item_code');
        $sortOrder = $request->get('order', 'asc');

        $items = $items->sortBy($sortBy, SORT_REGULAR, $sortOrder === 'desc');

        // Get filter options
        $categories = ItemCategory::all();
        $divisions = Department::all();
        $stockStatuses = ['in_stock', 'low_stock', 'out_of_stock'];

        return view('warehouse.reports.inventory', compact('items', 'categories', 'divisions', 'stockStatuses'));
    }

    public function inventoryItemDetails($itemId)
    {
        $item = Item::with(['category', 'inventories.division'])->findOrFail($itemId);

        // Get detailed pricing history from incoming operations
        $pricingHistory = IncomingOperationItem::with(['operation.supplier'])
            ->where('item_id', $itemId)
            ->where('quantity_delivered', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($incomingItem) {
                return (object) [
                    'date' => $incomingItem->created_at,
                    'supplier' => $incomingItem->operation->supplier->name ?? 'Unknown',
                    'quantity' => $incomingItem->quantity_delivered,
                    'unit_price' => $incomingItem->unit_price_aed,
                    'total_price' => $incomingItem->total_price_aed,
                    'currency' => $incomingItem->currency,
                    'original_unit_price' => $incomingItem->unit_price,
                    'batch_number' => $incomingItem->batch_number,
                    'operation_number' => $incomingItem->operation->operation_number ?? 'N/A'
                ];
            });

        // Get detailed inventory breakdown by division and batch
        $inventoryDetails = $item->inventories()
            ->with('division')
            ->where('quantity_available', '>', 0)
            ->orderBy('expiry_date', 'asc')
            ->get()
            ->map(function ($inventory) {
                return (object) [
                    'division' => $inventory->division->name ?? 'Unknown',
                    'location' => $inventory->location,
                    'batch_number' => $inventory->batch_number,
                    'quantity' => $inventory->quantity_available,
                    'unit_price' => $inventory->unit_price,
                    'total_value' => $inventory->total_value,
                    'production_date' => $inventory->production_date,
                    'expiry_date' => $inventory->expiry_date,
                    'status' => $inventory->status,
                    'days_until_expiry' => $inventory->days_until_expiry
                ];
            });

        return view('warehouse.reports.inventory-details', compact('item', 'pricingHistory', 'inventoryDetails'));
    }

    private function determineOverallStatus($statusCounts, $totalQuantity)
    {
        if ($totalQuantity <= 0) {
            return 'out_of_stock';
        }

        $lowStockQty = $statusCounts->get('low_stock')['quantity'] ?? 0;
        $inStockQty = $statusCounts->get('in_stock')['quantity'] ?? 0;

        if ($lowStockQty > 0 && $inStockQty === 0) {
            return 'low_stock';
        }

        return 'in_stock';
    }

    public function stockMovements()
    {
        return view('warehouse.reports.stock-movements');
    }

    public function transactions(Request $request)
    {
        // Collect all transactions from different sources
        $transactions = collect();

        // Get Incoming Operations (exclude site returns as they're separate)
        $incomingQuery = IncomingOperation::with(['supplier', 'project', 'division', 'receiver'])
            ->where('operation_type', '!=', 'site_return');

        // Get Outgoing Transactions
        $outgoingQuery = OutgoingTransaction::with(['project', 'division', 'requester']);

        // Get Direct Deliveries
        $directDeliveryQuery = DirectDelivery::with(['supplier', 'project', 'projectDivision']);

        // Get Material Transfer Requests
        $transferQuery = MaterialTransferRequest::with(['transferrerProject', 'receiverProject', 'requestedBy']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $incomingQuery->where(function($q) use ($search) {
                $q->where('operation_number', 'like', "%{$search}%")
                  ->orWhere('lpo_number', 'like', "%{$search}%")
                  ->orWhere('delivery_note_number', 'like', "%{$search}%");
            });

            $outgoingQuery->where(function($q) use ($search) {
                $q->where('transaction_number', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%");
            });

            $directDeliveryQuery->where(function($q) use ($search) {
                $q->where('reference_number', 'like', "%{$search}%");
            });

            $transferQuery->where(function($q) use ($search) {
                $q->where('transfer_request_number', 'like', "%{$search}%");
            });
        }

        if ($request->filled('date_from')) {
            $dateFrom = $request->date_from;
            $incomingQuery->whereDate('operation_date', '>=', $dateFrom);
            $outgoingQuery->whereDate('transaction_date', '>=', $dateFrom);
            $directDeliveryQuery->whereDate('delivery_date', '>=', $dateFrom);
            $transferQuery->whereDate('request_date', '>=', $dateFrom);
        }

        if ($request->filled('date_to')) {
            $dateTo = $request->date_to;
            $incomingQuery->whereDate('operation_date', '<=', $dateTo);
            $outgoingQuery->whereDate('transaction_date', '<=', $dateTo);
            $directDeliveryQuery->whereDate('delivery_date', '<=', $dateTo);
            $transferQuery->whereDate('request_date', '<=', $dateTo);
        }

        // Get the data
        $incoming = $incomingQuery->get();
        $outgoing = $outgoingQuery->get();
        $directDeliveries = $directDeliveryQuery->get();
        $transfers = $transferQuery->get();

        // Format and merge all transactions
        foreach ($incoming as $item) {
            $transactions->push([
                'id' => $item->operation_number,
                'date' => $item->operation_date,
                'type' => ucfirst($item->operation_type),
                'type_badge' => 'success',
                'reference' => $item->lpo_number ?: $item->delivery_note_number,
                'entity' => $item->supplier?->name ?: 'N/A',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->receiver?->name ?: 'System',
                'model' => 'incoming',
                'model_id' => $item->id
            ]);
        }

        foreach ($outgoing as $item) {
            $transactions->push([
                'id' => $item->transaction_number,
                'date' => $item->transaction_date,
                'type' => 'Outgoing',
                'type_badge' => 'info',
                'reference' => $item->reference_number,
                'entity' => $item->project?->project_name ?: 'N/A',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->requester?->name ?: 'System',
                'model' => 'outgoing',
                'model_id' => $item->id
            ]);
        }

        foreach ($directDeliveries as $item) {
            $transactions->push([
                'id' => $item->reference_number ?: $item->delivery_number,
                'date' => $item->delivery_date,
                'type' => 'Direct Delivery',
                'type_badge' => 'warning',
                'reference' => $item->reference_number ?: $item->delivery_number,
                'entity' => $item->supplier?->name ?: 'Direct Delivery',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => 'System',
                'model' => 'direct_delivery',
                'model_id' => $item->id
            ]);
        }

        foreach ($transfers as $item) {
            $transactions->push([
                'id' => $item->transfer_request_number,
                'date' => $item->request_date,
                'type' => 'Transfer',
                'type_badge' => 'primary',
                'reference' => $item->transfer_request_number,
                'entity' => $item->transferrerProject?->project_name ?: 'N/A',
                'project' => $item->receiverProject?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->requestedBy?->name ?: 'System',
                'model' => 'transfer',
                'model_id' => $item->id
            ]);
        }

        // Filter by type if specified
        if ($request->filled('type')) {
            $typeFilter = $request->type;
            $transactions = $transactions->filter(function($item) use ($typeFilter) {
                switch($typeFilter) {
                    case 'supplier_delivery':
                        return in_array(strtolower($item['type']), ['supply_delivery', 'supplier_delivery']);
                    case 'outgoing':
                        return strtolower($item['type']) === 'outgoing';
                    case 'direct_delivery':
                        return strtolower($item['type']) === 'direct delivery';
                    case 'transfer':
                        return strtolower($item['type']) === 'transfer';
                    default:
                        return true;
                }
            });
        }

        // Filter by status if specified
        if ($request->filled('status')) {
            $statusFilter = strtolower($request->status);
            $transactions = $transactions->filter(function($item) use ($statusFilter) {
                return strtolower($item['status']) === $statusFilter;
            });
        }

        // Sort by date descending
        $transactions = $transactions->sortByDesc(function ($item) {
            return $item['date'];
        })->values();

        // Calculate statistics (before pagination)
        $stats = [
            'total' => $transactions->count(),
            'incoming' => $incoming->count(),
            'outgoing' => $outgoing->count(),
            'direct_deliveries' => $directDeliveries->count(),
            'transfers' => $transfers->count()
        ];

        // Paginate the transactions
        $perPage = 20;
        $currentPage = request()->input('page', 1);
        $paginatedItems = $transactions->slice(($currentPage - 1) * $perPage, $perPage)->values();

        $transactions = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedItems,
            $transactions->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url()]
        );

        // Append query parameters to pagination links
        $transactions->appends($request->all());

        return view('warehouse.reports.transactions', compact('transactions', 'stats'));
    }

    private function getStatusBadge($status)
    {
        switch(strtolower($status)) {
            case 'completed':
            case 'approved':
                return 'success';
            case 'pending':
            case 'draft':
                return 'warning';
            case 'cancelled':
            case 'rejected':
                return 'danger';
            case 'in_progress':
            case 'partial':
                return 'info';
            default:
                return 'secondary';
        }
    }

    public function aging()
    {
        return view('warehouse.reports.aging');
    }

    public function lowStock()
    {
        return view('warehouse.reports.low-stock');
    }

    public function supplierPerformance()
    {
        return view('warehouse.reports.supplier-performance');
    }

    public function incoming(Request $request)
    {
        $allIncomingItems = collect();

        // Get data from IncomingOperationItems (standard incoming operations)
        $operationQuery = IncomingOperationItem::with([
            'operation.supplier',
            'operation.division',
            'item.category'
        ]);

        // Apply filters for operations
        if ($request->filled('supplier_id')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('supplier_id', $request->supplier_id);
            });
        }

        if ($request->filled('division_id')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('division_id', $request->division_id);
            });
        }

        if ($request->filled('category_id')) {
            $operationQuery->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('operation_type')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('operation_type', $request->operation_type);
            });
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $operationQuery->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('operation', function ($opQuery) use ($search) {
                    $opQuery->where('lpo_number', 'like', "%{$search}%")
                           ->orWhere('operation_number', 'like', "%{$search}%");
                })->orWhereHas('operation.supplier', function ($supQuery) use ($search) {
                    $supQuery->where('name', 'like', "%{$search}%")
                            ->orWhere('supplier_code', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->whereBetween('operation_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get standard incoming operation items
        $operationItems = $operationQuery->orderBy('created_at', 'desc')->get()->map(function ($incomingItem) {
            return (object) [
                'id' => $incomingItem->id,
                'supplier_code' => $incomingItem->operation?->supplier?->supplier_code ?? 'N/A',
                'supplier_name' => $incomingItem->operation?->supplier?->name ?? 'Unknown',
                'lpo_number' => $incomingItem->operation?->lpo_number ?? 'N/A',
                'division' => $incomingItem->operation?->division?->name ?? 'Unknown',
                'item_code' => $incomingItem->item->item_code,
                'item_description' => $incomingItem->item->item_description,
                'unit' => $incomingItem->item->unit_of_measure,
                'quantity' => $incomingItem->quantity_delivered,
                'unit_price' => $incomingItem->unit_price_aed,
                'total_price' => $incomingItem->total_price_aed,
                'production_date' => $incomingItem->production_date,
                'expiry_date' => $incomingItem->expiry_date,
                'operation_date' => $incomingItem->operation?->operation_date ?? null,
                'operation_type' => $incomingItem->operation?->operation_type ?? 'N/A',
                'status' => $incomingItem->operation?->status ?? 'N/A',
                'quality_status' => $incomingItem->quality_status,
                'batch_number' => $incomingItem->batch_number
            ];
        });

        // Get data from IncomingItems (direct delivery transactions)
        $transactionQuery = \App\Models\Warehouse\IncomingItem::with([
            'transaction.supplier',
            'item.category'
        ]);

        // Apply filters for transactions
        if ($request->filled('supplier_id')) {
            $transactionQuery->whereHas('transaction', function ($q) use ($request) {
                $q->where('supplier_id', $request->supplier_id);
            });
        }

        if ($request->filled('category_id')) {
            $transactionQuery->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('operation_type') && $request->operation_type !== 'direct_delivery') {
            // If filtering by operation type and it's not direct_delivery, exclude transaction items
            $transactionQuery->whereRaw('1 = 0'); // This will return no results
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $transactionQuery->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('transaction', function ($transQuery) use ($search) {
                    $transQuery->where('lpo_number', 'like', "%{$search}%")
                              ->orWhere('delivery_note_number', 'like', "%{$search}%");
                })->orWhereHas('transaction.supplier', function ($supQuery) use ($search) {
                    $supQuery->where('name', 'like', "%{$search}%")
                            ->orWhere('supplier_code', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $transactionQuery->whereHas('transaction', function ($q) use ($request) {
                $q->whereBetween('transaction_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get direct delivery transaction items
        $transactionItems = $transactionQuery->orderBy('created_at', 'desc')->get()->map(function ($incomingItem) {
            return (object) [
                'id' => 'T_' . $incomingItem->id, // Prefix to differentiate
                'supplier_code' => $incomingItem->transaction?->supplier?->supplier_code ?? 'N/A',
                'supplier_name' => $incomingItem->transaction?->supplier?->name ?? 'Unknown',
                'lpo_number' => $incomingItem->transaction?->lpo_number ?? 'N/A',
                'division' => 'Direct Delivery', // Custom division label for direct deliveries
                'item_code' => $incomingItem->item->item_code,
                'item_description' => $incomingItem->item->item_description,
                'unit' => $incomingItem->item->unit_of_measure,
                'quantity' => $incomingItem->quantity_received,
                'unit_price' => $incomingItem->unit_price,
                'total_price' => $incomingItem->total_price,
                'production_date' => $incomingItem->production_date,
                'expiry_date' => $incomingItem->expiry_date,
                'operation_date' => $incomingItem->transaction?->transaction_date ?? null,
                'operation_type' => $incomingItem->transaction?->transaction_type ?? 'direct_delivery',
                'status' => $incomingItem->transaction?->status ?? 'N/A',
                'quality_status' => $incomingItem->stock_status,
                'batch_number' => $incomingItem->batch_number
            ];
        });

        // Combine both collections and sort by operation_date
        $incomingItems = collect($operationItems)->merge(collect($transactionItems))
                                       ->sortByDesc(function ($item) {
                                           return $item->operation_date;
                                       })
                                       ->values();

        // Get filter options
        $suppliers = Supplier::all();
        $divisions = Department::all();
        $categories = ItemCategory::all();
        $operationTypes = ['purchase', 'supplier_delivery', 'site_return', 'direct_delivery'];

        return view('warehouse.reports.incoming', compact(
            'incomingItems', 'suppliers', 'divisions', 'categories', 'operationTypes'
        ));
    }

    public function outgoing(Request $request)
    {
        $query = OutgoingTransaction::with([
            'project.projectDivision',
            'division',
            'requester',
            'items.item.category'
        ])->withCount('items')
        // Only show actual material requests, not dispatch references
        // Exclude dispatch reference IDs which have suffixes:
        // -INV (Inventory Dispatch), -DD (Direct Delivery), -TR (Project Transfer), -REF (General Reference)
        // Example: MR-20251209-229-INV is a dispatch reference, MR-20251209-229 is the actual request
        ->whereNotNull('material_request_number')
        ->where(function($q) {
            $q->where('material_request_number', 'NOT LIKE', '%-INV')
              ->where('material_request_number', 'NOT LIKE', '%-DD')
              ->where('material_request_number', 'NOT LIKE', '%-TR')
              ->where('material_request_number', 'NOT LIKE', '%-REF');
        })
        // Also ensure we have a requested_by field (indicating it's an actual request, not a dispatch operation)
        ->whereNotNull('requested_by');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('material_request_number', 'like', "%{$search}%")
                  ->orWhereHas('project', function ($projectQ) use ($search) {
                      $projectQ->where('project_name', 'like', "%{$search}%")
                               ->orWhere('project_code', 'like', "%{$search}%");
                  })
                  ->orWhereHas('requester', function ($userQ) use ($search) {
                      $userQ->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('division_id')) {
            $query->where('division_id', $request->division_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereBetween('transaction_date', [$request->date_from, $request->date_to]);
        }

        // Get the outgoing requests with calculated values using the same logic as material issue page
        $outgoingRequests = $query->orderBy('created_at', 'desc')->orderBy('transaction_date', 'desc')->get();

        // Add calculated properties to each transaction
        foreach ($outgoingRequests as $transaction) {
            // Get fulfillments for this transaction grouped by item_id (exact same logic as MaterialIssueController lines 108-110)
            $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                ->get()
                ->groupBy('item_id');

            // Calculate remaining quantities for each item (exact same logic as MaterialIssueController lines 113-120)
            $totalRequested = 0;
            $totalReleased = 0;

            foreach ($transaction->items as $item) {
                $totalRequested += $item->quantity_requested;

                $issuedQty = 0;
                if (isset($fulfillments[$item->item_id])) {
                    $issuedQty = $fulfillments[$item->item_id]->sum('quantity_fulfilled');
                }
                $totalReleased += $issuedQty;

                // Set the same properties as MaterialIssueController for consistency
                $item->quantity_fulfilled = $issuedQty;
                $item->quantity_remaining = $item->quantity_requested - $issuedQty;
            }

            // Calculate balance
            $totalBalance = $totalRequested - $totalReleased;

            // Determine overall fulfillment status
            $fulfillmentStatus = 'pending';
            if ($totalReleased > 0) {
                $fulfillmentStatus = $totalBalance <= 0 ? 'completed' : 'partial';
            }

            // Count dispatch methods used based on fulfillments only (clean approach matching material issue page)
            $dispatchMethods = [];
            $fulfillmentTypes = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                ->distinct('fulfillment_type')
                ->pluck('fulfillment_type')
                ->toArray();

            foreach ($fulfillmentTypes as $type) {
                switch ($type) {
                    case 'inventory':
                    case 'inventory_dispatch':
                        $dispatchMethods[] = 'Inventory';
                        break;
                    case 'direct_delivery':
                        $dispatchMethods[] = 'Direct Delivery';
                        break;
                    case 'project_transfer':
                    case 'transfer':
                    case 'material_transfer':
                        $dispatchMethods[] = 'Project Transfer';
                        break;
                }
            }

            // Remove duplicates and sort
            $dispatchMethods = array_unique($dispatchMethods);

            // Add calculated properties directly to the Eloquent model
            $transaction->total_requested = $totalRequested;
            $transaction->total_released = $totalReleased;
            $transaction->total_balance = $totalBalance;
            $transaction->fulfillment_status = $fulfillmentStatus;
            $transaction->dispatch_methods = $dispatchMethods;
            $transaction->dispatch_methods_text = empty($dispatchMethods) ? 'Pending' : implode(', ', $dispatchMethods);
        }

        // Filter by fulfillment status if requested
        if ($request->filled('fulfillment_status')) {
            $outgoingRequests = $outgoingRequests->filter(function ($request_item) use ($request) {
                return $request_item->fulfillment_status === $request->fulfillment_status;
            });
        }

        // Get filter options
        $projects = Project::all();
        $divisions = ProjectDivision::all();
        $categories = ItemCategory::all();
        $statuses = ['pending', 'approved', 'partially_fulfilled', 'completed', 'cancelled'];
        $fulfillmentStatuses = ['pending', 'partial', 'completed'];

        return view('warehouse.reports.outgoing', compact(
            'outgoingRequests', 'projects', 'divisions', 'categories', 'statuses', 'fulfillmentStatuses'
        ));
    }

    public function incomingItemDetails($itemId)
    {
        $incomingItem = IncomingOperationItem::with([
            'operation.supplier',
            'operation.division',
            'operation.receiver',
            'operation.inspector',
            'operation.approver',
            'item.category'
        ])->findOrFail($itemId);

        // Get detailed operation information
        $operationDetails = (object) [
            'id' => $incomingItem->operation->id,
            'operation_number' => $incomingItem->operation->operation_number,
            'operation_type' => $incomingItem->operation->operation_type,
            'operation_date' => $incomingItem->operation->operation_date,
            'status' => $incomingItem->operation->status,
            'lpo_number' => $incomingItem->operation->lpo_number,
            'lpo_status' => $incomingItem->operation->lpo_status,
            'delivery_note_number' => $incomingItem->operation->delivery_note_number,
            'supplier_name' => $incomingItem->operation->supplier->name ?? 'Unknown',
            'supplier_code' => $incomingItem->operation->supplier->supplier_code ?? 'N/A',
            'division' => $incomingItem->operation->division ?? 'Unknown',
            'received_by' => $incomingItem->operation->receiver->name ?? $incomingItem->operation->received_by_name ?? 'Unknown',
            'inspector' => $incomingItem->operation->inspector->name ?? 'Not assigned',
            'inspection_datetime' => $incomingItem->operation->inspection_datetime,
            'inspection_result' => $incomingItem->operation->inspection_result,
            'inspection_notes' => $incomingItem->operation->inspection_notes,
            'approved_by' => $incomingItem->operation->approver->name ?? 'Not approved',
            'approved_at' => $incomingItem->operation->approved_at,
            'total_amount' => $incomingItem->operation->total_amount_aed,
            'currency' => $incomingItem->operation->currency,
            'notes' => $incomingItem->operation->notes,
            'created_at' => $incomingItem->operation->created_at,
            'updated_at' => $incomingItem->operation->updated_at
        ];

        // Get detailed item information
        $itemDetails = (object) [
            'id' => $incomingItem->item->id,
            'item_code' => $incomingItem->item->item_code,
            'item_description' => $incomingItem->item->item_description,
            'category_name' => $incomingItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $incomingItem->item->unit_of_measure,
            'quantity_ordered' => $incomingItem->quantity_ordered,
            'quantity_delivered' => $incomingItem->quantity_delivered,
            'quantity_accepted' => $incomingItem->quantity_accepted,
            'quantity_rejected' => $incomingItem->quantity_rejected,
            'unit_price' => $incomingItem->unit_price,
            'unit_price_aed' => $incomingItem->unit_price_aed,
            'total_price_aed' => $incomingItem->total_price_aed,
            'currency' => $incomingItem->currency,
            'batch_number' => $incomingItem->batch_number,
            'serial_number' => $incomingItem->serial_number,
            'production_date' => $incomingItem->production_date,
            'expiry_date' => $incomingItem->expiry_date,
            'quality_status' => $incomingItem->quality_status,
            'quality_notes' => $incomingItem->quality_notes,
            'rejection_reason' => $incomingItem->rejection_reason,
            'storage_location' => $incomingItem->storage_location,
            'manufacturer' => $incomingItem->manufacturer,
            'country_of_origin' => $incomingItem->country_of_origin,
            'warranty_period_months' => $incomingItem->warranty_period_months,
            'warranty_expiry' => $incomingItem->warranty_expiry,
            'hazardous_material' => $incomingItem->hazardous_material,
            'days_until_expiry' => $incomingItem->days_until_expiry,
            'acceptance_rate' => $incomingItem->acceptance_rate,
            'rejection_rate' => $incomingItem->rejection_rate,
            'created_at' => $incomingItem->created_at,
            'updated_at' => $incomingItem->updated_at
        ];

        return view('warehouse.reports.incoming-details', compact('operationDetails', 'itemDetails'));
    }

    public function outgoingItemDetails($itemId)
    {
        $outgoingItem = OutgoingItem::with([
            'transaction.project',
            'transaction.division',
            'transaction.requester',
            'item.category'
        ])->findOrFail($itemId);

        // Get detailed transaction information
        $transactionDetails = (object) [
            'id' => $outgoingItem->transaction->id,
            'material_request_number' => $outgoingItem->transaction->material_request_number,
            'transaction_date' => $outgoingItem->transaction->transaction_date,
            'status' => $outgoingItem->transaction->status,
            'project_name' => $outgoingItem->transaction->project->project_name ?? 'Unknown',
            'project_code' => $outgoingItem->transaction->project->project_code ?? $outgoingItem->transaction->project_id,
            'division_name' => $outgoingItem->transaction->division->division_name ?? 'Unknown',
            'requester_name' => $outgoingItem->transaction->requester->name ?? 'Unknown',
            'receiver_name' => $outgoingItem->transaction->receiver_name,
            'delivery_location' => $outgoingItem->transaction->delivery_location,
            'purpose' => $outgoingItem->transaction->purpose,
            'remarks' => $outgoingItem->transaction->remarks,
            'created_at' => $outgoingItem->transaction->created_at,
            'updated_at' => $outgoingItem->transaction->updated_at
        ];

        // Get item information
        $itemDetails = (object) [
            'id' => $outgoingItem->item->id,
            'item_code' => $outgoingItem->item->item_code,
            'item_description' => $outgoingItem->item->item_description,
            'category_name' => $outgoingItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $outgoingItem->item->unit_of_measure,
            'quantity_requested' => $outgoingItem->quantity_requested,
            'quantity_released' => $outgoingItem->quantity_released,
            'quantity_balance' => $outgoingItem->quantity_balance,
            'unit_price' => $outgoingItem->unit_price,
            'total_price' => $outgoingItem->total_price,
            'fulfillment_status' => $this->determineFulfillmentStatus($outgoingItem),
            'release_date' => $outgoingItem->release_date,
            'batch_number' => $outgoingItem->batch_number,
            'notes' => $outgoingItem->notes,
            'created_at' => $outgoingItem->created_at,
            'updated_at' => $outgoingItem->updated_at
        ];

        return view('warehouse.reports.outgoing-details', compact('transactionDetails', 'itemDetails'));
    }

    public function outgoingRequestDetails($requestId)
    {
        $request = OutgoingTransaction::with([
            'project.projectDivision',
            'division',
            'requester',
            'items.item.category'
        ])->findOrFail($requestId);

        // Get fulfillments for this transaction (same logic as MaterialIssueController and main report)
        $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $request->id)
            ->get()
            ->groupBy('item_id');

        // Calculate remaining quantities for each item and get dispatch details
        $totalRequested = 0;
        $totalReleased = 0;

        foreach ($request->items as $item) {
            $issuedQty = 0;
            $dispatchMethods = [];

            if (isset($fulfillments[$item->item_id])) {
                // Get fulfillment breakdown by type
                $itemFulfillments = $fulfillments[$item->item_id];
                $issuedQty = $itemFulfillments->sum('quantity_fulfilled');

                // Group by fulfillment type for display
                $fulfillmentsByType = $itemFulfillments->groupBy('fulfillment_type');

                // Inventory fulfillments
                $inventoryQty = $fulfillmentsByType->get('inventory', collect())->sum('quantity_fulfilled') +
                               $fulfillmentsByType->get('inventory_dispatch', collect())->sum('quantity_fulfilled');
                if ($inventoryQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Inventory Dispatch',
                        'quantity' => $inventoryQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'inventory')->first()->created_at ??
                                 $itemFulfillments->where('fulfillment_type', 'inventory_dispatch')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'inventory',
                        'color' => 'success'
                    ];
                }

                // Direct delivery fulfillments
                $deliveryQty = $fulfillmentsByType->get('direct_delivery', collect())->sum('quantity_fulfilled');
                if ($deliveryQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Direct Delivery',
                        'quantity' => $deliveryQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'direct_delivery')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'local_shipping',
                        'color' => 'info'
                    ];
                }

                // Project transfer fulfillments
                $transferQty = $fulfillmentsByType->get('project_transfer', collect())->sum('quantity_fulfilled');
                if ($transferQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Project Transfer',
                        'quantity' => $transferQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'project_transfer')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'swap_horiz',
                        'color' => 'primary'
                    ];
                }
            }

            // Set calculated properties
            $item->quantity_fulfilled = $issuedQty;
            $item->quantity_remaining = max(0, $item->quantity_requested - $issuedQty);
            $item->dispatch_methods = $dispatchMethods;
            $item->fulfillment_status = $item->quantity_remaining == 0 ? 'Fully Dispatched' :
                                      ($issuedQty == 0 ? 'Pending' : 'Partially Dispatched');

            $totalRequested += $item->quantity_requested;
            $totalReleased += $issuedQty;
        }

        // Request summary using calculated values
        $requestSummary = (object) [
            'material_request_number' => $request->material_request_number ?: 'MR-' . str_pad($request->id, 6, '0', STR_PAD_LEFT),
            'transaction_date' => $request->transaction_date,
            'project_name' => $request->project->project_name ?? 'N/A',
            'division_name' => $request->division->division_name ??
                              ($request->project->projectDivision->division_name ?? 'Main'),
            'requester_name' => $request->requester->name ?? 'N/A',
            'receiver_name' => $request->receiver_name,
            'delivery_location' => $request->delivery_location,
            'purpose' => $request->purpose,
            'total_items' => $request->items->count(),
            'total_requested' => $totalRequested,
            'total_released' => $totalReleased,
            'total_balance' => $totalRequested - $totalReleased,
            'overall_status' => ($totalRequested - $totalReleased) == 0 ? 'Fully Dispatched' :
                              ($totalReleased == 0 ? 'Pending' : 'Partially Dispatched')
        ];

        return view('warehouse.reports.outgoing-request-details', compact('requestSummary', 'request'));
    }

    public function returns(Request $request)
    {
        // Site returns are stored as IncomingOperationItems with operation_type = 'site_return'
        // They link to original outgoing items via outgoing_item_id
        $query = IncomingOperationItem::with([
            'operation.receiver',
            'item.category'
        ])->whereHas('operation', function ($q) {
            $q->where('operation_type', 'site_return');
        })->where(function($q) {
            $q->whereNotNull('outgoing_item_id')
              ->orWhereNotNull('direct_delivery_item_id')
              ->orWhereNotNull('material_transfer_item_id');
        });

        // Apply filters
        if ($request->filled('project_id')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->where('project_id', $request->project_id);
            });
        }

        if ($request->filled('division_id')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->where('division_id', $request->division_id);
            });
        }

        if ($request->filled('category_id')) {
            $query->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('quality_status')) {
            $query->where('quality_status', $request->quality_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('operation', function ($opQuery) use ($search) {
                    $opQuery->where('operation_number', 'like', "%{$search}%");
                })->orWhereHas('operation.project', function ($projQuery) use ($search) {
                    $projQuery->where('project_name', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->whereBetween('operation_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get the return items with calculated values
        $returnItems = $query->orderBy('created_at', 'desc')->get()->map(function ($returnItem) {
            // Get the source reference info based on the return type
            $sourceInfo = null;
            $projectInfo = null;

            if ($returnItem->outgoing_item_id) {
                // Material Request return
                $outgoingItem = OutgoingItem::with(['transaction.project', 'transaction.division'])
                    ->find($returnItem->outgoing_item_id);
                $sourceInfo = $outgoingItem;
                $projectInfo = $outgoingItem?->transaction?->project;
            } elseif ($returnItem->direct_delivery_item_id) {
                // Direct Delivery return
                $directDeliveryItem = \App\Models\DirectDeliveryItem::with(['directDelivery.project'])
                    ->find($returnItem->direct_delivery_item_id);
                $sourceInfo = $directDeliveryItem;
                $projectInfo = $directDeliveryItem?->directDelivery?->project;
            } elseif ($returnItem->material_transfer_item_id) {
                // Material Transfer return
                $materialTransferItem = \App\Models\Warehouse\MaterialTransferRequestItem::with(['transferRequest.receiverProject'])
                    ->find($returnItem->material_transfer_item_id);
                $sourceInfo = $materialTransferItem;
                $projectInfo = $materialTransferItem?->transferRequest?->receiverProject;
            }

            // Get production/expiry dates from the original incoming operation for this item
            $originalIncoming = IncomingOperationItem::where('item_id', $returnItem->item_id)
                ->where(function($q) {
                    $q->whereNotNull('production_date')->orWhereNotNull('expiry_date');
                })
                ->orderBy('created_at', 'desc')
                ->first();

            return (object) [
                'id' => $returnItem->id,
                'reference_number' => $this->getReferenceNumber($returnItem, $sourceInfo),
                'reference_type' => $this->getReferenceType($returnItem),
                'project_name' => $projectInfo?->project_name ?? $returnItem->operation->project?->project_name ?? 'Unknown',
                'project_number' => $projectInfo?->project_number ?? $projectInfo?->project_code ?? ($projectInfo?->id ?? 'N/A'),
                'division' => $this->getDivisionName($returnItem, $sourceInfo) ?? 'Unknown',
                'type_of_goods' => $returnItem->item->category->category_name ?? 'Unknown',
                'item_code' => $returnItem->item->item_code,
                'item_description' => $returnItem->item->item_description,
                'unit' => $returnItem->item->unit_of_measure,
                'quantity' => $returnItem->quantity_delivered, // For returns, this is quantity returned
                'unit_price' => $returnItem->unit_price_aed ?: ($originalIncoming->unit_price_aed ?? 0),
                'total_price' => $returnItem->total_price_aed ?: (($originalIncoming->unit_price_aed ?? 0) * $returnItem->quantity_delivered),
                'production_date' => $originalIncoming->production_date ?? null,
                'expiry_date' => $originalIncoming->expiry_date ?? null,
                'quality_status' => $returnItem->quality_status,
                'batch_number' => $returnItem->batch_number,
                'operation_number' => $returnItem->operation->operation_number,
                'operation_date' => $returnItem->operation->operation_date,
                'return_reason' => $returnItem->operation->notes ?? 'Site return',
                'received_by' => $returnItem->operation->receiver->name ?? $returnItem->operation->received_by_name ?? 'Unknown'
            ];
        });

        // Get filter options
        $projects = Project::all();
        $divisions = Department::all();
        $categories = ItemCategory::all();
        $qualityStatuses = ['pending', 'passed', 'failed', 'conditional'];

        return view('warehouse.reports.returns', compact(
            'returnItems', 'projects', 'divisions', 'categories', 'qualityStatuses'
        ));
    }

    public function returnItemDetails($itemId)
    {
        // Get the site return item (stored as IncomingOperationItem)
        $returnItem = IncomingOperationItem::with([
            'operation.receiver',
            'operation.inspector',
            'operation.approver',
            'item.category'
        ])->whereHas('operation', function ($q) {
            $q->where('operation_type', 'site_return');
        })->findOrFail($itemId);

        // Get the source reference info based on the return type
        $sourceInfo = null;
        $projectInfo = null;
        $divisionInfo = null;

        if ($returnItem->outgoing_item_id) {
            // Material Request return
            $outgoingItem = OutgoingItem::with(['transaction.project', 'transaction.division'])
                ->find($returnItem->outgoing_item_id);
            $sourceInfo = $outgoingItem;
            $projectInfo = $outgoingItem?->transaction?->project;
            $divisionInfo = $outgoingItem?->transaction?->division;
        } elseif ($returnItem->direct_delivery_item_id) {
            // Direct Delivery return
            $directDeliveryItem = \App\Models\DirectDeliveryItem::with(['directDelivery.project'])
                ->find($returnItem->direct_delivery_item_id);
            $sourceInfo = $directDeliveryItem;
            $projectInfo = $directDeliveryItem?->directDelivery?->project;
            $divisionInfo = $directDeliveryItem?->directDelivery?->project?->projectDivision;
        } elseif ($returnItem->material_transfer_item_id) {
            // Material Transfer return
            $materialTransferItem = \App\Models\Warehouse\MaterialTransferRequestItem::with(['transferRequest.receiverProject'])
                ->find($returnItem->material_transfer_item_id);
            $sourceInfo = $materialTransferItem;
            $projectInfo = $materialTransferItem?->transferRequest?->receiverProject;
            $divisionInfo = $materialTransferItem?->transferRequest?->receiverProject?->projectDivision;
        }

        // Get production/expiry dates from the original incoming operation for this item
        $originalIncoming = IncomingOperationItem::where('item_id', $returnItem->item_id)
            ->where(function($q) {
                $q->whereNotNull('production_date')->orWhereNotNull('expiry_date');
            })
            ->orderBy('created_at', 'desc')
            ->first();

        // Get detailed operation information
        $operationDetails = (object) [
            'id' => $returnItem->operation->id,
            'operation_number' => $returnItem->operation->operation_number,
            'operation_type' => $returnItem->operation->operation_type,
            'operation_date' => $returnItem->operation->operation_date,
            'status' => $returnItem->operation->status,
            'project_name' => $projectInfo?->project_name ?? $returnItem->operation->project?->project_name ?? 'Unknown',
            'project_code' => $projectInfo?->project_code ?? $projectInfo?->project_number ?? ($projectInfo?->id ?? 'N/A'),
            'division_name' => $divisionInfo?->division_name ?? 'Unknown',
            'return_reason' => $returnItem->operation->notes ?? 'Site return',
            'received_by' => $returnItem->operation->receiver->name ?? $returnItem->operation->received_by_name ?? 'Unknown',
            'inspector' => $returnItem->operation->inspector->name ?? 'Not assigned',
            'inspection_datetime' => $returnItem->operation->inspection_datetime,
            'inspection_result' => $returnItem->operation->inspection_result,
            'inspection_notes' => $returnItem->operation->inspection_notes,
            'approved_by' => $returnItem->operation->approver->name ?? 'Not approved',
            'approved_at' => $returnItem->operation->approved_at,
            'notes' => $returnItem->operation->notes,
            'created_at' => $returnItem->operation->created_at,
            'updated_at' => $returnItem->operation->updated_at
        ];

        // Get detailed item information
        $itemDetails = (object) [
            'id' => $returnItem->item->id,
            'item_code' => $returnItem->item->item_code,
            'item_description' => $returnItem->item->item_description,
            'category_name' => $returnItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $returnItem->item->unit_of_measure,
            'quantity_returned' => $returnItem->quantity_delivered, // For returns, this is quantity returned
            'unit_price' => $returnItem->unit_price_aed ?: ($originalIncoming->unit_price_aed ?? 0),
            'total_value' => $returnItem->total_price_aed ?: (($originalIncoming->unit_price_aed ?? 0) * $returnItem->quantity_delivered),
            'production_date' => $originalIncoming->production_date ?? null,
            'expiry_date' => $originalIncoming->expiry_date ?? null,
            'quality_status' => $returnItem->quality_status,
            'quality_notes' => $returnItem->quality_notes,
            'rejection_reason' => $returnItem->rejection_reason,
            'batch_number' => $returnItem->batch_number,
            'serial_number' => $returnItem->serial_number,
            'storage_location' => $returnItem->storage_location,
            'days_until_expiry' => $originalIncoming ? $originalIncoming->days_until_expiry : null,
            'acceptance_rate' => $returnItem->acceptance_rate,
            'created_at' => $returnItem->created_at,
            'updated_at' => $returnItem->updated_at
        ];

        return view('warehouse.reports.returns-details', compact('operationDetails', 'itemDetails'));
    }

    /**
     * Project to Project Transfer Report
     */
    public function projectTransfers(Request $request)
    {
        // Get Material Transfer Requests with related data
        $query = MaterialTransferRequest::with([
            'transferrerProject',
            'receiverProject',
            'requestedBy',
            'transferrerApprovedBy',
            'receiverApprovedBy',
            'items.item.category'
        ]);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('request_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('request_date', '<=', $request->date_to);
        }

        if ($request->filled('source_project')) {
            $query->where('transferrer_project_id', $request->source_project);
        }

        if ($request->filled('destination_project')) {
            $query->where('receiver_project_id', $request->destination_project);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transfer_request_number', 'like', "%{$search}%")
                  ->orWhere('transfer_reason', 'like', "%{$search}%");
            });
        }

        // Order by date descending and paginate
        $perPage = 20;
        $transfers = $query->orderBy('request_date', 'desc')->paginate($perPage);

        // Append query parameters to pagination links
        $transfers->appends($request->all());

        // Get all projects for filter dropdown
        $projects = Project::orderBy('project_name')->get();

        // Calculate statistics
        $stats = [
            'total_transfers' => MaterialTransferRequest::count(),
            'pending' => MaterialTransferRequest::where('status', 'pending')->count(),
            'approved' => MaterialTransferRequest::where('status', 'approved')->count(),
            'completed' => MaterialTransferRequest::where('status', 'completed')->count(),
            'cancelled' => MaterialTransferRequest::where('status', 'cancelled')->count(),
        ];

        return view('warehouse.reports.project-transfers', compact('transfers', 'projects', 'stats'));
    }

    /**
     * Supplier Direct Delivery Report
     */
    public function directDeliveries(Request $request)
    {
        // Get Direct Deliveries with related data
        $query = DirectDelivery::with([
            'supplier',
            'project',
            'projectDivision',
            'items.item.category'
        ]);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('delivery_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('delivery_date', '<=', $request->date_to);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('delivery_status')) {
            $query->where('status', $request->delivery_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('reference_number', 'like', "%{$search}%")
                  ->orWhere('delivery_number', 'like', "%{$search}%")
                  ->orWhere('purchase_order_number', 'like', "%{$search}%")
                  ->orWhere('lpo_number', 'like', "%{$search}%");
            });
        }

        // Order by date descending and paginate
        $perPage = 20;
        $deliveries = $query->orderBy('delivery_date', 'desc')->paginate($perPage);

        // Append query parameters to pagination links
        $deliveries->appends($request->all());

        // Get suppliers and projects for filter dropdowns
        $suppliers = Supplier::orderBy('name')->get();
        $projects = Project::orderBy('project_name')->get();

        // Calculate statistics
        $stats = [
            'total_deliveries' => DirectDelivery::count(),
            'pending' => DirectDelivery::where('status', 'pending')->count(),
            'delivered' => DirectDelivery::where('status', 'delivered')->count(),
            'partial' => DirectDelivery::where('status', 'partial')->count(),
            'returned' => DirectDelivery::where('status', 'returned')->count(),
            'total_value' => DirectDelivery::sum('total_amount_aed')
        ];

        return view('warehouse.reports.direct-deliveries', compact('deliveries', 'suppliers', 'projects', 'stats'));
    }

    public function stockOnProject(Request $request)
    {
        $query = Project::select('projects.*')
            ->addSelect(DB::raw('(
                SELECT COALESCE(SUM(project_inventory.quantity_available), 0)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
            ) as total_stock'))
            ->addSelect(DB::raw('(
                SELECT COUNT(DISTINCT project_inventory.item_id)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
                AND project_inventory.quantity_available > 0
            ) as unique_items'))
            ->addSelect(DB::raw('(
                SELECT COALESCE(SUM(project_inventory.quantity_available * COALESCE(project_inventory.unit_price, 0)), 0)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
            ) as total_value'))
            ->whereExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('project_inventory')
                    ->whereColumn('project_inventory.project_id', 'projects.id')
                    ->where('project_inventory.quantity_available', '>', 0);
            });

        // Apply filters
        if ($request->filled('project_status')) {
            $query->where('status', $request->project_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('project_name', 'like', "%{$search}%")
                  ->orWhere('project_number', 'like', "%{$search}%");
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereBetween('created_at', [$request->date_from, $request->date_to]);
        }

        // Add project filter if specified
        if ($request->filled('project_id')) {
            $query->where('projects.id', $request->project_id);
        }

        $projects = $query->orderBy('project_name')->get();

        // Get filter options
        $allProjects = Project::orderBy('project_name')->get();
        $projectStatuses = ['active', 'completed', 'on_hold', 'cancelled'];

        return view('warehouse.reports.stock-on-project', compact('projects', 'allProjects', 'projectStatuses', 'request'));
    }

    public function stockOnProjectDetails(Request $request, $projectId)
    {
        $project = Project::findOrFail($projectId);

        $query = Item::with('category')
            ->select('items.*')
            ->addSelect(DB::raw("(
                SELECT project_inventory.quantity_available
                FROM project_inventory
                WHERE project_inventory.project_id = {$projectId}
                AND project_inventory.item_id = items.id
            ) as project_stock"))
            ->addSelect(DB::raw("(
                SELECT COALESCE(project_inventory.unit_price, 0)
                FROM project_inventory
                WHERE project_inventory.project_id = {$projectId}
                AND project_inventory.item_id = items.id
            ) as effective_unit_price"))
            ->addSelect(DB::raw("(
                SELECT COALESCE(SUM(outgoing_fulfillments.quantity_fulfilled), 0)
                FROM outgoing_fulfillments
                INNER JOIN outgoing_transactions ON outgoing_fulfillments.outgoing_transaction_id = outgoing_transactions.id
                WHERE outgoing_transactions.project_id = {$projectId}
                AND outgoing_fulfillments.item_id = items.id
            ) as total_delivered"))
            ->whereExists(function ($query) use ($projectId) {
                $query->select(DB::raw(1))
                    ->from('project_inventory')
                    ->whereColumn('project_inventory.item_id', 'items.id')
                    ->where('project_inventory.project_id', $projectId)
                    ->where('project_inventory.quantity_available', '>', 0);
            });

        // Apply filters
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('item_description', 'like', "%{$search}%")
                  ->orWhere('item_code', 'like', "%{$search}%");
            });
        }

        if ($request->filled('stock_status')) {
            if ($request->stock_status === 'low_stock') {
                $query->whereRaw("(
                    SELECT project_inventory.quantity_available
                    FROM project_inventory
                    WHERE project_inventory.project_id = {$projectId}
                    AND project_inventory.item_id = items.id
                ) <= items.min_stock_level");
            } elseif ($request->stock_status === 'out_of_stock') {
                $query->whereRaw("(
                    SELECT COALESCE(project_inventory.quantity_available, 0)
                    FROM project_inventory
                    WHERE project_inventory.project_id = {$projectId}
                    AND project_inventory.item_id = items.id
                ) = 0");
            }
        }

        $items = $query->orderBy('item_description')->get();

        // Calculate totals
        $totalItems = $items->count();
        $totalStock = $items->sum('project_stock');
        $totalValue = $items->sum(function ($item) {
            return ($item->project_stock ?? 0) * ($item->effective_unit_price ?? 0);
        });

        // Get filter options
        $categories = ItemCategory::orderBy('category_name')->get();

        return view('warehouse.reports.stock-on-project-details', compact(
            'project', 'items', 'totalItems', 'totalStock', 'totalValue', 'categories', 'request'
        ));
    }

    private function determineFulfillmentStatus($outgoingItem)
    {
        if ($outgoingItem->quantity_released <= 0) {
            return 'pending';
        } elseif ($outgoingItem->quantity_balance > 0) {
            return 'partial';
        } else {
            return 'completed';
        }
    }

    /**
     * Show reference allocations page for a specific item in a project
     */
    public function itemReferenceAllocations($projectId, $itemId)
    {
        $project = Project::findOrFail($projectId);
        $item = Item::findOrFail($itemId);

        // Get Material Requests
        $materialRequests = OutgoingTransaction::with(['requester', 'division'])
            ->where('project_id', $projectId)
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_released', '>', 0);
            })
            ->get()
            ->map(function($request) use ($itemId) {
                $outgoingItem = $request->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this material request
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('outgoing_item_id', $outgoingItem->id)
                  ->sum('quantity_delivered');

                // Calculate transferred quantity via Material Transfers
                $transferredQty = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('outgoing_item_id', $outgoingItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $request->material_request_number ?: 'MR-' . str_pad($request->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['INV', 'INVENTORY', 'N/A'])) {
                    $referenceNumber = "MR#{$request->id}: {$referenceNumber}-{$request->id}";
                } else {
                    $referenceNumber = "MR#{$request->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Material Request',
                    'reference_number' => $referenceNumber,
                    'date' => $request->transaction_date->format('Y-m-d'),
                    'requested_by' => $request->requester ? $request->requester->name : 'N/A',
                    'division' => $request->division ? $request->division->division_name : 'N/A',
                    'status' => $outgoingItem->quantity_released == 0 ? 'Pending' :
                                ($outgoingItem->quantity_balance == 0 ? 'Fully Dispatched' : 'Partially Dispatched'),
                    'quantity_released' => $outgoingItem->quantity_released,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $transferredQty,
                    'net_quantity' => $outgoingItem->quantity_released - $returnedQty - $transferredQty,
                    'unit_cost' => $outgoingItem->unit_cost ?? 0,
                    'total_value' => ($outgoingItem->quantity_released - $returnedQty - $transferredQty) * ($outgoingItem->unit_cost ?? 0)
                ];
            });

        // Get Direct Deliveries (including those from multi-source material issue)
        $directDeliveryIds = \App\Models\Warehouse\OutgoingFulfillment::join('outgoing_transactions', 'outgoing_fulfillments.outgoing_transaction_id', '=', 'outgoing_transactions.id')
            ->where('outgoing_fulfillments.fulfillment_type', 'direct_delivery')
            ->where('outgoing_fulfillments.item_id', $itemId)
            ->where('outgoing_transactions.project_id', $projectId)
            ->pluck('outgoing_fulfillments.reference_id');

        $directDeliveries = \App\Models\DirectDelivery::with(['supplier'])
            ->whereIn('id', $directDeliveryIds)
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId);
            })
            ->get()
            ->map(function($delivery) use ($itemId, $projectId) {
                $deliveryItem = $delivery->items->where('item_id', $itemId)->first();

                // Get the actual issued quantity from fulfillment record
                $fulfilledQty = \App\Models\Warehouse\OutgoingFulfillment::join('outgoing_transactions', 'outgoing_fulfillments.outgoing_transaction_id', '=', 'outgoing_transactions.id')
                    ->where('outgoing_fulfillments.fulfillment_type', 'direct_delivery')
                    ->where('outgoing_fulfillments.reference_id', $delivery->id)
                    ->where('outgoing_fulfillments.item_id', $itemId)
                    ->where('outgoing_transactions.project_id', $projectId)
                    ->sum('outgoing_fulfillments.quantity_fulfilled');

                $deliveredQty = $fulfilledQty > 0 ? $fulfilledQty : ($deliveryItem->received_quantity > 0 ? $deliveryItem->received_quantity : $deliveryItem->delivered_quantity);

                // Calculate returned quantity for this direct delivery
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('direct_delivery_item_id', $deliveryItem->id)
                  ->sum('quantity_delivered');

                // Calculate transferred quantity via Material Transfers
                $transferredQty = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('direct_delivery_item_id', $deliveryItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $delivery->delivery_number ?? $delivery->reference_number ?? 'DD-' . str_pad($delivery->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['DD', 'DELIVERY', 'N/A']) || strlen($referenceNumber) <= 3) {
                    $referenceNumber = "DD#{$delivery->id}: {$referenceNumber}-{$delivery->id}";
                } else {
                    $referenceNumber = "DD#{$delivery->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Direct Delivery',
                    'reference_number' => $referenceNumber,
                    'date' => $delivery->delivery_date->format('Y-m-d'),
                    'requested_by' => $delivery->supplier ? $delivery->supplier->name : 'Direct Supply',
                    'division' => 'N/A',
                    'status' => $deliveredQty > 0 ? 'Delivered' : 'Pending',
                    'quantity_released' => $deliveredQty,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $transferredQty,
                    'net_quantity' => $deliveredQty - $returnedQty - $transferredQty,
                    'unit_cost' => $deliveryItem->unit_price ?? 0,
                    'total_value' => ($deliveredQty - $returnedQty - $transferredQty) * ($deliveryItem->unit_price ?? 0)
                ];
            });

        // Get Material Transfers (received by this project)
        $materialTransfers = \App\Models\Warehouse\MaterialTransferRequest::with(['transferrerProject', 'requestedBy'])
            ->where('receiver_project_id', $projectId)
            ->whereIn('status', ['approved', 'partially_approved'])
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_approved', '>', 0);
            })
            ->get()
            ->map(function($transfer) use ($itemId) {
                $transferItem = $transfer->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this material transfer
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('material_transfer_item_id', $transferItem->id)
                  ->sum('quantity_delivered');

                // Calculate subsequent transfers from this material transfer
                $subsequentTransfers = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('material_transfer_item_id', $transferItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $transfer->transfer_request_number ?? 'MT-' . str_pad($transfer->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['MT', 'TRANSFER', 'N/A']) || strlen($referenceNumber) <= 3) {
                    $referenceNumber = "MT#{$transfer->id}: {$referenceNumber}-{$transfer->id}";
                } else {
                    $referenceNumber = "MT#{$transfer->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Material Transfer',
                    'reference_number' => $referenceNumber,
                    'date' => $transfer->request_date->format('Y-m-d'),
                    'requested_by' => $transfer->requestedBy ? $transfer->requestedBy->name : 'N/A',
                    'division' => 'From: ' . ($transfer->transferrerProject ? $transfer->transferrerProject->project_name : 'N/A'),
                    'status' => $transferItem->quantity_approved > 0 ? 'Transferred' : 'Pending',
                    'quantity_released' => $transferItem->quantity_approved,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $subsequentTransfers,
                    'net_quantity' => $transferItem->quantity_approved - $returnedQty - $subsequentTransfers,
                    'unit_cost' => $transferItem->estimated_unit_price ?? 0,
                    'total_value' => ($transferItem->quantity_approved - $returnedQty - $subsequentTransfers) * ($transferItem->estimated_unit_price ?? 0)
                ];
            });

        // Combine all references
        $references = collect()
            ->merge($materialRequests)
            ->merge($directDeliveries)
            ->merge($materialTransfers)
            ->sortByDesc(function ($item) {
                return $item['date'];
            })
            ->values();

        // Calculate totals
        $totals = [
            'total_released' => $references->sum('quantity_released'),
            'total_returned' => $references->sum('quantity_returned'),
            'total_transferred' => $references->sum('quantity_transferred'),
            'total_net' => $references->sum('net_quantity'),
            'total_value' => $references->sum('total_value')
        ];

        return view('warehouse.reports.item-reference-allocations', compact(
            'project', 'item', 'references', 'totals'
        ));
    }

    /**
     * Get division name based on return source type
     */
    private function getDivisionName($returnItem, $sourceInfo)
    {
        if ($returnItem->outgoing_item_id && $sourceInfo) {
            // Material Request return
            return $sourceInfo->transaction?->division?->division_name;
        } elseif ($returnItem->direct_delivery_item_id && $sourceInfo) {
            // Direct Delivery return
            return $sourceInfo->directDelivery?->project?->projectDivision?->division_name ?? 'Direct Delivery';
        } elseif ($returnItem->material_transfer_item_id && $sourceInfo) {
            // Material Transfer return
            return $sourceInfo->transferRequest?->receiverProject?->projectDivision?->division_name ?? 'Material Transfer';
        }

        // Fallback to operation project division if available
        return $returnItem->operation->project?->projectDivision?->division_name;
    }

    /**
     * Get reference number based on return source type
     */
    private function getReferenceNumber($returnItem, $sourceInfo)
    {
        if ($returnItem->outgoing_item_id && $sourceInfo) {
            // Material Request return
            return $sourceInfo->transaction?->material_request_number ?: 'MR-' . str_pad($sourceInfo->transaction?->id, 6, '0', STR_PAD_LEFT);
        } elseif ($returnItem->direct_delivery_item_id && $sourceInfo) {
            // Direct Delivery return
            return $sourceInfo->directDelivery?->delivery_number ?? 'DD-' . $sourceInfo->directDelivery?->id;
        } elseif ($returnItem->material_transfer_item_id && $sourceInfo) {
            // Material Transfer return
            return $sourceInfo->transferRequest?->transfer_request_number ?? 'MT-' . $sourceInfo->transferRequest?->id;
        }

        // Fallback to operation number
        return $returnItem->operation->operation_number ?? 'SR-' . $returnItem->operation->id;
    }

    /**
     * Get reference type based on return source
     */
    private function getReferenceType($returnItem)
    {
        if ($returnItem->outgoing_item_id) {
            return 'Material Request';
        } elseif ($returnItem->direct_delivery_item_id) {
            return 'Direct Delivery';
        } elseif ($returnItem->material_transfer_item_id) {
            return 'Material Transfer';
        }

        return 'Site Return';
    }
}