<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class IncomingOperation extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'operation_number',
        'operation_type',
        'operation_date',
        'status',

        // Reference Information
        'supplier_id',
        'project_id',
        'material_request_id',
        'direct_delivery_id',
        'material_transfer_id',
        'division',

        // LPO Management
        'lpo_number',
        'lpo_status',
        'lpo_budget_amount',
        'currency',
        'exchange_rate',

        // Delivery Information
        'delivery_note_number',

        // Return Control
        'force_close_return',
        'force_close_reason',
        'inspector_id',
        'inspection_datetime',
        'inspection_result',
        'inspection_notes',
        'inspection_checklist',

        // Financial Information
        'total_amount',
        'total_amount_aed',
        'vat_amount',
        'vat_rate',

        // Documents
        'attached_documents',
        'delivery_note_path',
        'photo_documentation',

        // Workflow
        'received_by',
        'received_by_name',
        'approved_by',
        'approved_at',
        'approval_notes',

        // Compliance & Audit
        'regulatory_compliant',
        'compliance_checklist',
        'notes',
        'rejection_reason'
    ];

    protected $casts = [
        'operation_date' => 'date',
        'delivery_datetime' => 'datetime',
        'inspection_datetime' => 'datetime',
        'approved_at' => 'datetime',
        'total_amount' => 'decimal:2',
        'total_amount_aed' => 'decimal:2',
        'vat_amount' => 'decimal:2',
        'vat_rate' => 'decimal:2',
        'lpo_budget_amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'force_close_return' => 'boolean',
        'regulatory_compliant' => 'boolean',
        'gps_coordinates' => 'array',
        'inspection_checklist' => 'array',
        'attached_documents' => 'array',
        'photo_documentation' => 'array',
        'compliance_checklist' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($operation) {
            if (empty($operation->operation_number)) {
                $operation->operation_number = static::generateOperationNumber();
            }
        });

        static::saving(function ($operation) {
            // Auto-calculate VAT and AED amounts
            $operation->vat_amount = $operation->total_amount * ($operation->vat_rate / 100);

            if ($operation->currency !== 'AED') {
                $operation->total_amount_aed = $operation->total_amount * $operation->exchange_rate;
            } else {
                $operation->total_amount_aed = $operation->total_amount;
            }
        });
    }

    /**
     * Generate operation number
     */
    public static function generateOperationNumber()
    {
        $date = now()->format('Ymd');
        $prefix = 'INC-' . $date . '-';

        // Find the highest sequence number for today's date (including soft-deleted)
        $lastOperation = static::withTrashed()
                              ->where('operation_number', 'LIKE', $prefix . '%')
                              ->orderBy('operation_number', 'desc')
                              ->first();

        if ($lastOperation) {
            $lastSequence = (int) substr($lastOperation->operation_number, -4);
            $sequence = $lastSequence + 1;
        } else {
            $sequence = 1;
        }

        return $prefix . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get the supplier
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the direct delivery
     */
    public function directDelivery(): BelongsTo
    {
        return $this->belongsTo(\App\Models\DirectDelivery::class, 'direct_delivery_id');
    }

    /**
     * Get the project
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the division
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the receiver
     */
    public function receiver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Get the inspector
     */
    public function inspector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'inspector_id');
    }

    /**
     * Get the approver
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get all items in this operation
     */
    public function items(): HasMany
    {
        return $this->hasMany(IncomingOperationItem::class);
    }

    /**
     * Get related LPO
     */
    public function lpo(): BelongsTo
    {
        return $this->belongsTo(LpoManagement::class, 'lpo_number', 'lpo_number');
    }

    /**
     * Get the related material request (outgoing transaction) for site returns
     */
    public function materialRequest(): BelongsTo
    {
        return $this->belongsTo(OutgoingTransaction::class, 'material_request_id');
    }

    /**
     * Get the related material transfer for site returns
     */
    public function materialTransfer(): BelongsTo
    {
        return $this->belongsTo(MaterialTransferRequest::class, 'material_transfer_id');
    }

    /**
     * Scope for operation type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('operation_type', $type);
    }

    /**
     * Scope for status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for date range
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('operation_date', [$startDate, $endDate]);
    }

    /**
     * Scope for project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Check if operation requires inspection
     */
    public function shouldForceClose()
    {
        return $this->force_close_return === true;
    }

    /**
     * Check if operation is approved
     */
    public function isApproved()
    {
        return $this->status === 'approved';
    }

    /**
     * Check if operation is completed
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        return match($this->status) {
            'draft' => 'badge-secondary',
            'pending_inspection' => 'badge-warning',
            'approved' => 'badge-success',
            'rejected' => 'badge-danger',
            'completed' => 'badge-primary',
            default => 'badge-secondary'
        };
    }

    /**
     * Get operation type label
     */
    public function getOperationTypeLabelAttribute()
    {
        return match($this->operation_type) {
            'supplier_delivery' => 'Supplier Delivery',
            'site_return' => 'Site Return',
            'inter_division_transfer' => 'Inter-Division Transfer',
            'emergency_procurement' => 'Emergency Procurement',
            default => ucfirst(str_replace('_', ' ', $this->operation_type))
        };
    }

    /**
     * Get formatted total amount
     */
    public function getFormattedTotalAmountAttribute()
    {
        return $this->currency . ' ' . number_format($this->total_amount, 2);
    }

    /**
     * Get formatted total amount in AED
     */
    public function getFormattedTotalAmountAedAttribute()
    {
        return 'AED ' . number_format($this->total_amount_aed, 2);
    }

    /**
     * Complete the operation
     */
    public function complete()
    {
        if ($this->status !== 'approved') {
            throw new \Exception('Operation must be approved before completion');
        }

        $this->status = 'completed';
        $this->save();

        $this->processToInventory();
    }

    /**
     * Process operation items to inventory (can be called directly for auto-processing)
     */
    public function processToInventory()
    {
        // Update inventory for each item
        foreach ($this->items as $item) {
            // For direct processing, set quantity_accepted = quantity_delivered
            if ($item->quantity_accepted == 0) {
                $item->quantity_accepted = $item->quantity_delivered;
                $item->quality_status = 'passed';
                $item->save();
            }

            $this->updateInventory($item);
            $this->updateProjectInventory($item);
            $this->createStockMovement($item);
        }

        // Update LPO if exists
        if ($this->lpo) {
            $this->lpo->updateDeliveryStatus();
        }
    }

    /**
     * Update inventory for an item
     */
    protected function updateInventory($item)
    {
        $inventory = Inventory::firstOrNew([
            'item_id' => $item->item_id,
            'batch_number' => $item->batch_number,
        ]);

        $inventory->quantity_available += $item->quantity_accepted;
        $inventory->unit_price = $item->unit_price_aed;
        $inventory->production_date = $item->production_date ?: $item->manufacturing_date;
        $inventory->expiry_date = $item->expiry_date;
        $inventory->location = 'Main Warehouse'; // Default location
        $inventory->save();
    }

    /**
     * Update project inventory based on operation type
     */
    protected function updateProjectInventory($item)
    {
        // Only update project inventory for certain operation types
        if (!in_array($this->operation_type, ['site_return', 'supplier_delivery'])) {
            return;
        }

        if ($this->operation_type === 'site_return') {
            // For site returns, SUBTRACT from project inventory (materials coming back to warehouse)
            $this->subtractFromProjectInventory($item);

            // Update direct delivery item's returned_quantity if this is a direct delivery return
            if ($item->direct_delivery_item_id) {
                $directDeliveryItem = \App\Models\DirectDeliveryItem::find($item->direct_delivery_item_id);
                if ($directDeliveryItem) {
                    $directDeliveryItem->recordReturn($item->quantity_accepted);
                }
            }

            // Update material transfer item's returned quantity if this is a material transfer return
            if ($item->material_transfer_item_id) {
                // For material transfer returns, the tracking is handled through IncomingOperationItem
                // with the material_transfer_item_id relationship. No additional tracking needed
                // as we can calculate returned quantities by querying IncomingOperationItems
                \Log::info("Material Transfer Return Processed", [
                    'material_transfer_item_id' => $item->material_transfer_item_id,
                    'quantity_returned' => $item->quantity_accepted,
                    'item_id' => $item->item_id,
                    'operation_id' => $this->id
                ]);
            }
        } elseif ($this->operation_type === 'supplier_delivery' && $this->project_id) {
            // For direct supplier deliveries to project, ADD to project inventory
            $this->addToProjectInventory($item);
        }
    }

    /**
     * Add items to project inventory (for direct deliveries)
     */
    protected function addToProjectInventory($item)
    {
        $projectInventory = ProjectInventory::where('project_id', $this->project_id)
            ->where('item_id', $item->item_id)
            ->first();

        if ($projectInventory) {
            // Update existing record
            $projectInventory->updateQuantity($item->quantity_accepted, 'add');
        } else {
            // Create new record
            ProjectInventory::create([
                'project_id' => $this->project_id,
                'item_id' => $item->item_id,
                'quantity_available' => $item->quantity_accepted,
                'allocated_quantity' => 0,
                'unit_price' => $item->unit_price_aed ?: 0,
                'total_value' => $item->quantity_accepted * ($item->unit_price_aed ?: 0)
            ]);
        }
    }

    /**
     * Subtract items from project inventory (for site returns)
     */
    protected function subtractFromProjectInventory($item)
    {
        // Determine the project ID to subtract from
        $targetProjectId = null;

        if ($this->project_id) {
            // Direct delivery return - use the site return's project_id
            $targetProjectId = $this->project_id;
        } elseif ($this->material_request_id && $item->outgoing_item_id) {
            // Material request return - get project from original material request
            $materialRequest = \App\Models\Warehouse\OutgoingTransaction::find($this->material_request_id);
            if ($materialRequest && $materialRequest->project_id) {
                $targetProjectId = $materialRequest->project_id;
            }
        } elseif ($this->material_transfer_id && $item->material_transfer_item_id) {
            // Material transfer return - get project from original material transfer (receiver project)
            $materialTransfer = \App\Models\Warehouse\MaterialTransferRequest::find($this->material_transfer_id);
            if ($materialTransfer && $materialTransfer->receiver_project_id) {
                $targetProjectId = $materialTransfer->receiver_project_id;
            }
        }

        if (!$targetProjectId) {
            return;
        }

        $projectInventory = ProjectInventory::where('project_id', $targetProjectId)
            ->where('item_id', $item->item_id)
            ->first();

        if ($projectInventory) {
            // Subtract the returned quantity
            $projectInventory->updateQuantity($item->quantity_accepted, 'subtract');

            // Remove the record if quantity becomes zero or negative
            if ($projectInventory->quantity_available <= 0) {
                $projectInventory->delete();
            }
        }
    }

    /**
     * Create stock movement record
     */
    protected function createStockMovement($item)
    {
        StockMovement::create([
            'item_id' => $item->item_id,
            'movement_type' => 'in',
            'reference_type' => $this->operation_type,
            'reference_id' => $this->id,
            'quantity_before' => 0, // Should be calculated
            'quantity_moved' => $item->quantity_accepted,
            'quantity_after' => 0, // Should be calculated
            'user_id' => $this->received_by,
            'notes' => "Incoming operation: {$this->operation_number} - Division: {$this->division}"
        ]);
    }
}