<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OutgoingItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'outgoing_transaction_id',
        'item_id',
        'quantity_requested',
        'quantity_released',
        'quantity_balance',
        'quantity_transferred',
        'transfer_status',
        'unit_price',
        'total_price'
    ];

    protected $casts = [
        'quantity_requested' => 'decimal:2',
        'quantity_released' => 'decimal:2',
        'quantity_balance' => 'decimal:2',
        'quantity_transferred' => 'decimal:2',
        'transfer_status' => 'string',
        'unit_price' => 'decimal:2',
        'total_price' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($item) {
            // Auto-calculate balance
            $item->quantity_balance = $item->quantity_requested - $item->quantity_released;

            // Auto-calculate total price
            $item->total_price = $item->quantity_released * $item->unit_price;
        });
    }

    /**
     * Get the transaction that owns the item
     */
    public function transaction(): BelongsTo
    {
        return $this->belongsTo(OutgoingTransaction::class, 'outgoing_transaction_id');
    }

    /**
     * Get the item details
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Check if item is fully fulfilled
     */
    public function getIsFullyFulfilledAttribute()
    {
        return $this->quantity_balance <= 0;
    }

    /**
     * Check if item is partially fulfilled
     */
    public function getIsPartiallyFulfilledAttribute()
    {
        return $this->quantity_released > 0 && $this->quantity_balance > 0;
    }

    /**
     * Get fulfillment percentage
     */
    public function getFulfillmentPercentageAttribute()
    {
        if ($this->quantity_requested <= 0) {
            return 0;
        }

        return round(($this->quantity_released / $this->quantity_requested) * 100, 2);
    }

    /**
     * Release quantity from inventory
     */
    public function releaseQuantity($quantity)
    {
        $this->quantity_released += $quantity;
        $this->save();
    }
}