<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ReturnTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'transaction_date',
        'project_id',
        'division_id',
        'return_number',
        'return_reason',
        'returned_by',
        'received_by',
        'status',
        'notes'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'status' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($transaction) {
            if (!$transaction->return_number) {
                $transaction->return_number = 'RET-' . date('Ymd') . '-' . str_pad(static::count() + 1, 4, '0', STR_PAD_LEFT);
            }
        });
    }

    /**
     * Get the project for the transaction
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the division for the transaction
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the user who returned the materials
     */
    public function returner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'returned_by');
    }

    /**
     * Get the user who received the returned materials
     */
    public function receiver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Get all items in this transaction
     */
    public function items(): HasMany
    {
        return $this->hasMany(ReturnItem::class);
    }

    /**
     * Get stock movements created by this transaction
     */
    public function stockMovements()
    {
        return $this->morphMany(StockMovement::class, 'reference');
    }

    /**
     * Scope for pending returns
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for completed returns
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for returns by project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Complete the return transaction and update inventory
     */
    public function complete()
    {
        $this->status = 'completed';
        $this->save();

        // Process each returned item
        foreach ($this->items as $item) {
            $this->processReturnedItem($item);
            $this->createStockMovement($item);
        }
    }

    /**
     * Process a returned item based on its condition
     */
    protected function processReturnedItem($returnItem)
    {
        switch ($returnItem->action_taken) {
            case 'restocked':
                $this->restockItem($returnItem);
                break;
            case 'discarded':
                // Log as waste/discarded
                break;
            case 'returned_to_supplier':
                // Process supplier return
                break;
        }
    }

    /**
     * Restock a returned item to inventory
     */
    protected function restockItem($returnItem)
    {
        $inventory = Inventory::firstOrNew([
            'item_id' => $returnItem->item_id,
            'division_id' => $this->division_id,
        ]);

        $inventory->quantity_available += $returnItem->quantity_returned;
        $inventory->unit_price = $returnItem->unit_price;
        $inventory->save();
    }

    /**
     * Create stock movement record
     */
    protected function createStockMovement($returnItem)
    {
        $movementType = $returnItem->action_taken === 'restocked' ? 'return' : 'adjustment';

        StockMovement::create([
            'item_id' => $returnItem->item_id,
            'movement_type' => $movementType,
            'reference_type' => 'return',
            'reference_id' => $this->id,
            'quantity_before' => 0, // This should be calculated
            'quantity_moved' => $returnItem->quantity_returned,
            'quantity_after' => 0, // This should be calculated
            'division_id' => $this->division_id,
            'user_id' => $this->received_by,
            'notes' => "Return from project: {$this->project->project_name} - {$returnItem->action_taken}"
        ]);
    }

    /**
     * Get total return value
     */
    public function getTotalReturnValueAttribute()
    {
        return $this->items()->sum('total_value');
    }

    /**
     * Get items grouped by condition
     */
    public function getItemsByConditionAttribute()
    {
        return $this->items()->get()->groupBy('condition');
    }
}