<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Facades\Schema;
use App\Models\SiteSetting;

class SiteSettingsServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Share site settings with all views
        View::composer('*', function ($view) {
            // Check if we're in a context where we can access the database
            if ($this->canAccessDatabase()) {
                try {
                    // Ensure default settings exist
                    SiteSetting::ensureDefaults();
                    
                    // Get all active settings grouped by group
                    $settings = SiteSetting::where('is_active', true)
                        ->orderBy('group')
                        ->orderBy('sort_order')
                        ->get()
                        ->groupBy('group');
                    
                    // Create a flat array for easy access
                    $siteSettings = [];
                    foreach ($settings as $group => $groupSettings) {
                        foreach ($groupSettings as $setting) {
                            $siteSettings[$setting->key] = $setting->value;
                        }
                    }
                    
                    $view->with('siteSettings', $siteSettings);
                    $view->with('settingsByGroup', $settings);
                } catch (\Exception $e) {
                    // Fallback to default values if database is not available
                    $view->with('siteSettings', $this->getDefaultSettings());
                    $view->with('settingsByGroup', collect());
                }
            } else {
                // Fallback to default values
                $view->with('siteSettings', $this->getDefaultSettings());
                $view->with('settingsByGroup', collect());
            }
        });
    }
    
    /**
     * Check if we can access the database
     */
    private function canAccessDatabase(): bool
    {
        try {
            // Check if we're not in console or if we are, we're not in a migration
            if (app()->runningInConsole()) {
                return false;
            }
            
            // Check if the site_settings table exists
            return Schema::hasTable('site_settings');
        } catch (\Exception $e) {
            return false;
        }
    }
    
    /**
     * Get default settings as fallback
     */
    private function getDefaultSettings(): array
    {
        return [
            'site_name' => 'UFG - United Float Glass',
            'site_description' => 'Leading manufacturer of high-quality float glass products',
            'site_logo' => 'assets/images/aufg-logo.png',
            'site_favicon' => 'images/favicon.ico',
            'admin_favicon' => 'images/admin-favicon.ico',
            'contact_email' => 'info@ufg.com',
            'contact_phone' => '+1 (555) 123-4567',
            'contact_address' => '123 Glass Street, Industrial District, City, Country',
            'business_hours' => 'Monday - Friday: 8:00 AM - 6:00 PM\nSaturday: 9:00 AM - 4:00 PM\nSunday: Closed',
            'facebook_url' => 'https://facebook.com/ufg',
            'twitter_url' => 'https://twitter.com/ufg',
            'linkedin_url' => 'https://linkedin.com/company/ufg',
            'instagram_url' => 'https://instagram.com/ufg',
            'whatsapp_number' => '+1234567890',
            'whatsapp_message' => 'Hello! I would like to inquire about your glass products.',
            'whatsapp_enabled' => '1',
            'meta_title' => 'UFG - United Float Glass | Premium Glass Manufacturing',
            'meta_description' => 'Leading manufacturer of high-quality float glass products for architectural and automotive applications. Premium glass solutions with international standards.',
            'meta_keywords' => 'float glass, architectural glass, automotive glass, glass manufacturing, premium glass',
            'footer_text' => '© 2025 UFG - United Float Glass. All rights reserved.',
            'footer_description' => 'Leading manufacturer of premium float glass products, serving architectural and automotive industries worldwide with cutting-edge technology and exceptional quality.',
        ];
    }
}