<?php

namespace App\Repositories\Warehouse;

use App\Repositories\BaseRepository;
use App\Models\Warehouse\StockMovement;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class StockMovementRepository extends BaseRepository
{
    /**
     * Create a new repository instance.
     */
    public function __construct(StockMovement $model)
    {
        $this->model = $model;
    }

    /**
     * Get movements by item.
     */
    public function getByItem(int $itemId, array $filters = [])
    {
        $query = $this->model->where('item_id', $itemId)
                            ->with(['item', 'user']);

        return $this->applyFilters($query, $filters)
                   ->orderBy('created_at', 'desc')
                   ->get();
    }

    /**
     * Get movements by type.
     */
    public function getByType(string $type, array $filters = [])
    {
        $query = $this->model->where('type', $type)
                            ->with(['item', 'user']);

        return $this->applyFilters($query, $filters)
                   ->orderBy('created_at', 'desc')
                   ->get();
    }

    /**
     * Get movements by date range.
     */
    public function getByDateRange(Carbon $startDate, Carbon $endDate, array $filters = [])
    {
        $query = $this->model->whereBetween('created_at', [$startDate, $endDate])
                            ->with(['item', 'user']);

        return $this->applyFilters($query, $filters)
                   ->orderBy('created_at', 'desc')
                   ->get();
    }

    /**
     * Get recent movements.
     */
    public function getRecent(int $limit = 50)
    {
        return $this->model->with(['item', 'user'])
                          ->orderBy('created_at', 'desc')
                          ->limit($limit)
                          ->get();
    }

    /**
     * Get movement summary by type.
     */
    public function getSummaryByType(array $filters = [])
    {
        $query = $this->model->selectRaw('
            type,
            COUNT(*) as total_movements,
            SUM(quantity) as total_quantity
        ');

        $query = $this->applyFilters($query, $filters);

        return $query->groupBy('type')->get();
    }

    /**
     * Get movement statistics for date range.
     */
    public function getStatistics(Carbon $startDate = null, Carbon $endDate = null)
    {
        $query = $this->model;

        if ($startDate && $endDate) {
            $query = $query->whereBetween('created_at', [$startDate, $endDate]);
        } elseif ($startDate) {
            $query = $query->where('created_at', '>=', $startDate);
        } elseif ($endDate) {
            $query = $query->where('created_at', '<=', $endDate);
        }

        return [
            'total_movements' => $query->count(),
            'incoming_movements' => $query->whereIn('type', ['increase', 'transfer_in', 'return'])->count(),
            'outgoing_movements' => $query->whereIn('type', ['decrease', 'transfer_out', 'dispatch'])->count(),
            'adjustment_movements' => $query->where('type', 'adjustment')->count(),
            'reservation_movements' => $query->whereIn('type', ['reserve', 'release'])->count(),
        ];
    }

    /**
     * Apply movement-specific search filters.
     */
    protected function applySearchFilter(Builder $query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('reason', 'LIKE', "%{$search}%")
              ->orWhereHas('item', function ($itemQuery) use ($search) {
                  $itemQuery->where('name', 'LIKE', "%{$search}%")
                           ->orWhere('item_code', 'LIKE', "%{$search}%");
              });
        });
    }

    /**
     * Apply movement-specific filters.
     */
    protected function applyFilters(Builder $query, array $filters)
    {
        foreach ($filters as $key => $value) {
            if ($value === null || $value === '') {
                continue;
            }

            switch ($key) {
                case 'type':
                    $query->where('type', $value);
                    break;
                case 'item_id':
                    $query->where('item_id', $value);
                    break;
                case 'user_id':
                    $query->where('user_id', $value);
                    break;
                case 'movable_type':
                    $query->where('movable_type', $value);
                    break;
                case 'movable_id':
                    $query->where('movable_id', $value);
                    break;
                case 'date_from':
                    $query->where('created_at', '>=', $value);
                    break;
                case 'date_to':
                    $query->where('created_at', '<=', $value);
                    break;
                case 'quantity_min':
                    $query->where('quantity', '>=', $value);
                    break;
                case 'quantity_max':
                    $query->where('quantity', '<=', $value);
                    break;
                case 'search':
                    $this->applySearchFilter($query, $value);
                    break;
            }
        }

        return $query;
    }
}