// Stats Counter Animation - Exact Match to Elementor Behavior
(function() {
    'use strict';

    class StatsCounter {
        constructor() {
            this.counters = [];
            this.observer = null;
            this.animating = new Set();
            this.scrollAnimation = true;
            this.hasTriggered = false;
            
            this.init();
        }

        init() {
            // Wait for DOM to be ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => this.setup());
            } else {
                this.setup();
            }
            
            // Also try after a short delay to ensure all styles are loaded
            setTimeout(() => {
                console.log('Delayed initialization...');
                this.setup();
            }, 500);
        }

        setup() {
            console.log('Setup called, DOM ready state:', document.readyState);
            this.findCounters();
            if (this.counters.length > 0) {
                this.setupIntersectionObserver();
                this.setupScrollAnimation();
                console.log('Setup complete with', this.counters.length, 'counters');
            } else {
                console.warn('No counters found during setup');
            }
        }

        findCounters() {
            const counterElements = document.querySelectorAll('.stat-number[data-target]');
            console.log('Found counter elements:', counterElements.length);
            
            counterElements.forEach(element => {
                const target = parseInt(element.getAttribute('data-target'));
                const duration = parseInt(element.getAttribute('data-duration')) || 4000; // Increased from 2000 to 4000ms
                
                console.log('Counter element:', element, 'Target:', target, 'Duration:', duration);
                
                this.counters.push({
                    element: element,
                    target: target,
                    duration: duration,
                    hasAnimated: false
                });
            });
            
            console.log('Total counters initialized:', this.counters.length);
        }

        setupIntersectionObserver() {
            const options = {
                root: null,
                rootMargin: '-10% 0px -10% 0px',
                threshold: 0.5
            };

            this.observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const counter = this.counters.find(c => c.element === entry.target);
                        if (counter && !counter.hasAnimated && !this.animating.has(counter.element)) {
                            console.log('Counter coming into view, starting animation:', counter.target);
                            this.animateCounter(counter);
                        }
                    } else {
                        // Reset animation when element goes out of view
                        const counter = this.counters.find(c => c.element === entry.target);
                        if (counter) {
                            counter.hasAnimated = false;
                            counter.element.textContent = '0';
                            counter.element.classList.remove('counting');
                            counter.element.style.transform = 'scale(1)';
                            counter.element.style.color = 'white';
                            this.animating.delete(counter.element);
                            console.log('Counter out of view, reset for next time:', counter.target);
                        }
                    }
                });
            }, options);

            // Observe all counter elements
            this.counters.forEach(counter => {
                this.observer.observe(counter.element);
                console.log('Observing counter element:', counter.element, 'Target:', counter.target);
            });
            
            // Setup background image visibility observer
            this.setupBackgroundVisibility();
            
            // Setup stat card fade animations
            this.setupCardFadeAnimations();
        }
        
        setupBackgroundVisibility() {
            const statsSection = document.querySelector('.stats-section');
            if (!statsSection) return;
            
            const bgOptions = {
                root: null,
                rootMargin: '-10% 0px -10% 0px',
                threshold: 0.1
            };
            
            const bgObserver = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('in-view');
                        console.log('Stats section in view - showing background');
                    } else {
                        entry.target.classList.remove('in-view');
                        console.log('Stats section out of view - hiding background');
                    }
                });
            }, bgOptions);
            
            bgObserver.observe(statsSection);
        }
        
        setupCardFadeAnimations() {
            const statCards = document.querySelectorAll('.stat-card');
            console.log('Setting up fade animations for', statCards.length, 'stat cards');
            
            const fadeOptions = {
                root: null,
                rootMargin: '0px 0px -20% 0px',
                threshold: 0.3
            };
            
            const fadeObserver = new IntersectionObserver((entries) => {
                entries.forEach((entry, index) => {
                    if (entry.isIntersecting) {
                        // Stagger the animations with a delay
                        setTimeout(() => {
                            entry.target.classList.add('fade-in');
                            console.log('Fading in stat card:', index);
                        }, index * 150); // 150ms delay between each card
                    } else {
                        // Fade out when scrolling away
                        entry.target.classList.remove('fade-in');
                        console.log('Fading out stat card:', index);
                    }
                });
            }, fadeOptions);
            
            // Observe all stat cards
            statCards.forEach((card, index) => {
                fadeObserver.observe(card);
                console.log('Observing stat card:', index);
            });
        }

        animateCounter(counter) {
            if (this.animating.has(counter.element)) {
                console.log('Already animating:', counter.element);
                return;
            }
            
            console.log('=== STARTING ANIMATION ===');
            console.log('Element:', counter.element);
            console.log('Target:', counter.target);
            console.log('Duration:', counter.duration);
            
            this.animating.add(counter.element);
            counter.hasAnimated = true;

            // Ensure starting value is 0 and make it visible
            counter.element.textContent = '0';
            counter.element.style.color = '#ff0000'; // Start with red to make it obvious

            const startTime = performance.now();
            const startValue = 0;
            const endValue = counter.target;
            const duration = counter.duration;
            let frameCount = 0;

            const animate = (currentTime) => {
                frameCount++;
                const elapsed = currentTime - startTime;
                const progress = Math.min(elapsed / duration, 1);
                
                // Linear progression for testing
                const currentValue = Math.floor(startValue + (endValue - startValue) * progress);
                
                // Format number with commas for large numbers
                const formattedValue = this.formatNumber(currentValue);
                counter.element.textContent = formattedValue;
                
                // Log every 30 frames to avoid spam
                if (frameCount % 30 === 0) {
                    console.log('Animation frame:', frameCount, 'Progress:', Math.round(progress * 100) + '%', 'Value:', formattedValue);
                }
                
                // Add visual effects during counting
                if (progress < 1) {
                    counter.element.classList.add('counting');
                    // Bright colors to make it obvious
                    counter.element.style.color = progress < 0.5 ? '#00ffff' : '#ffff00';
                    counter.element.style.transform = `scale(${1.2})`;
                    counter.element.style.fontWeight = 'bold';
                    requestAnimationFrame(animate);
                } else {
                    // Final value with completion effects
                    counter.element.textContent = this.formatNumber(endValue);
                    counter.element.classList.remove('counting');
                    counter.element.style.transform = 'scale(1)';
                    counter.element.style.color = 'white'; // Return to white
                    counter.element.style.fontWeight = '700';
                    this.animating.delete(counter.element);
                    console.log('=== ANIMATION COMPLETE ===');
                    console.log('Final value:', this.formatNumber(endValue));
                    console.log('Total frames:', frameCount);
                }
            };

            requestAnimationFrame(animate);
        }

        formatNumber(num) {
            // Add commas for large numbers (like 250,000)
            return num.toLocaleString();
        }

        // Reset animation (useful for testing)
        resetAnimation() {
            this.counters.forEach(counter => {
                counter.hasAnimated = false;
                counter.element.textContent = '0';
                counter.element.classList.remove('counting');
            });
            this.animating.clear();
        }

        // Manual trigger for testing
        triggerAnimation() {
            console.log('triggerAnimation called, counters:', this.counters.length);
            this.counters.forEach((counter, index) => {
                console.log(`Checking counter ${index}:`, counter.element, 'hasAnimated:', counter.hasAnimated);
                if (!counter.hasAnimated) {
                    console.log(`Triggering animation for counter ${index}`);
                    this.animateCounter(counter);
                } else {
                    console.log(`Counter ${index} already animated`);
                }
            });
        }

        setupScrollAnimation() {
            if (!this.scrollAnimation) return;
            
            let ticking = false;
            
            const updateParallax = () => {
                const statsSection = document.querySelector('.stats-section');
                const backgroundImage = document.querySelector('.stats-background-image');
                if (!statsSection || !backgroundImage) return;
                
                const rect = statsSection.getBoundingClientRect();
                const scrollY = window.pageYOffset;
                
                // Create fixed background effect by moving image opposite to scroll
                // This simulates background-attachment: fixed
                const parallaxOffset = scrollY * 0.5; // Adjust speed as needed
                backgroundImage.style.transform = `translateY(${parallaxOffset}px)`;
                
                ticking = false;
            };
            
            const requestParallaxUpdate = () => {
                if (!ticking) {
                    requestAnimationFrame(updateParallax);
                    ticking = true;
                }
            };
            
            // Throttled scroll listener
            window.addEventListener('scroll', requestParallaxUpdate, { passive: true });
            window.addEventListener('resize', requestParallaxUpdate, { passive: true });
        }
    }

    // Initialize the stats counter
    const statsCounter = new StatsCounter();

    // Export for external access
    window.StatsCounter = statsCounter;
    
    // Add global test function
    window.testCounters = function() {
        console.log('=== TESTING COUNTERS ===');
        statsCounter.resetAnimation();
        setTimeout(() => {
            statsCounter.triggerAnimation();
        }, 100);
    };

    // Additional event listeners for smooth experience
    window.addEventListener('load', () => {
        // Trigger animation if elements are already in view on page load
        setTimeout(() => {
            console.log('Load event: Manually triggering animation...');
            statsCounter.triggerAnimation();
        }, 1500);
    });
    
    // Also trigger on scroll for debugging
    let scrollTriggered = false;
    window.addEventListener('scroll', () => {
        if (!scrollTriggered) {
            console.log('Scroll triggered animation...');
            statsCounter.triggerAnimation();
            scrollTriggered = true;
        }
    });
    
    // Trigger immediately when in view
    setTimeout(() => {
        console.log('Immediate trigger check...');
        const statsSection = document.querySelector('.stats-section');
        if (statsSection) {
            const rect = statsSection.getBoundingClientRect();
            if (rect.top < window.innerHeight && rect.bottom > 0) {
                console.log('Stats section is in view, triggering animation...');
                statsCounter.triggerAnimation();
            }
        }
    }, 2000);

})();