// Admin Panel Navigation React Component
const { useState, useEffect } = React;

// Get page title based on route
function getPageTitle(route) {
    if (route.includes('dashboard')) return 'Dashboard';
    if (route.includes('applications')) return 'Job Applications';
    if (route.includes('job-positions')) return 'Job Positions';
    return 'Admin Panel';
}

function AdminNavigation({ currentRoute }) {
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [careersMenuOpen, setCareersMenuOpen] = useState(true);

    // Close sidebar on outside click (mobile)
    useEffect(() => {
        const handleClickOutside = (event) => {
            if (sidebarOpen && !event.target.closest('.sidebar') && !event.target.closest('.mobile-menu-btn')) {
                setSidebarOpen(false);
            }
        };

        document.addEventListener('click', handleClickOutside);
        return () => document.removeEventListener('click', handleClickOutside);
    }, [sidebarOpen]);

    const isActiveRoute = (route) => {
        return currentRoute.includes(route);
    };

    return React.createElement('div', { className: 'flex h-screen bg-gray-100' },
        // Sidebar
        React.createElement('div', {
            className: `fixed inset-y-0 left-0 z-50 w-64 bg-white shadow-lg transform transition-transform duration-200 ease-in-out sidebar ${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} lg:translate-x-0`
        },
            // Logo
            React.createElement('div', { className: 'flex items-center justify-center p-6 border-b' },
                React.createElement('img', {
                    src: '/images/aufg-logo.png',
                    alt: 'UFG Logo',
                    className: 'h-12'
                })
            ),
            
            // Navigation
            React.createElement('nav', { className: 'mt-6 px-4' },
                // Dashboard Link
                React.createElement('a', {
                    href: '/admin/dashboard',
                    className: `flex items-center p-3 mb-2 rounded-lg transition-colors ${isActiveRoute('dashboard') ? 'bg-indigo-50 text-indigo-700' : 'text-gray-700 hover:bg-gray-100'}`
                },
                    React.createElement('i', { className: 'fas fa-home w-5 h-5 mr-3' }),
                    'Dashboard'
                ),
                
                // Careers Menu with Dropdown
                React.createElement('div', { className: 'mb-2' },
                    React.createElement('button', {
                        onClick: () => setCareersMenuOpen(!careersMenuOpen),
                        className: `flex items-center justify-between w-full p-3 rounded-lg transition-colors ${isActiveRoute('job-positions') || isActiveRoute('applications') ? 'bg-indigo-50 text-indigo-700 font-medium hover:bg-indigo-100' : 'text-gray-700 hover:bg-gray-100'}`
                    },
                        React.createElement('div', { className: 'flex items-center' },
                            React.createElement('i', { className: 'fas fa-briefcase w-5 h-5 mr-3' }),
                            'Careers'
                        ),
                        React.createElement('i', {
                            className: `fas fa-chevron-down w-4 h-4 transition-transform duration-200 ${careersMenuOpen ? 'rotate-180' : ''}`
                        })
                    ),
                    
                    // Submenu
                    careersMenuOpen && React.createElement('div', {
                        className: 'ml-6 mt-1 space-y-1 transition-all duration-200'
                    },
                        React.createElement('a', {
                            href: '/admin/job-positions',
                            className: `flex items-center p-2 pl-4 rounded-lg transition-colors text-sm border-l-2 border-gray-200 hover:border-indigo-300 ${isActiveRoute('job-positions') ? 'bg-indigo-50 text-indigo-700 font-medium border-indigo-400' : 'text-gray-600 hover:bg-gray-50 hover:text-gray-800'}`
                        },
                            React.createElement('i', { className: 'fas fa-file-text w-4 h-4 mr-2' }),
                            'Job Positions'
                        ),
                        React.createElement('a', {
                            href: '/admin/applications',
                            className: `flex items-center p-2 pl-4 rounded-lg transition-colors text-sm border-l-2 border-gray-200 hover:border-indigo-300 ${isActiveRoute('applications') ? 'bg-indigo-50 text-indigo-700 font-medium border-indigo-400' : 'text-gray-600 hover:bg-gray-50 hover:text-gray-800'}`
                        },
                            React.createElement('i', { className: 'fas fa-users w-4 h-4 mr-2' }),
                            'Applications'
                        )
                    )
                )
            ),
            
            // Logout
            React.createElement('div', { className: 'absolute bottom-4 left-4 right-4' },
                React.createElement('form', {
                    method: 'POST',
                    action: '/admin/logout'
                },
                    React.createElement('input', {
                        type: 'hidden',
                        name: '_token',
                        value: document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }),
                    React.createElement('button', {
                        type: 'submit',
                        className: 'flex items-center w-full p-3 text-red-600 hover:bg-red-50 rounded-lg transition-colors'
                    },
                        React.createElement('i', { className: 'fas fa-sign-out-alt w-5 h-5 mr-3' }),
                        'Logout'
                    )
                )
            )
        ),
        
        // Mobile backdrop
        sidebarOpen && React.createElement('div', {
            className: 'fixed inset-0 z-40 bg-black bg-opacity-50 transition-opacity lg:hidden',
            onClick: () => setSidebarOpen(false)
        }),
        
        // Main content
        React.createElement('div', { className: 'flex-1 flex flex-col' },
            // Header
            React.createElement('header', { className: 'bg-white shadow-sm border-b border-gray-200 p-4 lg:pl-72' },
                React.createElement('div', { className: 'flex items-center justify-between' },
                    React.createElement('div', { className: 'flex items-center' },
                        React.createElement('button', {
                            onClick: () => setSidebarOpen(true),
                            className: 'lg:hidden p-2 rounded-lg text-gray-500 hover:bg-gray-100 mobile-menu-btn'
                        },
                            React.createElement('i', { className: 'fas fa-bars w-6 h-6' })
                        ),
                        React.createElement('h1', { className: 'ml-4 lg:ml-0 text-2xl font-bold text-gray-900' }, getPageTitle(currentRoute))
                    )
                )
            ),
            
            // Main content area - this will be filled by the parent page
            React.createElement('main', { 
                className: 'flex-1 lg:pl-64 p-6 overflow-auto',
                id: 'main-content' 
            })
        )
    );
}

// Initialize React component
window.initAdminNavigation = function(currentRoute) {
    const container = document.getElementById('admin-navigation-root');
    if (container) {
        ReactDOM.render(
            React.createElement(AdminNavigation, { currentRoute: currentRoute }),
            container
        );
    }
};