// News Section Entrance Animations & Hover Effects
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up News section entrance animations...');
    
    // Find the news section elements - only headings for entrance animations
    const newsTitle = document.querySelector('.news-title');
    const latestUpdatesTitle = document.querySelector('.news-updates-title');
    const newsItem1 = document.querySelector('.news-item-1');
    const newsItem2 = document.querySelector('.news-item-2');
    const newsItem3 = document.querySelector('.news-item-3');
    const newsItem4 = document.querySelector('.news-item-4');
    const newsSection = document.querySelector('.news-section');
    
    // Array of elements to animate - only headings
    const animateElements = [
        { element: newsTitle, name: 'News title', className: 'news-animate-in' },
        { element: latestUpdatesTitle, name: 'Latest Updates title', className: 'news-animate-in' }
    ].filter(item => item.element); // Only include found elements
    
    if (animateElements.length === 0) {
        console.log('No news heading elements found for animation');
        return;
    }
    
    console.log(`Found ${animateElements.length} news headings to animate`);
    
    // Create intersection observer for news headings only
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            const element = entry.target;
            const intersectionRatio = entry.intersectionRatio;
            
            console.log(`News heading ${element.dataset.id} intersection ratio:`, intersectionRatio);
            
            if (entry.isIntersecting && intersectionRatio >= 0.3) {
                const elementName = animateElements.find(item => item.element === element)?.name || 'unknown';
                console.log(`Triggering entrance animation for: ${elementName}`);
                
                element.classList.add('news-animate-in');
                
                // Unobserve after animation to prevent re-triggering
                observer.unobserve(element);
            }
        });
    }, {
        threshold: [0, 0.2, 0.3, 0.4, 0.5], // Multiple thresholds for smooth detection
        rootMargin: '0px 0px -10% 0px' // Trigger when element is 10% from bottom of viewport
    });
    
    // Start observing only headings
    animateElements.forEach(({ element, name }) => {
        console.log(`Starting to observe heading: ${name}`);
        observer.observe(element);
        
        // Fallback: trigger animation on scroll if intersection observer fails
        let hasTriggered = false;
        window.addEventListener('scroll', () => {
            if (hasTriggered) return;
            
            const rect = element.getBoundingClientRect();
            const windowHeight = window.innerHeight;
            
            // Trigger when element is 30% visible from bottom of viewport
            if (rect.top <= windowHeight * 0.7 && rect.bottom >= 0) {
                console.log(`Fallback scroll trigger for heading: ${name}`);
                element.classList.add('news-animate-in');
                hasTriggered = true;
            }
        }, { passive: true });
    });
    
    // Enhanced blue tint hover effects for news items - All 4 items
    const newsItems = [newsItem1, newsItem2, newsItem3, newsItem4].filter(Boolean);
    
    console.log(`Found ${newsItems.length} news items for hover effects`);
    
    newsItems.forEach((item, index) => {
        if (!item) return;
        
        // Add smooth enter/leave transitions with blue tint effect
        item.addEventListener('mouseenter', () => {
            console.log(`News item ${index + 1} hovered - activating blue tint`);
            item.style.willChange = 'transform, box-shadow, opacity';
            
            // Enhance the blue tint effect on hover
            const beforeElement = window.getComputedStyle(item, '::before');
            console.log('Blue tint overlay activated');
        });
        
        item.addEventListener('mouseleave', () => {
            console.log(`News item ${index + 1} hover ended - removing blue tint`);
            setTimeout(() => {
                item.style.willChange = 'auto';
            }, 500); // Wait for transition to complete
        });
        
        // Add click interaction feedback with blue tint
        item.addEventListener('click', () => {
            console.log(`News item ${index + 1} clicked with blue tint background`);
            
            // Brief pulse animation on click
            item.style.transform = 'translateY(-6px) scale(0.99)';
            
            setTimeout(() => {
                item.style.transform = '';
            }, 200);
        });
        
        // Add touch support for mobile blue tint effect
        item.addEventListener('touchstart', () => {
            console.log(`News item ${index + 1} touched - mobile blue tint`);
        }, { passive: true });
    });
    
    // Cleanup after heading animations complete
    setTimeout(() => {
        animateElements.forEach(({ element }) => {
            if (element) {
                element.style.willChange = 'auto';
            }
        });
        
        console.log('News heading animations performance cleanup completed');
    }, 2000);
    
    // Smooth scroll behavior for news section
    if (newsSection) {
        newsSection.style.scrollBehavior = 'smooth';
    }
    
    console.log('News section entrance animations initialized successfully');
});