@extends('layouts.admin-simple')

@section('title', 'Stock Adjustments')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item">Inventory</li>
    <li class="breadcrumb-item active">Stock Adjustments</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Stock Adjustments</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Track and manage all inventory adjustments</p>
        </div>
        @can('warehouse.inventory.adjust')
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="exportAdjustments()" class="btn btn-sm btn-info mr-2" title="Export adjustments to Excel file">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>

            <button type="button" onclick="showNewAdjustmentModal()" class="btn btn-sm btn-primary" title="Create new stock adjustment">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                New Adjustment
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Today's Adjustments"
                :value="$statistics['todays_adjustments'] ?? 0"
                icon="trending_up"
                color="success"
                subtitle="Adjustments made today"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="This Month"
                :value="$statistics['monthly_adjustments'] ?? 0"
                icon="history"
                color="info"
                subtitle="Total monthly adjustments"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Negative Adjustments"
                :value="$statistics['negative_adjustments'] ?? 0"
                icon="trending_down"
                color="warning"
                subtitle="Stock reductions this month"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Value Impact"
                :value="$statistics['value_impact'] ?? 0"
                icon="monetization_on"
                color="primary"
                subtitle="Total value affected"
                :currency="true"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Date Range</label>
                    <div class="input-group">
                        <input type="date" class="form-control" id="filter-start-date" value="{{ date('Y-m-d', strtotime('-7 days')) }}">
                        <div class="input-group-append input-group-prepend">
                            <span class="input-group-text">to</span>
                        </div>
                        <input type="date" class="form-control" id="filter-end-date" value="{{ date('Y-m-d') }}">
                    </div>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Adjustment Type</label>
                    <select id="filter-type" class="form-control">
                        <option value="">All Types</option>
                        <option value="increase">Increase</option>
                        <option value="decrease">Decrease</option>
                        <option value="set">Set Value</option>
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Location</label>
                    <select id="filter-location" class="form-control">
                        <option value="">All Locations</option>
                        <option value="Warehouse-A">Warehouse-A</option>
                        <option value="Warehouse-B">Warehouse-B</option>
                        <option value="Warehouse-C">Warehouse-C</option>
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">User</label>
                    <select id="filter-user" class="form-control">
                        <option value="">All Users</option>
                        <option value="Demo User">Demo User</option>
                        <option value="System">System</option>
                    </select>
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Adjustments Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Stock Adjustment History
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="adjustments-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Date & Time</th>
                            <th>Item</th>
                            <th>Type</th>
                            <th>Old Qty</th>
                            <th>Adjustment</th>
                            <th>New Qty</th>
                            <th>Reason</th>
                            <th>User</th>
                            <th>Location</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated by DataTables -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- New Adjustment Modal -->
    <div class="modal fade" id="newAdjustmentModal" tabindex="-1" role="dialog" aria-labelledby="newAdjustmentModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="newAdjustmentModalLabel">
                        <i class="material-icons mr-2">add_box</i>
                        Create Stock Adjustment
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="adjustmentForm">
                        @csrf
                        <div class="row">
                            <div class="col-md-12">
                                @php
                                    $modalItemOptions = [['value' => '', 'text' => 'Select Item']];
                                    foreach(\App\Models\Warehouse\Item::orderBy('item_description')->limit(10)->get() as $item) {
                                        $modalItemOptions[] = ['value' => $item->id, 'text' => $item->item_description . ' (' . $item->item_code . ')'];
                                    }
                                @endphp
                                <x-warehouse.select2-dropdown
                                    name="item_id"
                                    id="item_id"
                                    label="Item"
                                    placeholder="Select Item"
                                    :options="$modalItemOptions"
                                    required="true"
                                />
                            </div>
                        </div>

                        <!-- Hidden field for location since there's only one warehouse -->
                        <input type="hidden" name="location" value="Main Warehouse">

                        <div class="row">
                            <div class="col-md-6">
                                <x-warehouse.select2-dropdown
                                    name="type"
                                    id="type"
                                    label="Adjustment Type"
                                    placeholder="Select Type"
                                    :options="[
                                        ['value' => '', 'text' => 'Select Type'],
                                        ['value' => 'increase', 'text' => 'Increase Stock'],
                                        ['value' => 'decrease', 'text' => 'Decrease Stock'],
                                        ['value' => 'set', 'text' => 'Set Stock Level']
                                    ]"
                                    required="true"
                                    onchange="updateQuantityLabel(this.value)"
                                />
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="quantity">
                                        <span id="quantityLabel">Quantity</span>
                                        <span class="text-danger">*</span>
                                    </label>
                                    <input type="number" class="form-control" name="quantity" id="quantity" min="0" required>
                                    <small class="form-text text-muted" id="quantityHelp">Enter the adjustment amount</small>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="reason">Reason <span class="text-danger">*</span></label>
                            <textarea class="form-control" name="reason" id="reason" rows="3" placeholder="Explain the reason for this adjustment..." required></textarea>
                            <div class="invalid-feedback"></div>
                        </div>

                        <div class="alert alert-info">
                            <i class="material-icons mr-2">info</i>
                            <strong>Note:</strong> Stock adjustments will create a movement record and update the inventory levels immediately.
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="saveAdjustment()" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                        Create Adjustment
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- View Details Modal -->
    <div class="modal fade" id="viewDetailsModal" tabindex="-1" role="dialog" aria-labelledby="viewDetailsModalLabel" aria-hidden="true">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="viewDetailsModalLabel">
                        <i class="material-icons mr-2">visibility</i>
                        Adjustment Details
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body" id="adjustmentDetails">
                    <!-- Details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .adjustment-type-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            font-size: 0.75rem;
            font-weight: 500;
            line-height: 1;
            border-radius: 0.375rem;
        }

        .adjustment-increase {
            color: #155724;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
        }

        .adjustment-decrease {
            color: #721c24;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
        }

        .adjustment-set {
            color: #004085;
            background-color: #cce7ff;
            border: 1px solid #b3d7ff;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        /* Modal z-index fixes - highest priority */
        .modal,
        #newAdjustmentModal,
        #viewDetailsModal {
            z-index: 999999999 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        .modal-backdrop,
        .modal-backdrop.fade,
        .modal-backdrop.show {
            z-index: 999999998 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        /* Hide all potential overlapping elements */
        body.modal-open .navbar,
        body.modal-open .main-header,
        body.modal-open .sidebar,
        body.modal-open .main-sidebar,
        body.modal-open .content-wrapper,
        body.modal-open .overlay,
        body.modal-open .page-overlay,
        body.modal-open .loading-overlay,
        body.modal-open .nav,
        body.modal-open .navigation,
        body.modal-open .header,
        body.modal-open .footer,
        body.modal-open .breadcrumb,
        body.modal-open .topbar,
        body.modal-open .app-header {
            z-index: 1 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
            width: 100% !important;
            max-width: 95% !important;
            margin: 1.75rem auto !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
            width: 100% !important;
        }

        /* Override any fixed width settings */
        .modal-dialog,
        .modal-content,
        #newAdjustmentModal .modal-dialog,
        #newAdjustmentModal .modal-content,
        #viewDetailsModal .modal-dialog,
        #viewDetailsModal .modal-content {
            width: 100% !important;
            max-width: 95% !important;
        }

        /* Responsive modal sizing */
        @media (min-width: 576px) {
            .modal-dialog {
                max-width: 500px !important;
            }
        }

        @media (min-width: 992px) {
            .modal-dialog {
                max-width: 800px !important;
            }
        }

        /* Force modal to appear above everything */
        .modal.show,
        .modal.fade.show {
            display: block !important;
            z-index: 999999 !important;
            opacity: 1 !important;
        }

        /* Override any potential conflicts */
        .modal * {
            position: relative;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        /* Fix Select2 dropdown z-index in modals */
        .modal .select2-container--default .select2-dropdown,
        .modal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        .modal .select2-container--default.select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
        }

        .modal .select2-container {
            z-index: 1000000000 !important;
            position: relative !important;
        }

        .modal .select2-container .select2-selection {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Specific fixes for adjustment modals */
        #newAdjustmentModal .select2-dropdown,
        #newAdjustmentModal .select2-container--open .select2-dropdown,
        #viewDetailsModal .select2-dropdown,
        #viewDetailsModal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        /* Ensure Select2 dropdowns appear above modal backdrop */
        .select2-dropdown {
            z-index: 1000000002 !important;
        }

        /* Override Select2's default z-index when in modal context */
        .modal-open .select2-dropdown {
            z-index: 1000000002 !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        .modal-open .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        /* Force hide any overlay elements that might interfere */
        body.modal-open *[class*="overlay"],
        body.modal-open *[id*="overlay"],
        body.modal-open .fixed-top:not(.modal):not(.modal-backdrop),
        body.modal-open .fixed-bottom:not(.modal):not(.modal-backdrop),
        body.modal-open .sticky-top:not(.modal):not(.modal-backdrop) {
            display: none !important;
        }

        /* Ensure modal is always on top */
        .modal {
            z-index: 999999999 !important;
        }

        .modal-backdrop {
            z-index: 999999998 !important;
        }

        /* Enhanced modal Select2 styles */
        .modal-select2-dropdown {
            z-index: 9999999999 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
        }

        .modal-select2-container {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Force correct positioning for Select2 in modals */
        #newAdjustmentModal .select2-dropdown {
            z-index: 9999999999 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        #newAdjustmentModal .select2-container--open .select2-dropdown {
            z-index: 9999999999 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        /* Prevent overflow issues */
        #newAdjustmentModal .modal-body {
            overflow: visible !important;
        }

        /* Select2 Modal Positioning Fixes */
        .select2-container--default .select2-dropdown {
            z-index: 9999999999 !important;
        }

        .select2-dropdown {
            z-index: 9999999999 !important;
        }

        .select2-container {
            z-index: 9999999998 !important;
        }

        /* Additional positioning fixes for modal Select2 */
        body .select2-container--default.select2-container--open .select2-dropdown--below {
            z-index: 9999999999 !important;
        }

        body .select2-container--default.select2-container--open .select2-dropdown--above {
            z-index: 9999999999 !important;
        }

        /* Ensure modal content doesn't interfere */
        .modal-content {
            overflow: visible !important;
        }

        .modal-body {
            overflow: visible !important;
        }

        /* UAE Dirham Symbol */
        .dirham-symbol-inline {
            position: relative;
        }

        .dirham-symbol-inline::before {
            content: '';
            display: inline-block;
            width: 16px;
            height: 16px;
            margin-right: 4px;
            background-image: url("data:image/svg+xml,%3Csvg viewBox='0 0 1000 870' width='16' height='16' style='fill: currentColor;'%3E%3Cpath d='m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.7 0.6 19.7 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z'/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-size: contain;
            background-position: center;
            vertical-align: middle;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let adjustmentsTable;

        $(document).ready(function() {
            initializeDataTable();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                adjustmentsTable.page.len($(this).val()).draw();
            });

            // Prevent dropdown events from interfering with modals
            $(document).on('click', function(e) {
                // If a modal is open, prevent dropdown events from interfering
                if ($('.modal.show').length > 0) {
                    // Allow clicks inside modals
                    if ($(e.target).closest('.modal').length > 0) {
                        return;
                    }
                    // Prevent other dropdown/navigation events
                    e.stopPropagation();
                }
            });

            // Stop dropdown events when modal is shown
            $('.modal').on('shown.bs.modal', function() {
                try {
                    $(document).off('click.bs.dropdown.data-api');
                } catch(e) {
                    console.warn('Error disabling dropdown events:', e);
                }
            });

            // Restore dropdown events when modal is hidden
            $('.modal').on('hidden.bs.modal', function() {
                try {
                    // Use proper jQuery selector without causing guid errors
                    if (typeof $().dropdown === 'function') {
                        $('[data-toggle="dropdown"]').dropdown();
                    }
                } catch(e) {
                    console.warn('Error restoring dropdown events:', e);
                }
            });
        });

        function initializeDataTable() {
            adjustmentsTable = $('#adjustments-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.inventory.adjustments") }}',
                    type: 'GET',
                    data: function(d) {
                        d.start_date = $('#filter-start-date').val();
                        d.end_date = $('#filter-end-date').val();
                        d.type = $('#filter-type').val();
                        d.location = $('#filter-location').val();
                        d.user = $('#filter-user').val();
                        return d;
                    },
                    headers: {
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content'),
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    error: function(xhr, error, code) {
                        console.error('DataTable Ajax Error:', error, code);
                        console.error('Response Status:', xhr.status);
                        console.error('Response Text:', xhr.responseText);
                        showAlert('Error loading adjustment data. Please refresh the page.', 'error');
                    }
                },
                columns: [
                    {
                        data: 'date',
                        name: 'date',
                        render: function(data) {
                            const date = new Date(data);
                            return `
                                <div>
                                    <div class="font-weight-medium">${date.toLocaleDateString()}</div>
                                    <small class="text-muted">${date.toLocaleTimeString()}</small>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'item',
                        render: function(data) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-secondary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <span class="font-weight-bold" style="font-size: 14px;">${data.item_name.charAt(0).toUpperCase()}</span>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${data.item_name}</div>
                                        <small class="text-muted">${data.item_code}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'type',
                        name: 'type',
                        render: function(data) {
                            const typeClasses = {
                                'Increase': 'adjustment-increase',
                                'Decrease': 'adjustment-decrease',
                                'Set': 'adjustment-set'
                            };
                            return `<span class="adjustment-type-badge ${typeClasses[data] || 'adjustment-set'}">${data}</span>`;
                        }
                    },
                    {
                        data: 'old_quantity',
                        name: 'old_quantity',
                        className: 'text-right font-weight-medium'
                    },
                    {
                        data: 'adjustment',
                        name: 'adjustment',
                        className: 'text-right font-weight-bold',
                        render: function(data) {
                            // Ensure data is a string
                            const adjustmentStr = String(data || '0');

                            if (adjustmentStr.startsWith('+')) {
                                return `<span class="text-success">${adjustmentStr}</span>`;
                            } else if (adjustmentStr.startsWith('-')) {
                                return `<span class="text-danger">${adjustmentStr}</span>`;
                            } else {
                                return `<span class="text-primary">→ ${adjustmentStr}</span>`;
                            }
                        }
                    },
                    {
                        data: 'new_quantity',
                        name: 'new_quantity',
                        className: 'text-right font-weight-medium'
                    },
                    {
                        data: 'reason',
                        name: 'reason',
                        render: function(data) {
                            // Ensure data is a string
                            const reasonStr = String(data || '');
                            return reasonStr.length > 30 ?
                                `<span title="${reasonStr}">${reasonStr.substring(0, 30)}...</span>` :
                                reasonStr;
                        }
                    },
                    {
                        data: 'user',
                        name: 'user',
                        render: function(data) {
                            return `
                                <div>
                                    <i class="material-icons text-muted mr-1" style="font-size: 14px; vertical-align: middle;">person</i>
                                    <span>${data}</span>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'location',
                        name: 'location',
                        render: function(data) {
                            return `
                                <div>
                                    <i class="material-icons text-muted mr-1" style="font-size: 14px; vertical-align: middle;">location_on</i>
                                    <span>${data}</span>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        render: function(data) {
                            let actions = `
                                <div class="btn-group" role="group">
                                    <button onclick="viewDetails(${data.id})" class="btn btn-sm btn-outline-primary" title="View Details">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </button>`;

                            @can('warehouse.inventory.adjust')
                            actions += `
                                    <button onclick="deleteAdjustment(${data.id})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>`;
                            @endcan

                            actions += `</div>`;
                            return actions;
                        }
                    }
                ],
                order: [[0, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No stock adjustments found',
                    zeroRecords: 'No matching adjustments found'
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                }
            });
        }

        function applyFilters() {
            adjustmentsTable.draw();
        }

        function clearFilters() {
            $('#filter-start-date, #filter-end-date, #filter-type, #filter-location, #filter-user').val('');
            adjustmentsTable.draw();
        }

        function showNewAdjustmentModal() {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Clear form and validation errors
            $('#adjustmentForm')[0].reset();
            clearValidationErrors();

            // Force very high z-index before showing
            $('#newAdjustmentModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#newAdjustmentModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Fix accessibility and aggressively force z-index after modal is shown
            $('#newAdjustmentModal').on('shown.bs.modal', function() {
                // Fix aria-hidden accessibility issue aggressively
                const $modal = $(this);
                $modal.removeAttr('aria-hidden');
                $modal.attr('aria-modal', 'true');

                // Monitor and prevent aria-hidden from being set again
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'attributes' && mutation.attributeName === 'aria-hidden') {
                            if ($modal.attr('aria-hidden') === 'true') {
                                $modal.removeAttr('aria-hidden');
                                $modal.attr('aria-modal', 'true');
                                console.log('🔧 Fixed aria-hidden violation on adjustment modal');
                            }
                        }
                    });
                });

                observer.observe($modal[0], {
                    attributes: true,
                    attributeFilter: ['aria-hidden']
                });

                // Store observer for cleanup
                $modal.data('aria-observer', observer);
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative',
                    'width': '100%',
                    'max-width': '95%'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative',
                    'width': '100%'
                });

                // Load items dropdown dynamically
                loadItemsDropdown();

                // Initialize Select2 dropdowns with proper dropdownParent
                $('#item_id, #type').each(function() {
                    const $select = $(this);
                    const currentValue = $select.val();

                    // Destroy any existing Select2
                    if ($select.data('select2')) {
                        $select.select2('destroy');
                    }

                    // Initialize with modal-specific settings
                    $select.select2({
                        placeholder: $select.find('option:first').text() || 'Please select...',
                        allowClear: true,
                        theme: 'default',
                        width: '100%',
                        dropdownParent: $('body'), // Attach to body for better positioning
                        dropdownCssClass: 'modal-select2-dropdown',
                        containerCssClass: 'modal-select2-container'
                    });

                    // Restore value
                    $select.val(currentValue).trigger('change');

                    // Keep dropdown visible in modal with enhanced positioning
                    $select.on('select2:open', function() {
                        setTimeout(function() {
                            // Force maximum z-index values for all Select2 elements
                            $('.select2-dropdown').css('z-index', 9999999999);
                            $('.select2-container--default .select2-dropdown').css('z-index', 9999999999);
                            $('.modal-select2-dropdown').css('z-index', 9999999999);
                            $('.select2-results').css('z-index', 9999999999);
                            $('.select2-results__options').css('z-index', 9999999999);

                            // Additional positioning for dropdown container
                            $('.select2-container--open .select2-dropdown').css({
                                'z-index': '9999999999',
                                'position': 'fixed',
                                'display': 'block !important'
                            });

                            console.log('✅ Enhanced z-index and positioning applied to Select2 dropdown elements');
                        }, 10);
                    });

                    // Add change event for type dropdown to update quantity label
                    if ($select.attr('id') === 'type') {
                        $select.on('change', function() {
                            updateQuantityLabel($(this).val());
                        });
                    }
                });

                // Move modal to body if it's not already there
                if (!$('#newAdjustmentModal').parent().is('body')) {
                    $('#newAdjustmentModal').appendTo('body');
                }
            });

            // Cleanup observer when modal is hidden
            $('#newAdjustmentModal').on('hidden.bs.modal', function() {
                const observer = $(this).data('aria-observer');
                if (observer) {
                    observer.disconnect();
                    $(this).removeData('aria-observer');
                }
            });
        }

        function loadItemsDropdown() {
            $.ajax({
                url: '{{ route("warehouse.items.dropdown") }}',
                type: 'GET',
                success: function(response) {
                    if (response.success && response.data) {
                        const dropdown = $('#item_id');

                        // Destroy Select2 if it exists
                        if (dropdown.hasClass('select2-hidden-accessible')) {
                            dropdown.select2('destroy');
                        }

                        dropdown.empty();
                        dropdown.append('<option value="">Select Item</option>');

                        response.data.forEach(function(item) {
                            dropdown.append(`<option value="${item.id}" data-code="${item.item_code}">${item.item_description} (${item.item_code})</option>`);
                        });

                        // Reinitialize Select2 with proper positioning
                        if ($('#newAdjustmentModal').hasClass('show')) {
                            dropdown.select2({
                                placeholder: 'Select Item',
                                allowClear: true,
                                theme: 'default',
                                width: '100%',
                                dropdownParent: $('body'), // Attach to body instead of modal
                                dropdownCssClass: 'modal-select2-dropdown',
                                containerCssClass: 'modal-select2-container',
                                dropdownAutoWidth: false
                            });

                            // Force high z-index for dropdown elements
                            setTimeout(function() {
                                $('.modal-select2-dropdown').css('z-index', 9999999999);
                                $('.select2-container--default .select2-dropdown').css('z-index', 9999999999);
                                $('.select2-dropdown').css('z-index', 9999999999);
                            }, 100);
                        }

                        console.log('✅ Items loaded successfully');
                    }
                },
                error: function(xhr) {
                    console.error('Failed to load items:', xhr);
                    // Fallback to existing static items if API fails
                    const dropdown = $('#item_id');
                    if (dropdown.find('option').length <= 1) {
                        // Add some sample items as fallback
                        @foreach(\App\Models\Warehouse\Item::orderBy('item_description')->limit(10)->get() as $item)
                            dropdown.append(`<option value="{{ $item->id }}">{{ $item->item_description }} ({{ $item->item_code }})</option>`);
                        @endforeach
                    }
                }
            });
        }

        function saveAdjustment() {
            const form = $('#adjustmentForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveAdjustment()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.inventory.store-adjustment") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#newAdjustmentModal').modal('hide');
                        adjustmentsTable.draw();
                        showAlert('Stock adjustment created successfully!', 'success');

                        // Reset form
                        $('#adjustmentForm')[0].reset();
                        clearValidationErrors();
                    } else {
                        showAlert(response.message || 'Error creating adjustment', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        const message = xhr.responseJSON?.message || 'Error creating adjustment. Please try again.';
                        showAlert(message, 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function updateQuantityLabel(type) {
            const label = document.getElementById('quantityLabel');
            const help = document.getElementById('quantityHelp');

            switch(type) {
                case 'increase':
                    label.textContent = 'Increase Amount';
                    help.textContent = 'Enter the amount to add to current stock';
                    break;
                case 'decrease':
                    label.textContent = 'Decrease Amount';
                    help.textContent = 'Enter the amount to subtract from current stock';
                    break;
                case 'set':
                    label.textContent = 'New Stock Level';
                    help.textContent = 'Enter the new stock level to set';
                    break;
                default:
                    label.textContent = 'Quantity';
                    help.textContent = 'Enter the adjustment amount';
            }
        }

        function viewDetails(adjustmentId) {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Force very high z-index before showing
            $('#viewDetailsModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Load adjustment details via AJAX
            const detailsUrl = `{{ url('warehouse/inventory/adjustments') }}/${adjustmentId}/details`;
            console.log('🔍 Fetching adjustment details from:', detailsUrl);

            $.ajax({
                url: detailsUrl,
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.success) {
                        const data = response.data;
                        const detailsHtml = `
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Adjustment ID:</div>
                                <div class="col-sm-8">#ADJ-${adjustmentId.toString().padStart(5, '0')}</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Item:</div>
                                <div class="col-sm-8">${data.item_name} (${data.item_code})</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Previous Stock:</div>
                                <div class="col-sm-8">${data.quantity_before} units</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Adjustment:</div>
                                <div class="col-sm-8 ${data.adjustment_amount >= 0 ? 'text-success' : 'text-danger'}">${data.adjustment_display}</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">New Stock:</div>
                                <div class="col-sm-8">${data.quantity_after} units</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Date & Time:</div>
                                <div class="col-sm-8">${data.created_at}</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">User:</div>
                                <div class="col-sm-8">${data.user_name}</div>
                            </div>
                            <hr>
                            <div class="row mb-3">
                                <div class="col-sm-4 font-weight-medium">Reason:</div>
                                <div class="col-sm-8">${data.reason}</div>
                            </div>
                        `;

                        document.getElementById('adjustmentDetails').innerHTML = detailsHtml;
                    } else {
                        document.getElementById('adjustmentDetails').innerHTML = '<div class="alert alert-danger">Failed to load adjustment details.</div>';
                    }
                },
                error: function(xhr) {
                    console.error('❌ Failed to load adjustment details:', {
                        status: xhr.status,
                        statusText: xhr.statusText,
                        response: xhr.responseText,
                        url: detailsUrl
                    });

                    let errorMessage = 'Error loading adjustment details. Please try again.';
                    if (xhr.status === 404) {
                        errorMessage = 'Adjustment record not found.';
                    } else if (xhr.status === 403) {
                        errorMessage = 'You do not have permission to view this adjustment.';
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }

                    document.getElementById('adjustmentDetails').innerHTML = `<div class="alert alert-danger">${errorMessage}</div>`;
                }
            });

            // Show modal
            $('#viewDetailsModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Fix accessibility and aggressively force z-index after modal is shown
            $('#viewDetailsModal').on('shown.bs.modal', function() {
                // Fix aria-hidden accessibility issue aggressively
                const $modal = $(this);
                $modal.removeAttr('aria-hidden');
                $modal.attr('aria-modal', 'true');

                // Monitor and prevent aria-hidden from being set again
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'attributes' && mutation.attributeName === 'aria-hidden') {
                            if ($modal.attr('aria-hidden') === 'true') {
                                $modal.removeAttr('aria-hidden');
                                $modal.attr('aria-modal', 'true');
                                console.log('🔧 Fixed aria-hidden violation on view details modal');
                            }
                        }
                    });
                });

                observer.observe($modal[0], {
                    attributes: true,
                    attributeFilter: ['aria-hidden']
                });

                // Store observer for cleanup
                $modal.data('aria-observer', observer);
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative',
                    'width': '100%',
                    'max-width': '95%'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative',
                    'width': '100%'
                });

                // Move modal to body if it's not already there
                if (!$('#viewDetailsModal').parent().is('body')) {
                    $('#viewDetailsModal').appendTo('body');
                }
            });

            // Cleanup observer when modal is hidden
            $('#viewDetailsModal').on('hidden.bs.modal', function() {
                const observer = $(this).data('aria-observer');
                if (observer) {
                    observer.disconnect();
                    $(this).removeData('aria-observer');
                }
            });
        }

        function deleteAdjustment(adjustmentId) {
            if (confirm('Are you sure you want to delete this adjustment record? This action cannot be undone.')) {
                // Simulate deletion
                showAlert('Adjustment deleted successfully!', 'success');
                adjustmentsTable.draw();
            }
        }

        function exportAdjustments() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                showAlert('Export functionality is not yet implemented', 'info');
            }, 800);
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : (type === 'info' ? 'alert-info' : 'alert-danger');
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }
    </script>
@endpush