@extends('layouts.admin-simple')

@section('title', 'Inventory Valuation')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item">Inventory</li>
    <li class="breadcrumb-item active">Valuation</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Inventory Valuation</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Comprehensive inventory valuation and cost analysis</p>
        </div>
        @can('warehouse.inventory.view')
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="refreshValuation()" class="btn btn-sm btn-success" title="Refresh valuation data">
                <i class="material-icons mr-1" style="font-size: 16px;">refresh</i>
                Refresh
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Stock Value"
                :value="number_format($statistics['total_stock_value'], 2)"
                icon="attach_money"
                color="primary"
                subtitle="Current inventory worth"
                :currency="true"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Items"
                :value="number_format($statistics['total_items'])"
                icon="inventory"
                color="success"
                subtitle="Items with value"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Average Unit Value"
                :value="number_format($statistics['average_unit_value'], 2)"
                icon="trending_up"
                color="info"
                subtitle="Per unit average"
                :currency="true"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Categories"
                :value="number_format($statistics['categories_count'])"
                icon="category"
                color="warning"
                subtitle="Item categories valued"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Valuation Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Category</label>
                    <select id="filter-category" class="form-control">
                        <option value="">All Categories</option>
                        @foreach($categoryBreakdown->values() as $category)
                            <option value="{{ $category['category'] }}">{{ $category['category'] }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Value Range</label>
                    <select id="filter-value-range" class="form-control">
                        <option value="">All Values</option>
                        <option value="high">High Value (>10,000)</option>
                        <option value="medium">Medium Value (1,000-10,000)</option>
                        <option value="low">Low Value (<1,000)</option>
                        <option value="zero">Zero Value</option>
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Item Search</label>
                    <input type="text" id="filter-search" class="form-control" placeholder="Search by item name or code">
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Cost Method</label>
                    <select id="filter-cost-method" class="form-control">
                        <option value="">All Methods</option>
                        <option value="FIFO">FIFO</option>
                        <option value="LIFO">LIFO</option>
                        <option value="AVERAGE">Average Cost</option>
                        <option value="STANDARD">Standard Cost</option>
                    </select>
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Category Breakdown Card -->
    <div class="row mb-4">
        <div class="col-lg-12">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0 d-flex align-items-center">
                        <i class="material-icons mr-2">pie_chart</i>
                        Valuation by Category
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        @foreach($categoryBreakdown->take(6) as $category)
                        <div class="col-lg-2 col-md-4 mb-3">
                            <div class="text-center">
                                <div class="h5 mb-1 text-primary"><span class="dirham-symbol-inline">{{ number_format($category['total_value'], 0) }}</span></div>
                                <div class="small text-muted">{{ $category['category'] }}</div>
                                <div class="small text-success">{{ $category['item_count'] }} items</div>
                            </div>
                        </div>
                        @endforeach
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Valuation Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Inventory Valuation List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="valuation-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Item</th>
                            <th>Category</th>
                            <th>Stock Quantity</th>
                            <th>Unit Price</th>
                            <th>Total Value</th>
                            <th>Cost Method</th>
                            <th>Last Updated</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($valuationData as $item)
                            <tr data-item-id="{{ $item['id'] }}" class="valuation-row">
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="mr-3">
                                            <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                                <span class="font-weight-bold" style="font-size: 14px;">{{ substr($item['item_name'], 0, 1) }}</span>
                                            </div>
                                        </div>
                                        <div>
                                            <div class="font-weight-medium">{{ $item['item_name'] }}</div>
                                            <small class="text-muted">{{ $item['item_code'] }}</small>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div>
                                        <i class="material-icons text-muted mr-1" style="font-size: 14px; vertical-align: middle;">category</i>
                                        <span>{{ $item['category'] }}</span>
                                    </div>
                                </td>
                                <td>
                                    <div>
                                        <div class="font-weight-medium mb-1">{{ number_format($item['current_stock'], 2) }}</div>
                                        <small class="text-muted">{{ $item['unit'] }}</small>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-right">
                                        <div class="font-weight-medium"><span class="dirham-symbol-inline">{{ number_format($item['unit_price'], 2) }}</span></div>
                                        <small class="text-muted">per {{ $item['unit'] }}</small>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-right">
                                        @php
                                            $value = $item['total_value'];
                                            $badgeClass = $value > 10000 ? 'badge-success' : ($value > 1000 ? 'badge-info' : 'badge-secondary');
                                        @endphp
                                        <span class="badge {{ $badgeClass }}"><span class="dirham-symbol-inline">{{ number_format($value, 2) }}</span></span>
                                    </div>
                                </td>
                                <td>
                                    <span class="badge badge-outline-primary">{{ $item['cost_method'] }}</span>
                                </td>
                                <td>
                                    <div>
                                        <div class="small">{{ $item['last_updated']->format('M j, Y') }}</div>
                                        <div class="small text-muted">{{ $item['last_updated']->format('H:i') }}</div>
                                    </div>
                                </td>
                                <td>
                                    <button type="button"
                                            class="btn btn-sm btn-outline-primary view-details-btn"
                                            data-item-id="{{ $item['id'] }}"
                                            title="View Details">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </button>
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                    <tfoot>
                        <tr class="table-info">
                            <th colspan="4" class="text-right">Total Valuation:</th>
                            <th class="text-right">
                                <strong><span class="dirham-symbol-inline">{{ number_format($statistics['total_stock_value'], 2) }}</span></strong>
                            </th>
                            <th colspan="3"></th>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>


    <!-- Item Details Modal -->
    <div class="modal fade" id="itemDetailsModal" tabindex="-1" role="dialog" aria-labelledby="itemDetailsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="itemDetailsModalLabel">
                        <i class="material-icons mr-2">info</i>
                        Item Valuation Details
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body" id="itemDetailsContent">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .valuation-row:hover {
            background-color: #f8f9fa;
        }

        .valuation-row.high-value {
            background-color: #e8f5e8;
        }

        .valuation-row.zero-value {
            background-color: #fff3cd;
        }

        .badge-outline-primary {
            color: #007bff;
            border: 1px solid #007bff;
            background-color: transparent;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        /* Modal z-index fix - use same approach as suppliers page */
        .modal,
        #itemDetailsModal {
            z-index: 999999 !important;
        }

        .modal-backdrop {
            z-index: 999998 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        .category-card {
            border-left: 4px solid #007bff;
            transition: all 0.15s ease-in-out;
        }

        .category-card:hover {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }

        /* UAE Dirham Symbol */
        .dirham-symbol-inline {
            position: relative;
        }

        .dirham-symbol-inline::before {
            content: '';
            display: inline-block;
            width: 16px;
            height: 16px;
            margin-right: 4px;
            background-image: url("data:image/svg+xml,%3Csvg viewBox='0 0 1000 870' width='16' height='16' style='fill: currentColor;'%3E%3Cpath d='m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.7 0.6 19.7 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z'/%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-size: contain;
            background-position: center;
            vertical-align: middle;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let valuationTable;

        $(document).ready(function() {
            initializeDataTable();
            initializeEventHandlers();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                valuationTable.page.len($(this).val()).draw();
            });
        });

        function initializeDataTable() {
            valuationTable = $('#valuation-table').DataTable({
                processing: false,
                serverSide: false,
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                order: [[4, 'desc']], // Sort by total value descending
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No inventory items found for valuation',
                    zeroRecords: 'No matching items found'
                },
                columnDefs: [
                    { orderable: false, targets: [7] }, // Actions column
                    { searchable: false, targets: [7] }, // Actions column
                    { type: 'num', targets: [2, 3, 4] }, // Numeric columns
                    { className: 'text-right', targets: [3, 4] } // Right-align price columns
                ],
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                    initializeEventHandlers(); // Re-initialize handlers for new DOM elements
                },
                footerCallback: function(row, data, start, end, display) {
                    // Calculate total for displayed rows
                    const api = this.api();
                    const totalValue = api
                        .column(4, { page: 'current' })
                        .data()
                        .reduce(function(a, b) {
                            const value = parseFloat(b.replace(/[^\d.-]/g, '')) || 0;
                            return a + value;
                        }, 0);

                    // Update footer
                    $(api.column(4).footer()).html('<strong><span class="dirham-symbol-inline">' + totalValue.toLocaleString('en-US', {minimumFractionDigits: 2}) + '</span></strong>');
                }
            });
        }

        function initializeEventHandlers() {
            // View details button
            $('.view-details-btn').off('click').on('click', function() {
                const itemId = $(this).data('item-id');
                showItemDetails(itemId);
            });
        }

        function applyFilters() {
            const category = $('#filter-category').val();
            const valueRange = $('#filter-value-range').val();
            const search = $('#filter-search').val();
            const costMethod = $('#filter-cost-method').val();

            // Apply DataTable search
            valuationTable.search(search);

            // Apply custom filters
            $.fn.dataTable.ext.search.push(function(settings, data, dataIndex) {
                if (settings.nTable.id !== 'valuation-table') {
                    return true;
                }

                // Category filter
                if (category && data[1].indexOf(category) === -1) {
                    return false;
                }

                // Value range filter
                if (valueRange) {
                    const valueText = data[4];
                    const value = parseFloat(valueText.replace(/[^\d.-]/g, '')) || 0;

                    switch (valueRange) {
                        case 'high':
                            if (value <= 10000) return false;
                            break;
                        case 'medium':
                            if (value <= 1000 || value > 10000) return false;
                            break;
                        case 'low':
                            if (value >= 1000) return false;
                            break;
                        case 'zero':
                            if (value > 0) return false;
                            break;
                    }
                }

                // Cost method filter
                if (costMethod && data[5].indexOf(costMethod) === -1) {
                    return false;
                }

                return true;
            });

            valuationTable.draw();
        }

        function clearFilters() {
            $('#filter-category, #filter-value-range, #filter-search, #filter-cost-method').val('');

            // Clear custom filters
            $.fn.dataTable.ext.search.pop();

            valuationTable.search('').draw();
        }

        function refreshValuation() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Refreshing...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                location.reload();
            }, 1500);
        }


        function showItemDetails(itemId) {
            console.log('Opening item details for ID:', itemId);

            // Clear any existing modal content
            $('#itemDetailsContent').empty();

            // Set loading content
            $('#itemDetailsContent').html(`
                <div class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="sr-only">Loading...</span>
                    </div>
                    <p class="mt-2">Loading item details...</p>
                </div>
            `);

            // Force very high z-index before showing (same approach as suppliers page)
            $('#itemDetailsModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#itemDetailsModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown (same as suppliers page)
            $('#itemDetailsModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css('z-index', 999998);
            });

            // Simulate loading details
            setTimeout(() => {
                $('#itemDetailsContent').html(`
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="text-primary">Basic Information</h6>
                            <table class="table table-sm">
                                <tr><td><strong>Item Name:</strong></td><td>Sample Item ${itemId}</td></tr>
                                <tr><td><strong>Item Code:</strong></td><td>ITM-${String(itemId).padStart(4, '0')}</td></tr>
                                <tr><td><strong>Category:</strong></td><td>Sample Category</td></tr>
                                <tr><td><strong>Unit:</strong></td><td>pcs</td></tr>
                            </table>
                        </div>
                        <div class="col-md-6">
                            <h6 class="text-primary">Valuation Details</h6>
                            <table class="table table-sm">
                                <tr><td><strong>Current Stock:</strong></td><td>150.00 pcs</td></tr>
                                <tr><td><strong>Unit Price:</strong></td><td><span class="dirham-symbol-inline">45.50</span></td></tr>
                                <tr><td><strong>Total Value:</strong></td><td><span class="dirham-symbol-inline">6,825.00</span></td></tr>
                                <tr><td><strong>Cost Method:</strong></td><td>FIFO</td></tr>
                            </table>
                        </div>
                    </div>
                    <div class="row mt-3">
                        <div class="col-12">
                            <h6 class="text-primary">Recent Transactions</h6>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Type</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td>${new Date().toLocaleDateString()}</td>
                                            <td><span class="badge badge-success">IN</span></td>
                                            <td>50.00</td>
                                            <td><span class="dirham-symbol-inline">45.50</span></td>
                                            <td><span class="dirham-symbol-inline">2,275.00</span></td>
                                        </tr>
                                        <tr>
                                            <td>${new Date(Date.now() - 86400000).toLocaleDateString()}</td>
                                            <td><span class="badge badge-danger">OUT</span></td>
                                            <td>25.00</td>
                                            <td><span class="dirham-symbol-inline">45.50</span></td>
                                            <td><span class="dirham-symbol-inline">1,137.50</span></td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                `);
            }, 1000);
        }


        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : type === 'info' ? 'alert-info' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }
    </script>
@endpush