@extends('layouts.admin-simple')

@section('title', 'Items')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Items</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Items</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage inventory items and their details</p>
        </div>
        @can('warehouse.items.create')
        <div class="page-actions-right d-flex align-items-center">
            <button onclick="showImportModal()" class="btn btn-sm btn-info mr-2" title="Import items">
                <i class="material-icons mr-1" style="font-size: 16px;">file_upload</i>
                Import
            </button>
            <button onclick="showAddItemModal()" class="btn btn-sm btn-primary" title="Add new item">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                Add Item
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Filters Row -->
    <div class="card mb-3">
        <div class="card-body py-2">
            <div class="row align-items-center">
                <div class="col-md-3">
                    <div style="font-size: 12px; margin-bottom: 8px;">
                        @php
                            $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();
                            $categoryOptions = [['value' => '', 'text' => 'All Categories']];
                            foreach($categories as $category) {
                                $categoryOptions[] = ['value' => $category->id, 'text' => $category->category_name];
                            }
                        @endphp
                        <x-warehouse.select2-dropdown
                            name="filter-category"
                            id="filter-category"
                            label="Category"
                            placeholder="All Categories"
                            :options="$categoryOptions"
                            class="form-control-sm"
                        />
                    </div>
                </div>
                <div class="col-md-3">
                    <div style="font-size: 12px; margin-bottom: 8px;">
                        <x-warehouse.select2-dropdown
                            name="filter-status"
                            id="filter-status"
                            label="Status"
                            placeholder="All Statuses"
                            :options="[
                                ['value' => '', 'text' => 'All Statuses'],
                                ['value' => 'active', 'text' => 'Active'],
                                ['value' => 'inactive', 'text' => 'Inactive'],
                                ['value' => 'discontinued', 'text' => 'Discontinued']
                            ]"
                            class="form-control-sm"
                        />
                    </div>
                </div>
                <div class="col-md-6 text-right">
                    <button onclick="applyFilters()" class="btn btn-sm btn-secondary mr-2">
                        <i class="material-icons mr-1" style="font-size: 14px;">filter_list</i>Apply Filters
                    </button>
                    <button onclick="clearFilters()" class="btn btn-sm btn-outline-secondary">
                        <i class="material-icons mr-1" style="font-size: 14px;">clear</i>Clear
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Items Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">inventory_2</i>
                Items List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="items-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Item Info</th>
                            <th>Category</th>
                            <th>Unit</th>
                            <th>Status</th>
                            <th>Created Date</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated by DataTables -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

<!-- Add Item Modal -->
<div class="modal fade" id="addItemModal" tabindex="-1" role="dialog" aria-labelledby="addItemModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addItemModalLabel">
                    <i class="material-icons mr-2">inventory_2</i>
                    Add New Item
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="addItemForm">
                    @csrf
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="item_code">Item Code</label>
                                <div class="input-group">
                                    <input type="text" name="item_code" id="item_code" class="form-control" placeholder="Auto-generated if empty">
                                    <div class="input-group-append">
                                        <button type="button" onclick="generateItemCode()" class="btn btn-outline-secondary" title="Generate Code">
                                            <i class="material-icons" style="font-size: 16px;">refresh</i>
                                        </button>
                                    </div>
                                </div>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            @php
                                $modalCategoryOptions = [['value' => '', 'text' => 'Select Category']];
                                foreach($categories as $category) {
                                    $modalCategoryOptions[] = ['value' => $category->id, 'text' => $category->category_name];
                                }
                            @endphp
                            <x-warehouse.select2-dropdown
                                name="category_id"
                                id="category_id"
                                label="Category"
                                placeholder="Select Category"
                                :options="$modalCategoryOptions"
                                required="true"
                            />
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="item_description">Item Description <span class="text-danger">*</span></label>
                                <input type="text" name="item_description" id="item_description" class="form-control" required>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <x-warehouse.select2-dropdown
                                name="unit_id"
                                id="unit_id"
                                label="Unit of Measure"
                                placeholder="Loading units..."
                                :options="[]"
                                required="true"
                            />
                        </div>
                        <div class="col-md-6">
                            <x-warehouse.select2-dropdown
                                name="status"
                                id="status"
                                label="Status"
                                placeholder="Select Status"
                                :options="[
                                    ['value' => 'active', 'text' => 'Active'],
                                    ['value' => 'inactive', 'text' => 'Inactive'],
                                    ['value' => 'discontinued', 'text' => 'Discontinued']
                                ]"
                                selected="active"
                                required="true"
                            />
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="min_stock_level">Min Stock Level</label>
                                <input type="number" name="min_stock_level" id="min_stock_level" class="form-control" min="0" step="0.01">
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="max_stock_level">Max Stock Level</label>
                                <input type="number" name="max_stock_level" id="max_stock_level" class="form-control" min="0" step="0.01">
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="reorder_point">Reorder Point</label>
                                <input type="number" name="reorder_point" id="reorder_point" class="form-control" min="0" step="0.01">
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="notes">Notes</label>
                                <textarea name="notes" id="notes" class="form-control" rows="3" placeholder="Optional notes about this item"></textarea>
                                <div class="invalid-feedback"></div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" onclick="saveItem()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                    Save Item
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Import Items Modal -->
<div class="modal fade" id="importModal" tabindex="-1" role="dialog" aria-labelledby="importModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="importModalLabel">
                    <i class="material-icons mr-2">file_upload</i>
                    Import Items
                </h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <!-- Step 1: Download Template -->
                <div class="card mb-3">
                    <div class="card-body py-3">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0">Step 1: Download Template</h6>
                            <button type="button" onclick="downloadTemplate()" class="btn btn-sm btn-outline-info">
                                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                                Download Template
                            </button>
                        </div>
                        <small class="text-muted">Download the Excel template with the correct column format.</small>
                    </div>
                </div>

                <hr>

                <!-- Step 2: Upload File -->
                <form id="importForm" enctype="multipart/form-data">
                    @csrf
                    <div class="form-group">
                        <h6 class="mb-3">Step 2: Upload Your File</h6>
                        <div class="custom-file">
                            <input type="file" name="file" class="custom-file-input" id="importFile" accept=".xlsx,.xls,.csv" required>
                            <label class="custom-file-label" for="importFile">Choose file...</label>
                        </div>
                        <small class="form-text text-muted mt-2">
                            <i class="material-icons" style="font-size: 14px;">info</i>
                            Supported formats: Excel (.xlsx, .xls) or CSV files. Maximum size: 10MB.
                        </small>
                    </div>

                    <!-- Progress Bar -->
                    <div class="progress mb-3" id="uploadProgress" style="display: none;">
                        <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                    </div>

                    <!-- Import Results -->
                    <div id="importResults" style="display: none;">
                        <div class="alert alert-info">
                            <h6><i class="material-icons mr-1">assessment</i> Import Results</h6>
                            <div id="importSummary"></div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" onclick="importItems()" class="btn btn-primary" id="importBtn">
                    <i class="material-icons mr-1" style="font-size: 16px;">cloud_upload</i>
                    Import Items
                </button>
            </div>
        </div>
    </div>
</div>

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-active {
            background-color: #d4edda;
            color: #155724;
        }

        .status-inactive {
            background-color: #f8d7da;
            color: #721c24;
        }

        .status-discontinued {
            background-color: #e2e3e5;
            color: #383d41;
        }

        /* Modal z-index fix - highest priority */
        .modal,
        #addItemModal,
        #importModal {
            z-index: 999999999 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        .modal-backdrop,
        .modal-backdrop.fade,
        .modal-backdrop.show {
            z-index: 999999998 !important;
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
        }

        /* Hide all potential overlapping elements */
        body.modal-open .navbar,
        body.modal-open .main-header,
        body.modal-open .sidebar,
        body.modal-open .main-sidebar,
        body.modal-open .content-wrapper,
        body.modal-open .overlay,
        body.modal-open .page-overlay,
        body.modal-open .loading-overlay,
        body.modal-open .nav,
        body.modal-open .navigation,
        body.modal-open .header,
        body.modal-open .footer,
        body.modal-open .breadcrumb,
        body.modal-open .topbar,
        body.modal-open .app-header {
            z-index: 1 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Force modal to appear above everything */
        .modal.show,
        .modal.fade.show {
            display: block !important;
            z-index: 999999 !important;
            opacity: 1 !important;
        }

        /* Override any potential conflicts */
        .modal * {
            position: relative;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        /* Fix Select2 dropdown z-index in modals */
        .modal .select2-container--default .select2-dropdown,
        .modal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        .modal .select2-container--default.select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
        }

        .modal .select2-container {
            z-index: 1000000000 !important;
            position: relative !important;
        }

        .modal .select2-container .select2-selection {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Specific fixes for import modal */
        #importModal .select2-dropdown,
        #importModal .select2-container--open .select2-dropdown,
        #addItemModal .select2-dropdown,
        #addItemModal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
        }

        /* Ensure Select2 dropdowns appear above modal backdrop */
        .select2-dropdown {
            z-index: 1000000002 !important;
        }

        /* Override Select2's default z-index when in modal context */
        .modal-open .select2-dropdown {
            z-index: 1000000002 !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        .modal-open .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        /* Force hide any overlay elements that might interfere */
        body.modal-open *[class*="overlay"],
        body.modal-open *[id*="overlay"],
        body.modal-open .fixed-top:not(.modal):not(.modal-backdrop),
        body.modal-open .fixed-bottom:not(.modal):not(.modal-backdrop),
        body.modal-open .sticky-top:not(.modal):not(.modal-backdrop) {
            display: none !important;
        }

        /* Ensure modal is always on top */
        .modal {
            z-index: 999999999 !important;
        }

        .modal-backdrop {
            z-index: 999999998 !important;
        }

        /* Enhanced modal Select2 styles */
        .modal-select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
        }

        .modal-select2-container {
            z-index: 1000000001 !important;
            position: relative !important;
        }

        /* Force correct positioning for Select2 in modals */
        #addItemModal .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        #addItemModal .select2-container--open .select2-dropdown {
            z-index: 1000000002 !important;
            position: absolute !important;
            display: block !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        /* Prevent overflow issues */
        #addItemModal .modal-body {
            overflow: visible !important;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        /* Select2 Modal Positioning Fixes */
        .modal-select2-dropdown {
            z-index: 9999999999 !important;
        }

        .select2-container--default .select2-dropdown {
            z-index: 9999999999 !important;
        }

        .select2-dropdown {
            z-index: 9999999999 !important;
        }

        .modal-select2-container {
            z-index: 9999999998 !important;
        }

        .select2-container {
            z-index: 9999999998 !important;
        }

        /* Additional positioning fixes for modal Select2 */
        body .select2-container--default.select2-container--open .select2-dropdown--below {
            z-index: 9999999999 !important;
        }

        body .select2-container--default.select2-container--open .select2-dropdown--above {
            z-index: 9999999999 !important;
        }

        /* Ensure modal content doesn't interfere */
        .modal-content {
            overflow: visible !important;
        }

        .modal-body {
            overflow: visible !important;
        }

    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let itemsTable;

        $(document).ready(function() {
            initializeDataTable();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                if (itemsTable) {
                    itemsTable.page.len($(this).val()).draw();
                }
            });
        });

        function initializeDataTable() {
            itemsTable = $('#items-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.items.index") }}',
                    type: 'GET'
                },
                columns: [
                    {
                        data: 'item_description',
                        name: 'item_description',
                        render: function(data, type, row) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-info text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <i class="material-icons" style="font-size: 20px;">inventory_2</i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${row.item_description}</div>
                                        <small class="text-muted">${row.item_code}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'category_name',
                        name: 'category_name',
                        render: function(data) {
                            return `<span class="badge badge-secondary">${data || 'No Category'}</span>`;
                        }
                    },
                    {
                        data: 'unit_of_measure',
                        name: 'unit_of_measure',
                        render: function(data) {
                            return `<span class="badge badge-light">${data || 'N/A'}</span>`;
                        }
                    },
                    {
                        data: 'status',
                        name: 'status',
                        render: function(data) {
                            const statusClasses = {
                                'active': 'status-active',
                                'inactive': 'status-inactive',
                                'discontinued': 'status-discontinued'
                            };
                            const status = data || 'inactive';
                            return `<span class="status-badge ${statusClasses[status] || 'status-inactive'}">${status.charAt(0).toUpperCase() + status.slice(1)}</span>`;
                        }
                    },
                    {
                        data: null,
                        name: 'created_at',
                        render: function(data, type, row) {
                            return `
                                <div>
                                    <div class="small">${row.created_at}</div>
                                    <div class="small text-muted">by ${row.created_by}</div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        render: function(data, type, row) {
                            let actions = `
                                <div class="btn-group" role="group">
                                    <a href="{{ url('warehouse/items') }}/${row.id}" class="btn btn-sm btn-outline-primary" title="View">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>`;

                            @can('warehouse.items.edit')
                            actions += `
                                    <a href="{{ url('warehouse/items') }}/${row.id}/edit" class="btn btn-sm btn-outline-success" title="Edit">
                                        <i class="material-icons" style="font-size: 16px;">edit</i>
                                    </a>`;
                            @endcan

                            @can('warehouse.items.delete')
                            actions += `
                                    <button onclick="deleteItem(${row.id})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>`;
                            @endcan

                            actions += `</div>`;
                            return actions;
                        }
                    }
                ],
                order: [[4, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No items found',
                    zeroRecords: 'No matching items found'
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                }
            });
        }

        function applyFilters() {
            itemsTable.draw();
        }

        function clearFilters() {
            $('#filter-status, #filter-category').val('');
            itemsTable.draw();
        }

        function deleteItem(id) {
            if (confirm('Are you sure you want to delete this item?')) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: `{{ url('warehouse/items') }}/${id}`,
                    type: 'DELETE',
                    success: function(response) {
                        if (response.success) {
                            itemsTable.draw();
                            showAlert('Item deleted successfully', 'success');
                        } else {
                            showAlert(response.message, 'error');
                        }
                    },
                    error: function(xhr) {
                        showAlert('Error deleting item', 'error');
                    }
                });
            }
        }

        function showAddItemModal() {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Clear form and validation errors
            $('#addItemForm')[0].reset();
            clearValidationErrors();

            // Force very high z-index before showing
            $('#addItemModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#addItemModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Fix accessibility and aggressively force z-index after modal is shown
            $('#addItemModal').on('shown.bs.modal', function() {
                // Fix aria-hidden accessibility issue aggressively
                const $modal = $(this);
                $modal.removeAttr('aria-hidden');
                $modal.attr('aria-modal', 'true');

                // Monitor and prevent aria-hidden from being set again
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'attributes' && mutation.attributeName === 'aria-hidden') {
                            if ($modal.attr('aria-hidden') === 'true') {
                                $modal.removeAttr('aria-hidden');
                                $modal.attr('aria-modal', 'true');
                                console.log('🔧 Fixed aria-hidden violation on add item modal');
                            }
                        }
                    });
                });

                observer.observe($modal[0], {
                    attributes: true,
                    attributeFilter: ['aria-hidden']
                });

                // Store observer for cleanup
                $modal.data('aria-observer', observer);
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Reinitialize Select2 dropdowns with proper dropdownParent
                $('#addItemModal .select2-dropdown').each(function() {
                    $(this).css('z-index', 1000000002);
                });

                // Load units dropdown
                loadUnitsDropdown();

                // Enhanced Select2 initialization with better positioning
                $('#category_id, #unit_id, #status').each(function() {
                    const $select = $(this);
                    const currentValue = $select.val();

                    // Destroy any existing Select2
                    if ($select.data('select2')) {
                        $select.select2('destroy');
                    }

                    // Initialize with modal-specific settings
                    $select.select2({
                        placeholder: $select.data('placeholder') || 'Please select...',
                        allowClear: true,
                        theme: 'default',
                        width: '100%',
                        dropdownParent: $('body'), // Attach to body for better positioning
                        dropdownCssClass: 'modal-select2-dropdown',
                        containerCssClass: 'modal-select2-container'
                    });

                    // Restore value
                    $select.val(currentValue).trigger('change');

                    // Keep dropdown visible in modal with enhanced positioning
                    $select.on('select2:open', function() {
                        setTimeout(function() {
                            // Force maximum z-index values for all Select2 elements
                            $('.select2-dropdown').css('z-index', 9999999999);
                            $('.select2-container--default .select2-dropdown').css('z-index', 9999999999);
                            $('.modal-select2-dropdown').css('z-index', 9999999999);
                            $('.select2-results').css('z-index', 9999999999);
                            $('.select2-results__options').css('z-index', 9999999999);

                            // Additional positioning for dropdown container
                            $('.select2-container--open .select2-dropdown').css({
                                'z-index': '9999999999',
                                'position': 'fixed',
                                'display': 'block !important'
                            });

                            console.log('✅ Enhanced z-index and positioning applied to Select2 dropdown elements');
                        }, 10);
                    });
                });

                // Move modal to body if it's not already there
                if (!$('#addItemModal').parent().is('body')) {
                    $('#addItemModal').appendTo('body');
                }
            });

            // Cleanup observer when modal is hidden
            $('#addItemModal').on('hidden.bs.modal', function() {
                const observer = $(this).data('aria-observer');
                if (observer) {
                    observer.disconnect();
                    $(this).removeData('aria-observer');
                }
            });
        }

        function loadUnitsDropdown() {
            $.ajax({
                url: '{{ route("warehouse.items.units-dropdown") }}',
                type: 'GET',
                success: function(response) {
                    if (response.success && response.data) {
                        const dropdown = $('#unit_id');

                        // Destroy Select2 if it exists
                        if (dropdown.hasClass('select2-hidden-accessible')) {
                            dropdown.select2('destroy');
                        }

                        dropdown.empty();
                        dropdown.append('<option value="">Select Unit</option>');

                        response.data.forEach(function(unit) {
                            dropdown.append(`<option value="${unit.value}" data-unit-symbol="${unit.unit_symbol}" data-unit-type="${unit.unit_type}">${unit.text}</option>`);
                        });

                        // Reinitialize Select2 with proper positioning
                        if ($('#addItemModal').hasClass('show')) {
                            dropdown.select2({
                                placeholder: 'Select Unit',
                                allowClear: true,
                                theme: 'default',
                                width: '100%',
                                dropdownParent: $('body'), // Attach to body instead of modal
                                dropdownCssClass: 'modal-select2-dropdown',
                                containerCssClass: 'modal-select2-container',
                                dropdownAutoWidth: false
                            });

                            // Force high z-index for dropdown elements
                            setTimeout(function() {
                                $('.modal-select2-dropdown').css('z-index', 9999999999);
                                $('.select2-container--default .select2-dropdown').css('z-index', 9999999999);
                                $('.select2-dropdown').css('z-index', 9999999999);
                            }, 100);
                        }

                        console.log('✅ Units loaded successfully');
                    }
                },
                error: function(xhr) {
                    console.error('Failed to load units:', xhr);
                    showAlert('Failed to load units for dropdown', 'error');
                }
            });
        }

        function generateItemCode() {
            // Simple item code generation based on category and timestamp
            const categorySelect = $('#category_id');
            const selectedCategory = categorySelect.find(':selected').text();
            const categoryPrefix = selectedCategory.substring(0, 3).toUpperCase();
            const timestamp = Date.now().toString().slice(-6);
            const generatedCode = `${categoryPrefix}${timestamp}`;
            $('#item_code').val(generatedCode);
        }

        function saveItem() {
            const form = $('#addItemForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveItem()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.items.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#addItemModal').modal('hide');
                        itemsTable.draw();
                        showAlert('Item added successfully!', 'success');
                    } else {
                        showAlert(response.message || 'Error adding item', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        showAlert('Error adding item. Please try again.', 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }


        // Import functionality
        function showImportModal() {
            // Stop any propagating events that might interfere
            if (event) {
                event.preventDefault();
                event.stopPropagation();
                event.stopImmediatePropagation();
            }

            // Clear form and results
            $('#importForm')[0].reset();
            $('#importResults').hide();
            $('#uploadProgress').hide();
            $('.custom-file-label').text('Choose file...');

            // Force modal positioning and z-index
            $('#importModal').css({
                'z-index': 99999999999,
                'position': 'fixed',
                'top': '0',
                'left': '0',
                'width': '100%',
                'height': '100%',
                'display': 'flex',
                'align-items': 'flex-start',
                'justify-content': 'center'
            });

            // Add body class and hide overlapping elements
            $('body').addClass('modal-open').css('overflow', 'hidden');

            // Hide all potential overlapping elements with comprehensive selector
            $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '1');

            // Force hide any overlay elements
            $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').hide();
            $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').hide();

            $('#importModal').modal({
                backdrop: 'static',
                keyboard: false,
                show: true
            });

            // Fix accessibility and ensure modal appears on top after showing
            $('#importModal').on('shown.bs.modal', function() {
                // Fix aria-hidden accessibility issue aggressively
                const $modal = $(this);
                $modal.removeAttr('aria-hidden');
                $modal.attr('aria-modal', 'true');

                // Monitor and prevent aria-hidden from being set again
                const observer = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.type === 'attributes' && mutation.attributeName === 'aria-hidden') {
                            if ($modal.attr('aria-hidden') === 'true') {
                                $modal.removeAttr('aria-hidden');
                                $modal.attr('aria-modal', 'true');
                                console.log('🔧 Fixed aria-hidden violation on import modal');
                            }
                        }
                    });
                });

                observer.observe($modal[0], {
                    attributes: true,
                    attributeFilter: ['aria-hidden']
                });

                // Store observer for cleanup
                $modal.data('aria-observer', observer);
                // Force maximum z-index values
                $(this).css({
                    'z-index': 99999999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 99999999999,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000000,
                    'margin': '80px auto'
                });
                $('.modal-content').css({
                    'z-index': 1000000001
                });

                // Force body class and hide overlapping elements again
                $('body').addClass('modal-open').css('overflow', 'hidden');
                $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '1');

                // Fix Select2 dropdowns in modal
                $('#importModal .select2-dropdown').each(function() {
                    $(this).css('z-index', 1000000002);
                });


                // Focus on the modal
                $(this).focus();
            });

            // Cleanup when modal is hidden
            $('#importModal').on('hidden.bs.modal', function() {
                // Cleanup aria observer
                const observer = $(this).data('aria-observer');
                if (observer) {
                    observer.disconnect();
                    $(this).removeData('aria-observer');
                }

                $('body').removeClass('modal-open').css('overflow', '');
                $('.sidebar, .main-sidebar, .navbar, .main-header, .content-wrapper, .overlay, .page-overlay, .loading-overlay, .nav, .navigation, .header, .footer, .breadcrumb, .topbar, .app-header, .fixed-top, .fixed-bottom, .sticky-top').css('z-index', '');

                // Restore hidden overlay elements
                $('*[class*="overlay"]:not(.modal):not(.modal-backdrop)').show();
                $('*[id*="overlay"]:not(.modal):not(.modal-backdrop)').show();
            });
        }

        function downloadTemplate() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Downloading...';
            btn.disabled = true;

            // Create download link - use route helper with proper authentication context
            const downloadUrl = '{{ route("warehouse.items.template") }}';

            console.log('🔗 Attempting to download template from:', downloadUrl);

            fetch(downloadUrl, {
                method: 'GET',
                credentials: 'include',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
                .then(response => {
                    console.log('📡 Response received:', response.status, response.statusText);
                    if (!response.ok) {
                        throw new Error('Template download failed');
                    }
                    return response.blob();
                })
                .then(blob => {
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'items_import_template.xlsx';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);

                    showAlert('Template downloaded successfully!', 'success');
                })
                .catch(error => {
                    console.error('💥 Download error:', error);
                    console.error('🔗 Failed URL:', downloadUrl);
                    showAlert('Failed to download template: ' + error.message, 'error');
                })
                .finally(() => {
                    btn.innerHTML = originalHtml;
                    btn.disabled = false;
                });
        }

        function importItems() {
            const fileInput = $('#importFile')[0];

            // Validate inputs
            if (!fileInput.files.length) {
                showAlert('Please select a file to import', 'error');
                return;
            }

            const formData = new FormData();
            formData.append('file', fileInput.files[0]);
            formData.append('_token', '{{ csrf_token() }}');

            // Show progress bar
            $('#uploadProgress').show();
            $('#importResults').hide();

            // Update import button
            const importBtn = $('#importBtn');
            const originalHtml = importBtn.html();
            importBtn.prop('disabled', true);
            importBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Importing...');

            $.ajax({
                url: '{{ route("warehouse.items.import") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                xhr: function() {
                    var xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener("progress", function(evt) {
                        if (evt.lengthComputable) {
                            var percentComplete = (evt.loaded / evt.total) * 100;
                            $('.progress-bar').css('width', percentComplete + '%');
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    $('.progress-bar').css('width', '100%');

                    if (response.success) {
                        $('#importModal').modal('hide');
                        itemsTable.draw();

                        let message = `Import completed successfully! `;
                        if (response.imported) message += `${response.imported} items imported.`;
                        if (response.updated) message += ` ${response.updated} items updated.`;
                        if (response.skipped) message += ` ${response.skipped} rows skipped.`;

                        showAlert(message, 'success');

                        // Reset form
                        document.getElementById('importForm').reset();
                        $('.custom-file-label').text('Choose file...');
                    } else {
                        showAlert(response.message || 'Import failed', 'error');
                    }
                },
                error: function(xhr) {
                    let errorMessage = 'Import failed. Please check your file and try again.';

                    if (xhr.status === 422) {
                        const response = xhr.responseJSON;
                        if (response.errors) {
                            errorMessage = Object.values(response.errors).flat().join(' ');
                        } else if (response.message) {
                            errorMessage = response.message;
                        }
                    } else if (xhr.status === 413) {
                        errorMessage = 'File is too large. Please reduce file size and try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }

                    showAlert(errorMessage, 'error');
                },
                complete: function() {
                    // Reset import button
                    importBtn.prop('disabled', false);
                    importBtn.html(originalHtml);

                    // Hide progress bar after delay
                    setTimeout(() => {
                        $('#uploadProgress').hide();
                        $('.progress-bar').css('width', '0%');
                    }, 1000);
                }
            });
        }

        // File input label update
        $(document).on('change', '#importFile', function() {
            const fileName = $(this)[0].files[0]?.name || 'Choose file...';
            $(this).next('.custom-file-label').text(fileName);
        });
    </script>
@endpush