@extends('layouts.admin-simple')

@section('title', 'Material Issue Decision - Multi-Source')

@push('styles')
<style>
    .allocation-warning {
        margin: 0.5rem 0;
        animation: warningPulse 2s ease-in-out;
    }

    @keyframes warningPulse {
        0% { background-color: #f8d7da; }
        50% { background-color: #f5c6cb; }
        100% { background-color: #f8d7da; }
    }

    .quantity-input-wrapper {
        position: relative;
    }

    .quantity-input-warning {
        border: 2px solid #dc3545 !important;
        background-color: #f8d7da !important;
    }

    .allocation-summary {
        background: linear-gradient(135deg, #e3f2fd 0%, #f8f9fa 100%);
        border-left: 4px solid #2196f3;
    }
</style>
@endpush

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.material-issue.index') }}">Material Issue</a></li>
    <li class="breadcrumb-item active">Material Issue</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Material Issue</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Issue material request from multiple sources</p>
        </div>
        <div class="page-actions-right">
            <a href="{{ route('warehouse.material-issue.index') }}" class="btn btn-outline-secondary">
                <i class="material-icons mr-1" style="font-size: 16px;">arrow_back</i>
                Back to List
            </a>
        </div>
    </div>
@endsection

@section('content')
<div class="container-fluid">

    <!-- Request Information -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="material-icons mr-2">assignment</i>
                        Material Request Information
                    </h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-3">
                            <div class="info-item">
                                <label>Request Number</label>
                                <div class="text-primary font-weight-bold">
                                    {{ $transaction->material_request_number ?: 'MR-' . str_pad($transaction->id, 6, '0', STR_PAD_LEFT) }}
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="info-item">
                                <label>Request Date</label>
                                <div>{{ $transaction->transaction_date->format('d-M-Y') }}</div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="info-item">
                                <label>Project</label>
                                <div>{{ $transaction->project->project_name ?? 'N/A' }}</div>
                                @if($transaction->division)
                                    <small class="text-muted">{{ $transaction->division->name }}</small>
                                @endif
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="info-item">
                                <label>Requested By</label>
                                <div>{{ $transaction->requester->name ?? 'N/A' }}</div>
                            </div>
                        </div>
                    </div>

                    <!-- Common Reference Area -->
                    <div class="mt-4">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0">
                                <i class="material-icons mr-2">shopping_cart</i>
                                Requested Items Reference
                            </h6>
                            <div>
                                <button type="button" class="btn btn-sm btn-outline-secondary mr-2" onclick="enableAllSources()">
                                    <i class="material-icons mr-1" style="font-size: 14px;">check_box</i>
                                    Enable All
                                </button>
                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="autoAllocate()">
                                    <i class="material-icons mr-1" style="font-size: 14px;">auto_fix_high</i>
                                    Auto-Allocate
                                </button>
                            </div>
                        </div>

                        <!-- Allocation Summary Table -->
                        <div class="card border-secondary mb-3">
                            <div class="card-header">
                                <h6 class="card-title mb-0">
                                    <i class="material-icons mr-2">assignment_turned_in</i>
                                    Allocation Summary
                                </h6>
                            </div>
                            <div class="card-body py-2">
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered mb-0">
                                        <thead class="bg-light">
                                            <tr>
                                                <th>Item Code</th>
                                                <th width="80">Requested</th>
                                                <th width="80">Already Issued</th>
                                                <th width="80">Inventory</th>
                                                <th width="80">Delivery</th>
                                                <th width="80">Transfer</th>
                                                <th width="80">Total</th>
                                                <th width="80">Remaining</th>
                                                <th width="100">Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($transaction->items as $item)
                                                @php
                                                    // Get fulfillment breakdown by type for this item (fixed to include all fulfillment types)
                                                    $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                                                        ->where('item_id', $item->item_id)
                                                        ->get()
                                                        ->groupBy('fulfillment_type');

                                                    // Get fulfillment data directly to avoid groupBy issues
                                                    $allFulfillmentData = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                                                        ->where('item_id', $item->item_id)
                                                        ->get();

                                                    // Calculate quantities directly from raw data
                                                    $inventoryQty = $allFulfillmentData->whereIn('fulfillment_type', ['inventory', 'inventory_dispatch'])->sum('quantity_fulfilled');
                                                    $deliveryQty = $allFulfillmentData->where('fulfillment_type', 'direct_delivery')->sum('quantity_fulfilled');
                                                    $transferQty = $allFulfillmentData->where('fulfillment_type', 'project_transfer')->sum('quantity_fulfilled');
                                                    $totalIssued = $inventoryQty + $deliveryQty + $transferQty;
                                                @endphp
                                                <tr id="summary_row_{{ $item->item_id }}">
                                                    <td>
                                                        <strong class="text-primary">{{ $item->item->item_code ?? 'N/A' }}</strong>
                                                    </td>
                                                    <td class="text-center">
                                                        <strong>{{ number_format($item->quantity_requested, 0) }}</strong>
                                                    </td>
                                                    <td class="text-center">
                                                        @if($totalIssued > 0)
                                                            <span class="badge badge-success">{{ number_format($totalIssued, 0) }}</span>
                                                        @else
                                                            <span class="badge badge-light">0</span>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        @if($inventoryQty > 0)
                                                            <span id="summary_inv_{{ $item->item_id }}" class="badge badge-success">{{ number_format($inventoryQty, 0) }}</span>
                                                        @else
                                                            <span id="summary_inv_{{ $item->item_id }}" class="badge badge-light">0</span>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        @if($deliveryQty > 0)
                                                            <span id="summary_del_{{ $item->item_id }}" class="badge badge-info">{{ number_format($deliveryQty, 0) }}</span>
                                                        @else
                                                            <span id="summary_del_{{ $item->item_id }}" class="badge badge-light">0</span>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        @if($transferQty > 0)
                                                            <span id="summary_tra_{{ $item->item_id }}" class="badge badge-primary">{{ number_format($transferQty, 0) }}</span>
                                                        @else
                                                            <span id="summary_tra_{{ $item->item_id }}" class="badge badge-light">0</span>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        <strong id="summary_total_{{ $item->item_id }}">{{ number_format($totalIssued, 0) }}</strong>
                                                    </td>
                                                    <td class="text-center">
                                                        <span id="summary_remaining_{{ $item->item_id }}" class="text-danger">{{ number_format($item->quantity_remaining ?? $item->quantity_requested, 0) }}</span>
                                                    </td>
                                                    <td class="text-center">
                                                        @if(isset($item->quantity_remaining) && $item->quantity_remaining <= 0)
                                                            <span id="summary_status_{{ $item->item_id }}" class="badge badge-success">Fully Issued</span>
                                                        @elseif(isset($item->quantity_fulfilled) && $item->quantity_fulfilled > 0)
                                                            <span id="summary_status_{{ $item->item_id }}" class="badge badge-warning">Partially Issued</span>
                                                        @else
                                                            <span id="summary_status_{{ $item->item_id }}" class="badge badge-secondary">Pending</span>
                                                        @endif
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Display Validation Errors -->
    @if ($errors->any())
        <div class="alert alert-danger">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- Display Session Messages -->
    @if(session('error'))
        <div class="alert alert-danger">
            {{ session('error') }}
        </div>
    @endif

    @if(session('success'))
        <div class="alert alert-success">
            {{ session('success') }}
        </div>
    @endif

    @php
        // Calculate if all items are fully issued
        $allFullyIssued = $transaction->items->every(function($item) {
            return isset($item->quantity_remaining) && $item->quantity_remaining <= 0;
        });
    @endphp

    @if($allFullyIssued)
        <!-- Fully Issued Notice -->
        <div class="alert alert-success">
            <i class="material-icons align-middle mr-2">check_circle</i>
            <strong>Material Request Fully Issued:</strong> All requested items have been completely issued to the project. No further action required.
        </div>
    @else
        <!-- Multi-Source Material Issue Forms -->
        <form action="{{ route('warehouse.material-issue.process-multi', $transaction->id) }}" method="POST" id="multiSourceForm">
            @csrf
            <input type="hidden" name="multi_source" value="true">

        <!-- Hidden inputs for boolean fields -->
        <input type="hidden" name="enable_inventory" value="0" id="hidden_enable_inventory">
        <input type="hidden" name="enable_delivery" value="0" id="hidden_enable_delivery">
        <input type="hidden" name="enable_transfer" value="0" id="hidden_enable_transfer">

        <!-- Inventory Dispatch Section -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card border-success">
                    <div class="card-header bg-success text-white">
                        <h6 class="card-title mb-0">
                            <i class="material-icons mr-2">inventory</i>
                            Inventory Dispatch
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="form-check mb-3">
                            <input type="checkbox" class="form-check-input" id="enable_inventory" onchange="toggleSection('inventory')">
                            <label class="form-check-label" for="enable_inventory">
                                Enable Inventory Dispatch
                            </label>
                        </div>

                        <div id="inventory_section" style="display: none;">
                            <div class="form-group">
                                <label>Dispatch Date <span class="text-danger">*</span></label>
                                <input type="date" name="inventory_dispatch_date" class="form-control form-control-sm"
                                       value="{{ date('Y-m-d') }}" min="{{ date('Y-m-d') }}">
                            </div>

                            <div class="form-group">
                                <label>Receiver Name <span class="text-danger">*</span></label>
                                <input type="text" name="inventory_receiver_name" class="form-control form-control-sm">
                            </div>

                            <!-- Inventory Items List -->
                            <div class="form-group">
                                <label>Items for Inventory Dispatch</label>
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered">
                                        <thead class="bg-success text-white">
                                            <tr>
                                                <th>Item Code</th>
                                                <th>Description</th>
                                                <th width="100">Requested</th>
                                                <th width="100">Available</th>
                                                <th width="120">Dispatch Qty</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($transaction->items as $item)
                                                @php
                                                    $availableStock = \App\Models\Warehouse\Inventory::where('item_id', $item->item_id)
                                                        ->where('quantity_available', '>', 0)
                                                        ->sum('quantity_available') ?: 0;
                                                @endphp
                                                <tr data-item-id="{{ $item->item_id }}">
                                                    <td>
                                                        <strong class="text-primary">{{ $item->item->item_code ?? 'N/A' }}</strong>
                                                    </td>
                                                    <td>
                                                        <small>{{ $item->item->item_description ?? 'N/A' }}</small>
                                                    </td>
                                                    <td class="text-center">
                                                        <strong>{{ number_format($item->quantity_remaining ?? $item->quantity_requested, 0) }}</strong>
                                                        @if(isset($item->quantity_fulfilled) && $item->quantity_fulfilled > 0)
                                                            <br><small class="text-success">({{ number_format($item->quantity_fulfilled, 0) }} issued)</small>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        <span class="badge {{ $availableStock > 0 ? 'badge-success' : 'badge-danger' }}">
                                                            {{ number_format($availableStock, 0) }}
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               class="form-control form-control-sm inventory-qty"
                                                               name="inventory[{{ $item->item_id }}]"
                                                               data-item-id="{{ $item->item_id }}"
                                                               step="0.01" min="0"
                                                               max="{{ min($availableStock, $item->quantity_remaining ?? $item->quantity_requested) }}"
                                                               value="0"
                                                               onchange="updateAllocations({{ $item->item_id }})">
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                            </div>


                            <div class="form-group">
                                <label>Notes</label>
                                <textarea name="inventory_notes" class="form-control form-control-sm" rows="2"></textarea>
                            </div>

                            <div class="alert alert-success alert-sm">
                                <small>
                                    <i class="material-icons" style="font-size: 14px;">info</i>
                                    Items will be immediately deducted from warehouse stock
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Direct Delivery Section -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card border-info">
                    <div class="card-header bg-info text-white">
                        <h6 class="card-title mb-0">
                            <i class="material-icons mr-2">local_shipping</i>
                            Direct Delivery
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="form-check mb-3">
                            <input type="checkbox" class="form-check-input" id="enable_delivery" onchange="toggleSection('delivery')">
                            <label class="form-check-label" for="enable_delivery">
                                Enable Direct Delivery
                            </label>
                        </div>

                        <div id="delivery_section" style="display: none;">
                            <div class="form-group">
                                <label>Supplier</label>
                                <x-warehouse.select2-dropdown
                                    name="delivery_supplier_id"
                                    id="delivery_supplier_select"
                                    placeholder="Select Supplier"
                                    :options="collect([['value' => '', 'text' => 'Select Supplier']])->merge($suppliers->map(function($supplier) {
                                        return ['value' => $supplier->id, 'text' => $supplier->name];
                                    }))"
                                    allowClear="true"
                                    class="form-control-sm"
                                />
                            </div>

                            <div class="form-group">
                                <label>Expected Delivery Date <span class="text-danger">*</span></label>
                                <input type="date" name="delivery_date" class="form-control form-control-sm"
                                       value="{{ date('Y-m-d', strtotime('+7 days')) }}">
                            </div>

                            <div class="form-group">
                                <label>LPO Number</label>
                                <input type="text" name="delivery_lpo_number" class="form-control form-control-sm"
                                       placeholder="LPO-XXXXX">
                            </div>

                            <div class="form-group">
                                <label>Delivery Note Number <span class="text-danger">*</span></label>
                                <input type="text" name="delivery_note_number" class="form-control form-control-sm"
                                       placeholder="DN-XXXXX">
                            </div>

                            <!-- Direct Delivery Items List -->
                            <div class="form-group">
                                <label>Items for Direct Delivery</label>
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered">
                                        <thead class="bg-info text-white">
                                            <tr>
                                                <th>Item Code</th>
                                                <th>Description</th>
                                                <th width="80">Requested</th>
                                                <th width="100">Order Qty</th>
                                                <th width="100">Unit Price</th>
                                                <th width="120">Batch Number</th>
                                                <th width="120">Production Date</th>
                                                <th width="120">Expiry Date</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($transaction->items as $item)
                                                <tr data-item-id="{{ $item->item_id }}">
                                                    <td>
                                                        <strong class="text-info">{{ $item->item->item_code ?? 'N/A' }}</strong>
                                                    </td>
                                                    <td>
                                                        <small>{{ $item->item->item_description ?? 'N/A' }}</small>
                                                    </td>
                                                    <td class="text-center">
                                                        <strong>{{ number_format($item->quantity_remaining ?? $item->quantity_requested, 0) }}</strong>
                                                        @if(isset($item->quantity_fulfilled) && $item->quantity_fulfilled > 0)
                                                            <br><small class="text-success">({{ number_format($item->quantity_fulfilled, 0) }} issued)</small>
                                                        @endif
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               class="form-control form-control-sm delivery-qty"
                                                               name="delivery[{{ $item->item_id }}]"
                                                               data-item-id="{{ $item->item_id }}"
                                                               step="0.01" min="0"
                                                               max="{{ $item->quantity_requested }}"
                                                               value="0"
                                                               onchange="updateAllocations({{ $item->item_id }})">
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               class="form-control form-control-sm"
                                                               name="delivery_unit_price[{{ $item->item_id }}]"
                                                               step="0.01" min="0"
                                                               placeholder="0.00">
                                                    </td>
                                                    <td>
                                                        <input type="text"
                                                               class="form-control form-control-sm"
                                                               name="delivery_batch[{{ $item->item_id }}]"
                                                               placeholder="BATCH-XXXX">
                                                    </td>
                                                    <td>
                                                        <input type="date"
                                                               class="form-control form-control-sm"
                                                               name="delivery_production_date[{{ $item->item_id }}]">
                                                    </td>
                                                    <td>
                                                        <input type="date"
                                                               class="form-control form-control-sm"
                                                               name="delivery_expiry_date[{{ $item->item_id }}]">
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <div class="form-group">
                                <label>Delivery Site</label>
                                <input type="text" name="delivery_site_location" class="form-control form-control-sm"
                                       value="{{ $transaction->project->project_name ?? '' }}">
                            </div>

                            <div class="form-group">
                                <label>Notes</label>
                                <textarea name="delivery_notes" class="form-control form-control-sm" rows="2"></textarea>
                            </div>

                            <div class="alert alert-info alert-sm">
                                <small>
                                    <i class="material-icons" style="font-size: 14px;">info</i>
                                    Creates a direct delivery order for supplier
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Project Transfer Section -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card border-primary">
                    <div class="card-header bg-primary text-white">
                        <h6 class="card-title mb-0">
                            <i class="material-icons mr-2">swap_horiz</i>
                            Project Transfer
                        </h6>
                    </div>
                    <div class="card-body">
                        <div class="form-check mb-3">
                            <input type="checkbox" class="form-check-input" id="enable_transfer" onchange="toggleSection('transfer')">
                            <label class="form-check-label" for="enable_transfer">
                                Enable Project Transfer
                            </label>
                        </div>

                        <div id="transfer_section" style="display: none;">
                            <div class="form-group">
                                <label>Source Project <span class="text-danger">*</span></label>
                                <x-warehouse.select2-dropdown
                                    name="transfer_source_project_id"
                                    id="transfer_source_project_select"
                                    placeholder="Select Source Project"
                                    :options="collect([['value' => '', 'text' => 'Select Source Project']])->merge($projects->map(function($project) {
                                        return ['value' => $project->id, 'text' => $project->project_name];
                                    }))"
                                    allowClear="true"
                                    class="form-control-sm"
                                    :data="['onchange' => 'updateTransferReferences()']"
                                />
                            </div>

                            <div class="form-group" id="transfer_reference_section" style="display: none;">
                                <label>Reference Number <span class="text-danger">*</span></label>
                                <select name="transfer_reference_id" id="transfer_reference_select" class="form-control select2" style="width: 100%;">
                                    <option value="">Select Reference Number</option>
                                    <!-- Options will be populated via JavaScript -->
                                </select>
                                <small class="form-text text-muted">Select a specific reference for this transfer request</small>
                            </div>

                            <!-- Project Transfer Items List -->
                            <div class="form-group">
                                <label>Items for Project Transfer</label>
                                <div class="table-responsive">
                                    <table class="table table-sm table-bordered">
                                        <thead class="bg-primary text-white">
                                            <tr>
                                                <th>Item Code</th>
                                                <th>Description</th>
                                                <th width="100">Requested</th>
                                                <th width="100">Available</th>
                                                <th width="120">Transfer Qty</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($transaction->items as $item)
                                                <tr data-item-id="{{ $item->item_id }}">
                                                    <td>
                                                        <strong class="text-primary">{{ $item->item->item_code ?? 'N/A' }}</strong>
                                                    </td>
                                                    <td>
                                                        <small>{{ $item->item->item_description ?? 'N/A' }}</small>
                                                    </td>
                                                    <td class="text-center">
                                                        <strong>{{ number_format($item->quantity_remaining ?? $item->quantity_requested, 0) }}</strong>
                                                        @if(isset($item->quantity_fulfilled) && $item->quantity_fulfilled > 0)
                                                            <br><small class="text-success">({{ number_format($item->quantity_fulfilled, 0) }} issued)</small>
                                                        @endif
                                                    </td>
                                                    <td class="text-center">
                                                        <span class="badge badge-light" id="project_available_{{ $item->item_id }}">Select project</span>
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               class="form-control form-control-sm transfer-qty"
                                                               name="transfer[{{ $item->item_id }}]"
                                                               data-item-id="{{ $item->item_id }}"
                                                               step="0.01" min="0"
                                                               max="{{ $item->quantity_requested }}"
                                                               value="0"
                                                               onchange="updateAllocations({{ $item->item_id }})">
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <div class="form-group">
                                <label>Transfer Date <span class="text-danger">*</span></label>
                                <input type="date" name="transfer_date" class="form-control form-control-sm"
                                       value="{{ date('Y-m-d', strtotime('+3 days')) }}">
                            </div>

                            <div class="form-group">
                                <label>Priority</label>
                                <x-warehouse.select2-dropdown
                                    name="transfer_priority"
                                    id="transfer_priority_select"
                                    placeholder="Select Priority"
                                    :options="[
                                        ['value' => 'medium', 'text' => 'Medium'],
                                        ['value' => 'high', 'text' => 'High'],
                                        ['value' => 'urgent', 'text' => 'Urgent'],
                                        ['value' => 'low', 'text' => 'Low']
                                    ]"
                                    selected="medium"
                                    allowClear="false"
                                    class="form-control-sm"
                                />
                            </div>

                            <div class="form-group">
                                <label>Transfer Reason</label>
                                <input type="text" name="transfer_reason" class="form-control form-control-sm"
                                       value="Material shortage - multi-source issue">
                            </div>

                            <div class="form-group">
                                <label>Notes</label>
                                <textarea name="transfer_notes" class="form-control form-control-sm" rows="2"></textarea>
                            </div>

                            <div class="alert alert-primary alert-sm">
                                <small>
                                    <i class="material-icons" style="font-size: 14px;">info</i>
                                    Creates transfer request requiring approval
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Summary and Actions -->
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="material-icons mr-2">summarize</i>
                    Issue Summary
                </h6>
            </div>
            <div class="card-body">
                @php
                    $totalRequested = 0;
                    $totalFulfilled = 0;
                    $totalRemaining = 0;
                    $totalInventory = 0;
                    $totalDelivery = 0;
                    $totalTransfer = 0;

                    foreach($transaction->items as $item) {
                        $totalRequested += $item->quantity_requested;
                        $totalFulfilled += $item->quantity_fulfilled ?? 0;
                        $totalRemaining += $item->quantity_remaining ?? $item->quantity_requested;

                        // Get fulfillment data directly to avoid groupBy issues (same approach as above)
                        $itemFulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                            ->where('item_id', $item->item_id)
                            ->get();

                        // Calculate totals using direct filtering
                        $totalInventory += $itemFulfillments->whereIn('fulfillment_type', ['inventory', 'inventory_dispatch'])->sum('quantity_fulfilled');
                        $totalDelivery += $itemFulfillments->where('fulfillment_type', 'direct_delivery')->sum('quantity_fulfilled');
                        $totalTransfer += $itemFulfillments->where('fulfillment_type', 'project_transfer')->sum('quantity_fulfilled');
                    }
                @endphp

                <div class="row mb-3">
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-muted">Original Requested</label>
                            <h5 id="summary_original">{{ number_format($totalRequested, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-warning">Already Issued</label>
                            <h5 id="summary_already_issued" class="text-warning">{{ number_format($totalFulfilled, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-danger">Remaining to Issue</label>
                            <h5 id="summary_remaining_total" class="text-danger">{{ number_format($totalRemaining, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-muted">Current Session</label>
                            <h5 id="summary_current_session" class="text-muted">0.00</h5>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-success">From Inventory</label>
                            <h5 id="summary_inventory" class="text-success">{{ number_format($totalInventory, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-info">Direct Delivery</label>
                            <h5 id="summary_delivery" class="text-info">{{ number_format($totalDelivery, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-primary">Project Transfer</label>
                            <h5 id="summary_transfer" class="text-primary">{{ number_format($totalTransfer, 2) }}</h5>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="summary-item">
                            <label class="text-secondary">Total Issued</label>
                            <h5 id="summary_total_allocated" class="text-secondary">{{ number_format($totalFulfilled, 2) }}</h5>
                        </div>
                    </div>
                </div>

                <div class="alert alert-warning mt-3" id="warning_message" style="display: none;">
                    <i class="material-icons align-middle mr-1">warning</i>
                    <span id="warning_text"></span>
                </div>

                <div class="alert alert-success mt-3" id="success_message" style="display: none;">
                    <i class="material-icons align-middle mr-1">check_circle</i>
                    <span id="success_text">All items fully allocated!</span>
                </div>

                <hr>

                <div class="text-right">
                    <a href="{{ route('warehouse.material-issue.index') }}" class="btn btn-secondary mr-2">
                        <i class="material-icons mr-1" style="font-size: 16px;">cancel</i>
                        Cancel
                    </a>
                    <button type="submit" class="btn btn-success" id="submitBtn" disabled>
                        <i class="material-icons mr-1" style="font-size: 16px;">send</i>
                        Process Material Issue
                    </button>
                </div>
            </div>
        </div>
    </form>
    @endif
</div>
@endsection

@section('floating-tracker')
<!-- Floating Allocation Tracker - Rendered at body level -->
<div id="floating-tracker" class="d-none" style="position: fixed !important; top: 80px !important; right: 20px !important; z-index: 99999 !important;">
    <div class="card border-primary shadow">
        <div class="card-header bg-primary text-white py-2">
            <div class="d-flex justify-content-between align-items-center">
                <h6 class="mb-0" style="font-size: 14px;">
                    <i class="material-icons mr-1" style="font-size: 16px; vertical-align: middle;">track_changes</i>
                    Quick Tracker
                </h6>
                <button type="button" class="btn btn-sm btn-outline-light" onclick="toggleFloatingTracker()" style="padding: 2px 6px;">
                    <i class="material-icons" style="font-size: 14px;">close</i>
                </button>
            </div>
        </div>
        <div class="card-body py-2">
            <div id="floating-tracker-content">
                <!-- Content will be populated by JavaScript -->
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
.info-item label {
    font-size: 12px;
    color: #6c757d;
    text-transform: uppercase;
    font-weight: 600;
    margin-bottom: 2px;
}

.summary-item label {
    font-size: 12px;
    text-transform: uppercase;
    font-weight: 600;
    margin-bottom: 2px;
}

.summary-item h5 {
    margin: 0;
    font-weight: bold;
}

.alert-sm {
    padding: 0.5rem 0.75rem;
    font-size: 0.875rem;
}

.table-bordered th {
    border: 1px solid #dee2e6;
}

.form-control-sm {
    font-size: 0.875rem;
}

.reference-item {
    transition: border-color 0.3s ease;
}

.reference-item:hover {
    border-color: #007bff !important;
}

.inventory-items-list .item-row,
.delivery-items-list .item-row,
.transfer-items-list .item-row {
    background-color: #f8f9fa;
    transition: background-color 0.3s ease;
}

.inventory-items-list .item-row:hover {
    background-color: #e3f2e3;
}

.delivery-items-list .item-row:hover {
    background-color: #e3f2fd;
}

.transfer-items-list .item-row:hover {
    background-color: #e8f0fe;
}

#reference-panel {
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.progress {
    background-color: #e9ecef;
}

.progress-bar {
    transition: width 0.3s ease;
}

.allocation-status {
    font-size: 11px;
    font-weight: 600;
}

/* Fixed Tracker Styles - Toast Style */
#floating-tracker {
    position: fixed !important;
    top: 80px !important;
    right: 20px !important;
    width: 320px;
    max-height: 400px;
    z-index: 9999;
    transition: opacity 0.3s ease-in-out, transform 0.3s ease-in-out;
    animation: slideInFromRight 0.4s ease-out;
}

#floating-tracker .card {
    border: 1px solid #007bff;
    box-shadow: 0 4px 20px rgba(0,123,255,0.2);
    border-radius: 8px;
    background-color: white;
}

#floating-tracker .card-header {
    border-radius: 7px 7px 0 0;
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    border-bottom: 1px solid rgba(255,255,255,0.2);
    padding: 8px 12px;
}

#floating-tracker .card-body {
    max-height: 320px;
    overflow-y: auto;
    padding: 8px;
}

#floating-tracker .card-body::-webkit-scrollbar {
    width: 3px;
}

#floating-tracker .card-body::-webkit-scrollbar-track {
    background: #f8f9fa;
    border-radius: 2px;
}

#floating-tracker .card-body::-webkit-scrollbar-thumb {
    background: #007bff;
    border-radius: 2px;
}

#floating-tracker .card-body::-webkit-scrollbar-thumb:hover {
    background: #0056b3;
}

/* Slide in animation */
@keyframes slideInFromRight {
    0% {
        transform: translateX(100%);
        opacity: 0;
    }
    100% {
        transform: translateX(0);
        opacity: 1;
    }
}

/* Slide out animation */
#floating-tracker.hiding {
    animation: slideOutToRight 0.3s ease-in forwards;
}

@keyframes slideOutToRight {
    0% {
        transform: translateX(0);
        opacity: 1;
    }
    100% {
        transform: translateX(100%);
        opacity: 0;
    }
}

@media (max-width: 768px) {
    #floating-tracker {
        width: 280px;
        right: 15px;
        top: 15px;
        max-height: 350px;
    }

    #floating-tracker .card-body {
        max-height: 280px;
    }
}

@media (max-width: 576px) {
    #floating-tracker {
        width: calc(100vw - 30px);
        left: 15px;
        right: 15px;
        max-height: 300px;
    }

    #floating-tracker .card-body {
        max-height: 230px;
    }
}

/* Summary row highlighting */
.summary-row-updated {
    background-color: #fff3cd !important;
    animation: fadeHighlight 1s ease-out;
}

@keyframes fadeHighlight {
    0% { background-color: #d4edda; }
    100% { background-color: #fff3cd; }
}

/* Custom Select2 styling for decision page */
.select2-container {
    width: 100% !important;
}

.select2-container .select2-selection--single {
    height: 31px !important; /* Match form-control-sm height */
    border: 1px solid #ced4da;
    border-radius: 0.25rem;
    font-size: 0.875rem;
}

.select2-container--default .select2-selection--single .select2-selection__rendered {
    line-height: 29px;
    padding-left: 8px;
    font-size: 0.875rem;
}

.select2-container--default .select2-selection--single .select2-selection__arrow {
    height: 29px;
    right: 8px;
}

.select2-dropdown {
    border-radius: 0.25rem;
    border: 1px solid #ced4da;
    font-size: 0.875rem;
    width: 100% !important; /* Fixed width instead of auto */
    min-width: 200px !important;
    max-width: 300px !important; /* Limit maximum width */
}

.select2-container--open .select2-dropdown--below {
    width: 100% !important;
    min-width: 200px !important;
    max-width: 300px !important;
}

.select2-search__field {
    border: 1px solid #ced4da !important;
    border-radius: 0.25rem;
    font-size: 0.875rem !important;
    padding: 4px 8px !important;
}

.select2-container--default .select2-results__option {
    padding: 6px 12px;
    font-size: 0.875rem;
}

.select2-container--default .select2-results__option--highlighted[aria-selected] {
    background-color: #007bff;
}

.select2-container--default .select2-results__option[aria-selected=true] {
    background-color: #e7f3ff;
    color: #495057;
}
</style>
@endpush

@push('scripts')
<script>
let itemData = {};
let projectInventoryData = {}; // Store project inventory data globally

// Initialize item data
@foreach($transaction->items as $item)
    itemData[{{ $item->item_id }}] = {
        requested: {{ $item->quantity_remaining ?? $item->quantity_requested }},
        inventory: 0,
        delivery: 0,
        transfer: 0
    };
@endforeach

function toggleSection(type) {
    const checkbox = document.getElementById('enable_' + type);
    const section = document.getElementById(type + '_section');
    const hiddenInput = document.getElementById('hidden_enable_' + type);

    if (checkbox.checked) {
        section.style.display = 'block';
        hiddenInput.value = '1';

        // Add required attribute to relevant fields when section is enabled
        if (type === 'inventory') {
            const receiverName = document.querySelector('input[name="inventory_receiver_name"]');
            const dispatchDate = document.querySelector('input[name="inventory_dispatch_date"]');
            if (receiverName) receiverName.setAttribute('required', 'required');
            if (dispatchDate) dispatchDate.setAttribute('required', 'required');
        } else if (type === 'delivery') {
            const deliveryNote = document.querySelector('input[name="delivery_note_number"]');
            const deliveryDate = document.querySelector('input[name="delivery_date"]');
            if (deliveryNote) deliveryNote.setAttribute('required', 'required');
            if (deliveryDate) deliveryDate.setAttribute('required', 'required');
        } else if (type === 'transfer') {
            const sourceProject = document.querySelector('select[name="transfer_source_project_id"]');
            const transferDate = document.querySelector('input[name="transfer_date"]');
            if (sourceProject) sourceProject.setAttribute('required', 'required');
            if (transferDate) transferDate.setAttribute('required', 'required');
        }
    } else {
        section.style.display = 'none';
        hiddenInput.value = '0';

        // Remove required attribute when section is disabled
        if (type === 'inventory') {
            const receiverName = document.querySelector('input[name="inventory_receiver_name"]');
            const dispatchDate = document.querySelector('input[name="inventory_dispatch_date"]');
            if (receiverName) receiverName.removeAttribute('required');
            if (dispatchDate) dispatchDate.removeAttribute('required');
        } else if (type === 'delivery') {
            const deliveryNote = document.querySelector('input[name="delivery_note_number"]');
            const deliveryDate = document.querySelector('input[name="delivery_date"]');
            if (deliveryNote) deliveryNote.removeAttribute('required');
            if (deliveryDate) deliveryDate.removeAttribute('required');
        } else if (type === 'transfer') {
            const sourceProject = document.querySelector('select[name="transfer_source_project_id"]');
            const transferDate = document.querySelector('input[name="transfer_date"]');
            const transferReference = document.querySelector('select[name="transfer_reference_id"]');
            if (sourceProject) sourceProject.removeAttribute('required');
            if (transferDate) transferDate.removeAttribute('required');
            if (transferReference) transferReference.removeAttribute('required');
        }

        // Reset quantities for this type
        document.querySelectorAll('.' + type + '-qty').forEach(input => {
            input.value = 0;
            updateAllocations(input.dataset.itemId);
        });
    }

    validateForm();
    updateTrackerVisibility(); // Auto show/hide tracker
}

function updateAllocations(itemId) {
    console.log('🔄 updateAllocations called for item:', itemId);
    console.log('📞 Call stack:', new Error().stack);

    const inventoryInput = document.querySelector(`input[name="inventory[${itemId}]"]`);
    const deliveryInput = document.querySelector(`input[name="delivery[${itemId}]"]`);
    const transferInput = document.querySelector(`input[name="transfer[${itemId}]"]`);

    // Log input element existence and values
    console.log('🎯 Input elements:', {
        inventoryInput: !!inventoryInput,
        inventoryValue: inventoryInput ? inventoryInput.value : 'N/A',
        inventoryDisabled: inventoryInput ? inventoryInput.disabled : 'N/A',
        deliveryInput: !!deliveryInput,
        deliveryValue: deliveryInput ? deliveryInput.value : 'N/A',
        deliveryDisabled: deliveryInput ? deliveryInput.disabled : 'N/A',
        transferInput: !!transferInput,
        transferValue: transferInput ? transferInput.value : 'N/A',
        transferDisabled: transferInput ? transferInput.disabled : 'N/A'
    });

    const inventoryQty = inventoryInput ? parseFloat(inventoryInput.value) || 0 : 0;
    const deliveryQty = deliveryInput ? parseFloat(deliveryInput.value) || 0 : 0;
    const transferQty = transferInput ? parseFloat(transferInput.value) || 0 : 0;

    // Get the remaining quantity for this item
    const remainingQty = itemData[itemId] ? itemData[itemId].requested : 0;
    const totalAllocated = inventoryQty + deliveryQty + transferQty;

    console.log('Quantities for item', itemId, ':', {
        inventory: inventoryQty,
        delivery: deliveryQty,
        transfer: transferQty,
        total: totalAllocated,
        remaining: remainingQty,
        overAllocated: totalAllocated > remainingQty,
        exactDifference: (totalAllocated - remainingQty),
        withinTolerance: (totalAllocated <= remainingQty + 0.01)
    });

    // Validate allocation - show warning if over-allocated (with small tolerance for decimal precision)
    if (totalAllocated > remainingQty + 0.01) {
        const overAllocation = totalAllocated - remainingQty;
        showAllocationWarning(itemId, overAllocation, remainingQty);
    } else {
        clearAllocationWarning(itemId);
    }

    if (itemData[itemId]) {
        const requested = itemData[itemId].requested;
        const total = inventoryQty + deliveryQty + transferQty;

        // Update item data
        itemData[itemId] = {
            requested: requested,
            inventory: inventoryQty,
            delivery: deliveryQty,
            transfer: transferQty
        };

        console.log('Updated itemData for', itemId, ':', itemData[itemId]);

        // Update reference panel for this item
        updateReferencePanel(itemId, inventoryQty, deliveryQty, transferQty, requested);

        // Update summary
        updateSummary();
        validateForm();
    } else {
        console.error('itemData not found for itemId:', itemId);
    }
}

function updateReferencePanel(itemId, inventoryQty, deliveryQty, transferQty, requested) {
    const total = inventoryQty + deliveryQty + transferQty;
    const remaining = Math.max(0, requested - total);

    // Update summary table
    updateSummaryRow(itemId, inventoryQty, deliveryQty, transferQty, total, remaining, requested);

    // Update floating tracker if visible
    updateFloatingTracker();
}

function updateSummaryRow(itemId, inventoryQty, deliveryQty, transferQty, total, remaining, requested) {
    console.log(`🔄 Updating summary for item ${itemId}:`, {
        inventoryQty, deliveryQty, transferQty, total, remaining, requested,
        difference: Math.abs(total - requested),
        isComplete: Math.abs(total - requested) < 0.01
    });

    // Update allocation badges
    const invBadge = document.getElementById(`summary_inv_${itemId}`);
    const delBadge = document.getElementById(`summary_del_${itemId}`);
    const traBadge = document.getElementById(`summary_tra_${itemId}`);

    if (invBadge) {
        invBadge.textContent = inventoryQty > 0 ? Math.round(inventoryQty) : '0';
        invBadge.className = inventoryQty > 0 ? 'badge badge-success' : 'badge badge-light';
    }

    if (delBadge) {
        delBadge.textContent = deliveryQty > 0 ? Math.round(deliveryQty) : '0';
        delBadge.className = deliveryQty > 0 ? 'badge badge-info' : 'badge badge-light';
    }

    if (traBadge) {
        traBadge.textContent = transferQty > 0 ? Math.round(transferQty) : '0';
        traBadge.className = transferQty > 0 ? 'badge badge-primary' : 'badge badge-light';
    }

    // Update total
    const totalElement = document.getElementById(`summary_total_${itemId}`);
    if (totalElement) {
        totalElement.textContent = Math.round(total);
        totalElement.className = total > 0 ? 'text-success font-weight-bold' : 'text-muted';
    }

    // Update remaining
    const remainingElement = document.getElementById(`summary_remaining_${itemId}`);
    if (remainingElement) {
        if (remaining > 0) {
            remainingElement.textContent = Math.round(remaining);
            remainingElement.className = 'text-danger font-weight-bold';
        } else if (total > requested) {
            remainingElement.textContent = '+' + Math.round(total - requested);
            remainingElement.className = 'text-warning font-weight-bold';
        } else {
            remainingElement.textContent = '0';
            remainingElement.className = 'text-success font-weight-bold';
        }
    }

    // Update status
    const statusElement = document.getElementById(`summary_status_${itemId}`);
    if (statusElement) {
        if (total === 0) {
            statusElement.textContent = 'Pending';
            statusElement.className = 'badge badge-secondary';
        } else if (Math.abs(total - requested) < 0.01) {
            statusElement.textContent = 'Complete';
            statusElement.className = 'badge badge-success';
        } else if (total < requested) {
            statusElement.textContent = 'Partial';
            statusElement.className = 'badge badge-warning';
        } else {
            statusElement.textContent = 'Excess';
            statusElement.className = 'badge badge-danger';
        }
    }

    // Highlight updated row
    const row = document.getElementById(`summary_row_${itemId}`);
    if (row) {
        row.classList.add('summary-row-updated');
        setTimeout(() => row.classList.remove('summary-row-updated'), 1000);
    }
}

function updateSummary() {
    let totalInventory = 0;
    let totalDelivery = 0;
    let totalTransfer = 0;
    let totalRemainingToIssue = 0;

    for (let itemId in itemData) {
        totalInventory += itemData[itemId].inventory;
        totalDelivery += itemData[itemId].delivery;
        totalTransfer += itemData[itemId].transfer;
        totalRemainingToIssue += itemData[itemId].requested; // This is the remaining quantity for each item
    }

    document.getElementById('summary_inventory').textContent = totalInventory.toFixed(2);
    document.getElementById('summary_delivery').textContent = totalDelivery.toFixed(2);
    document.getElementById('summary_transfer').textContent = totalTransfer.toFixed(2);

    const totalAllocated = totalInventory + totalDelivery + totalTransfer;
    document.getElementById('summary_current_session').textContent = totalAllocated.toFixed(2);
    document.getElementById('summary_total_allocated').textContent = totalAllocated.toFixed(2);

    // Update the remaining total with the correct calculated value
    document.getElementById('summary_remaining_total').textContent = totalRemainingToIssue.toFixed(2);

    const stillRemaining = totalRemainingToIssue - totalAllocated;

    const warningEl = document.getElementById('warning_message');
    const successEl = document.getElementById('success_message');

    if (stillRemaining > 0.01) {
        warningEl.style.display = 'block';
        successEl.style.display = 'none';
        document.getElementById('warning_text').textContent = `${stillRemaining.toFixed(2)} units still need to be allocated`;
    } else if (totalAllocated > remainingToIssue + 0.01) {
        warningEl.style.display = 'block';
        successEl.style.display = 'none';
        document.getElementById('warning_text').textContent = `Over-allocated by ${(totalAllocated - remainingToIssue).toFixed(2)} units`;
    } else if (Math.abs(stillRemaining) < 0.01) {
        warningEl.style.display = 'none';
        successEl.style.display = 'block';
    } else {
        warningEl.style.display = 'none';
        successEl.style.display = 'none';
    }
}

function validateForm() {
    const inventoryEnabled = document.getElementById('enable_inventory').checked;
    const deliveryEnabled = document.getElementById('enable_delivery').checked;
    const transferEnabled = document.getElementById('enable_transfer').checked;

    // Check if sections are actually visible
    const inventorySection = document.getElementById('inventory_section');
    const deliverySection = document.getElementById('delivery_section');
    const transferSection = document.getElementById('transfer_section');

    console.log('🔍 Section visibility:', {
        inventoryEnabled: inventoryEnabled,
        inventoryVisible: inventorySection ? inventorySection.style.display !== 'none' : false,
        deliveryEnabled: deliveryEnabled,
        deliveryVisible: deliverySection ? deliverySection.style.display !== 'none' : false,
        transferEnabled: transferEnabled,
        transferVisible: transferSection ? transferSection.style.display !== 'none' : false
    });

    // Check if at least one source is enabled
    const hasSource = inventoryEnabled || deliveryEnabled || transferEnabled;

    // Check if there are any allocations
    let hasAllocations = false;

    for (let itemId in itemData) {
        const total = itemData[itemId].inventory + itemData[itemId].delivery + itemData[itemId].transfer;
        if (total > 0) {
            hasAllocations = true;
            break; // Found at least one allocation, no need to continue
        }
    }

    // Check required fields for enabled sections
    let requiredFieldsValid = true;
    let missingFields = [];

    if (inventoryEnabled) {
        const receiverName = document.querySelector('input[name="inventory_receiver_name"]');
        const dispatchDate = document.querySelector('input[name="inventory_dispatch_date"]');

        if (!receiverName || !receiverName.value.trim()) {
            requiredFieldsValid = false;
            missingFields.push('Inventory Receiver Name');
        }
        if (!dispatchDate || !dispatchDate.value.trim()) {
            requiredFieldsValid = false;
            missingFields.push('Inventory Dispatch Date');
        }
    }

    if (deliveryEnabled) {
        const deliveryNoteNumber = document.querySelector('input[name="delivery_note_number"]');

        if (!deliveryNoteNumber || !deliveryNoteNumber.value.trim()) {
            requiredFieldsValid = false;
            missingFields.push('Delivery Note Number');
        }
    }

    if (transferEnabled) {
        const sourceProject = document.querySelector('select[name="transfer_source_project_id"]');
        const transferDate = document.querySelector('input[name="transfer_date"]');

        if (!sourceProject || !sourceProject.value.trim()) {
            requiredFieldsValid = false;
            missingFields.push('Transfer Source Project');
        }
        if (!transferDate || !transferDate.value.trim()) {
            requiredFieldsValid = false;
            missingFields.push('Transfer Date');
        }

        // Check transfer reference if source project is selected
        if (sourceProject && sourceProject.value.trim()) {
            const transferReference = document.querySelector('select[name="transfer_reference_id"]');
            if (!transferReference || !transferReference.value.trim()) {
                requiredFieldsValid = false;
                missingFields.push('Transfer Reference Number');
            }
        }
    }

    const submitBtn = document.getElementById('submitBtn');
    if (submitBtn) {
        submitBtn.disabled = !hasSource || !hasAllocations || !requiredFieldsValid;

        // Enhanced debug logging
        console.log('🔍 Form validation debug:', {
            hasSource: hasSource,
            hasAllocations: hasAllocations,
            requiredFieldsValid: requiredFieldsValid,
            missingFields: missingFields,
            inventoryEnabled: inventoryEnabled,
            deliveryEnabled: deliveryEnabled,
            transferEnabled: transferEnabled,
            buttonDisabled: submitBtn.disabled,
            itemDataCount: Object.keys(itemData).length,
            submitButtonExists: !!submitBtn
        });

        // Log each item's allocation details with more info
        for (let itemId in itemData) {
            const data = itemData[itemId];
            const total = data.inventory + data.delivery + data.transfer;
            const hasAnyAllocation = total > 0;

            // Get actual input values for verification
            const inventoryInput = document.querySelector(`input[name="inventory[${itemId}]"]`);
            const deliveryInput = document.querySelector(`input[name="delivery[${itemId}]"]`);
            const transferInput = document.querySelector(`input[name="transfer[${itemId}]"]`);

            const actualInventory = inventoryInput ? parseFloat(inventoryInput.value) || 0 : 0;
            const actualDelivery = deliveryInput ? parseFloat(deliveryInput.value) || 0 : 0;
            const actualTransfer = transferInput ? parseFloat(transferInput.value) || 0 : 0;

            console.log(`📊 Item ${itemId}:`, {
                requested: data.requested,
                storedData: { inventory: data.inventory, delivery: data.delivery, transfer: data.transfer },
                actualInputs: { inventory: actualInventory, delivery: actualDelivery, transfer: actualTransfer },
                total: total,
                hasAllocation: hasAnyAllocation,
                inputsMatch: data.inventory === actualInventory && data.delivery === actualDelivery && data.transfer === actualTransfer
            });
        }

        // Check if button should be enabled
        const shouldBeEnabled = hasSource && hasAllocations;
        console.log(`🎯 Button state: Should be ${shouldBeEnabled ? 'ENABLED' : 'DISABLED'} - Currently ${submitBtn.disabled ? 'DISABLED' : 'ENABLED'}`);

        if (shouldBeEnabled && submitBtn.disabled) {
            console.log('⚠️  Button should be enabled but is disabled!');
        } else if (!shouldBeEnabled && !submitBtn.disabled) {
            console.log('⚠️  Button should be disabled but is enabled!');
        }
    }
}

function enableAllSources() {
    ['inventory', 'delivery', 'transfer'].forEach(type => {
        const checkbox = document.getElementById('enable_' + type);
        if (!checkbox.checked) {
            checkbox.checked = true;
            toggleSection(type);
        }
    });
    updateTrackerVisibility(); // Show tracker when enabling sections
}

// Auto-allocate function
function autoAllocate() {
    @foreach($transaction->items as $item)
        @php
            $availableStock = \App\Models\Warehouse\Inventory::where('item_id', $item->item_id)
                ->where('quantity_available', '>', 0)
                ->sum('quantity_available') ?: 0;
        @endphp

        const itemId{{ $item->item_id }} = {{ $item->item_id }};
        const requested{{ $item->item_id }} = {{ $item->quantity_requested }};
        const available{{ $item->item_id }} = {{ $availableStock }};

        // First, allocate from inventory if available
        const inventoryAlloc{{ $item->item_id }} = Math.min(requested{{ $item->item_id }}, available{{ $item->item_id }});
        if (inventoryAlloc{{ $item->item_id }} > 0 && document.getElementById('enable_inventory').checked) {
            document.querySelector(`input[name="inventory[${itemId{{ $item->item_id }}}]"]`).value = inventoryAlloc{{ $item->item_id }};
        }

        // Remaining goes to direct delivery or transfer
        const remaining{{ $item->item_id }} = requested{{ $item->item_id }} - inventoryAlloc{{ $item->item_id }};
        if (remaining{{ $item->item_id }} > 0) {
            if (document.getElementById('enable_delivery').checked) {
                document.querySelector(`input[name="delivery[${itemId{{ $item->item_id }}}]"]`).value = remaining{{ $item->item_id }};
            } else if (document.getElementById('enable_transfer').checked) {
                document.querySelector(`input[name="transfer[${itemId{{ $item->item_id }}}]"]`).value = remaining{{ $item->item_id }};
            }
        }

        updateAllocations(itemId{{ $item->item_id }});
    @endforeach
}

// Function to update item availability based on selected reference
function updateReferenceItemAvailability() {
    const referenceSelect = $('#transfer_reference_select');
    const selectedReference = referenceSelect.val();
    const projectId = $('#transfer_source_project_select').val();

    console.log('Selected reference:', selectedReference);

    if (!selectedReference || !projectId) {
        // Reset availability if no reference selected
        @foreach($transaction->items as $item)
            const resetElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
            if (resetElement{{ $item->item_id }}) {
                resetElement{{ $item->item_id }}.textContent = 'Select reference first';
                resetElement{{ $item->item_id }}.className = 'badge badge-light';
            }
        @endforeach
        return;
    }

    // Parse reference type and ID
    const [refType, refId] = selectedReference.split('_');
    console.log('Reference type:', refType, 'ID:', refId);

    // Based on reference type, fetch specific item availability
    let fetchUrl = '';

    // First, show the stored project inventory while we fetch reference-specific data
    @foreach($transaction->items as $item)
        const tempElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
        if (tempElement{{ $item->item_id }}) {
            const tempQty = projectInventoryData[{{ $item->item_id }}] || 0;
            tempElement{{ $item->item_id }}.textContent = `Loading...`;
            tempElement{{ $item->item_id }}.className = 'badge badge-secondary';
        }
    @endforeach

    switch(refType) {
        case 'MR':
            // Material Request - get remaining items from this request
            fetchUrl = `{{ url('/warehouse/transfer-requests/material-request') }}/${refId}/items`;
            break;
        case 'DD':
            // Direct Delivery - get items from this delivery
            fetchUrl = `{{ url('/warehouse/transfer-requests/direct-delivery') }}/${refId}/items`;
            break;
        case 'MT':
            // Material Transfer - get approved transfer items
            fetchUrl = `{{ url('/warehouse/transfer-requests/material-transfer') }}/${refId}`;
            break;
        default:
            console.error('Unknown reference type:', refType);
            // Fall back to project inventory
            @foreach($transaction->items as $item)
                const fallbackElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                if (fallbackElement{{ $item->item_id }}) {
                    const fallbackQty = projectInventoryData[{{ $item->item_id }}] || 0;
                    fallbackElement{{ $item->item_id }}.textContent = `Available: ${fallbackQty}`;
                    fallbackElement{{ $item->item_id }}.className = fallbackQty > 0 ? 'badge badge-success' : 'badge badge-danger';
                }
            @endforeach
            return;
    }

    // Fetch reference-specific items
    console.log('Fetching reference items from:', fetchUrl);
    fetch(fetchUrl)
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Reference items data:', data);

            // Handle different response structures for different reference types
            const items = data.items || (data.data && data.data.items) || [];
            if (data.success && items.length > 0) {
                // Update availability based on reference items
                @foreach($transaction->items as $item)
                    const refElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                    if (refElement{{ $item->item_id }}) {
                        const refItem = items.find(i => i.item_id == {{ $item->item_id }});
                        let availableQty = 0;

                        if (refItem) {
                            // Determine available quantity based on reference type
                            if (refType === 'MR') {
                                // For Material Request: use remaining quantity
                                availableQty = refItem.quantity_remaining || refItem.quantity_balance || 0;
                            } else if (refType === 'DD') {
                                // For Direct Delivery: use net available quantity (already calculated server-side)
                                availableQty = refItem.net_available_quantity || 0;
                            } else if (refType === 'MT') {
                                // For Material Transfer: use remaining transferable quantity
                                availableQty = refItem.remaining_transferable || 0;
                            }
                        }

                        refElement{{ $item->item_id }}.textContent = `Available: ${availableQty}`;

                        // Update badge color
                        if (availableQty >= {{ $item->quantity_remaining ?? $item->quantity_requested }}) {
                            refElement{{ $item->item_id }}.className = 'badge badge-success';
                        } else if (availableQty > 0) {
                            refElement{{ $item->item_id }}.className = 'badge badge-warning';
                        } else {
                            refElement{{ $item->item_id }}.className = 'badge badge-danger';
                        }

                        // Update max value for transfer quantity input
                        const transferInput = document.querySelector(`input[name="transfer[{{ $item->item_id }}]"]`);
                        if (transferInput) {
                            const remainingQty = {{ $item->quantity_remaining ?? $item->quantity_requested }};
                            transferInput.max = Math.min(availableQty, remainingQty);
                        }
                    }
                @endforeach
            } else {
                // Fall back to project inventory if no reference items
                console.log('No reference items found, using project inventory');
                @foreach($transaction->items as $item)
                    const noRefElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                    if (noRefElement{{ $item->item_id }}) {
                        const projQty = projectInventoryData[{{ $item->item_id }}] || 0;
                        noRefElement{{ $item->item_id }}.textContent = `Available: ${projQty}`;
                        noRefElement{{ $item->item_id }}.className = projQty > 0 ? 'badge badge-success' : 'badge badge-danger';
                    }
                @endforeach
            }
        })
        .catch(error => {
            console.error('Error fetching reference items:', error);
            // Fall back to project inventory on error
            @foreach($transaction->items as $item)
                const errorElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                if (errorElement{{ $item->item_id }}) {
                    const errorQty = projectInventoryData[{{ $item->item_id }}] || 0;
                    errorElement{{ $item->item_id }}.textContent = `Available: ${errorQty} (Project Stock)`;
                    errorElement{{ $item->item_id }}.className = errorQty > 0 ? 'badge badge-info' : 'badge badge-danger';
                }
            @endforeach
        });
}

// Function to update transfer references based on selected project
function updateTransferReferences() {
    console.log('updateTransferReferences called');
    validateForm(); // Validate form when source project changes
    const sourceProjectSelect = $('#transfer_source_project_select');
    const projectId = sourceProjectSelect.val();
    const referenceSection = document.getElementById('transfer_reference_section');
    const referenceSelect = $('#transfer_reference_select');

    console.log('Selected project ID:', projectId);

    if (!projectId) {
        referenceSection.style.display = 'none';
        // Clear and reset reference dropdown
        referenceSelect.empty().append('<option value="">Select Reference Number</option>');
        // Clear stored inventory data
        projectInventoryData = {};
        // Reset project availability indicators
        @foreach($transaction->items as $item)
            const availableElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
            if (availableElement{{ $item->item_id }}) {
                availableElement{{ $item->item_id }}.textContent = 'Available: Select project first';
                availableElement{{ $item->item_id }}.className = 'badge badge-light';
            }
        @endforeach
        return;
    }

    // Show loading state
    referenceSelect.empty().append('<option value="">Loading references...</option>');
    referenceSection.style.display = 'block';

    // Fetch project references
    console.log('Fetching references for project:', projectId);
    fetch(`{{ url('/warehouse/transfer-requests/project') }}/${projectId}/combined-references`)
        .then(response => {
            console.log('Response status:', response.status);
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Received data:', data);
            // Clear and populate reference dropdown
            referenceSelect.empty();
            referenceSelect.append('<option value="">Select Reference Number</option>');

            if (data.success && data.data && data.data.length > 0) {
                data.data.forEach(ref => {
                    referenceSelect.append(`<option value="${ref.id}">${ref.text}</option>`);
                });
            } else {
                referenceSelect.append('<option value="" disabled>No recent references available</option>');
            }
        })
        .catch(error => {
            console.error('Error fetching project references:', error);
            referenceSelect.empty().append('<option value="" disabled>Error loading references</option>');
        });

    // Fetch project inventory
    fetch(`{{ url('/warehouse/transfer-requests/project') }}/${projectId}/inventory`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            // Store inventory data globally
            if (data.success && data.data) {
                projectInventoryData = {};
                data.data.forEach(item => {
                    projectInventoryData[item.item_id] = item.quantity_available;
                });
                console.log('Stored project inventory data:', projectInventoryData);
            }

            // Update project inventory availability for each item
            @foreach($transaction->items as $item)
                const availableElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                if (availableElement{{ $item->item_id }}) {
                    const itemInventory = data.success && data.data ? data.data.find(inv => inv.item_id == {{ $item->item_id }}) : null;
                    const availableQty = itemInventory ? itemInventory.quantity_available : 0;

                    availableElement{{ $item->item_id }}.textContent = `Available: ${availableQty}`;
                    availableElement{{ $item->item_id }}.className = availableQty > 0 ? 'badge badge-success' : 'badge badge-danger';

                    // Update max value for transfer quantity input
                    const transferInput = document.querySelector(`input[name="transfer[{{ $item->item_id }}]"]`);
                    if (transferInput) {
                        const remainingQty = {{ $item->quantity_remaining ?? $item->quantity_requested }};
                        transferInput.max = Math.min(availableQty, remainingQty);
                    }
                }
            @endforeach
        })
        .catch(error => {
            console.error('Error fetching project data:', error);
            // Update availability indicators to show error
            @foreach($transaction->items as $item)
                const errorElement{{ $item->item_id }} = document.getElementById('project_available_{{ $item->item_id }}');
                if (errorElement{{ $item->item_id }}) {
                    errorElement{{ $item->item_id }}.textContent = 'Error loading availability';
                    errorElement{{ $item->item_id }}.className = 'badge badge-danger';
                }
            @endforeach
        });
}

// Toggle floating tracker with animation
function toggleFloatingTracker() {
    const tracker = document.getElementById('floating-tracker');

    if (tracker.classList.contains('d-none')) {
        // Show tracker with slide-in animation
        tracker.classList.remove('d-none');
        tracker.classList.remove('hiding');
        updateFloatingTracker();
    } else {
        // Hide tracker with slide-out animation
        tracker.classList.add('hiding');
        setTimeout(() => {
            tracker.classList.add('d-none');
            tracker.classList.remove('hiding');
        }, 300); // Match animation duration
    }
}

// Automatically show/hide tracker based on section status
function updateTrackerVisibility() {
    const inventoryEnabled = document.getElementById('enable_inventory').checked;
    const deliveryEnabled = document.getElementById('enable_delivery').checked;
    const transferEnabled = document.getElementById('enable_transfer').checked;
    const tracker = document.getElementById('floating-tracker');

    // Check if any section is enabled
    const anySectionEnabled = inventoryEnabled || deliveryEnabled || transferEnabled;

    if (anySectionEnabled) {
        // Show tracker if hidden
        if (tracker.classList.contains('d-none')) {
            tracker.classList.remove('d-none');
            tracker.classList.remove('hiding');
            updateFloatingTracker();
        }
    } else {
        // Hide tracker if all sections disabled
        if (!tracker.classList.contains('d-none')) {
            tracker.classList.add('hiding');
            setTimeout(() => {
                tracker.classList.add('d-none');
                tracker.classList.remove('hiding');
            }, 300);
        }
    }
}

// Update floating tracker content
function updateFloatingTracker() {
    const tracker = document.getElementById('floating-tracker');
    if (tracker.classList.contains('d-none')) return;

    const content = document.getElementById('floating-tracker-content');
    let html = '<div class="table-responsive"><table class="table table-sm mb-0">';
    html += '<thead class="bg-light"><tr>';
    html += '<th style="font-size:11px;">Item</th>';
    html += '<th style="font-size:11px;" width="40">Inv</th>';
    html += '<th style="font-size:11px;" width="40">Del</th>';
    html += '<th style="font-size:11px;" width="40">Tra</th>';
    html += '<th style="font-size:11px;" width="40">Rem</th>';
    html += '</tr></thead><tbody>';

    for (let itemId in itemData) {
        const data = itemData[itemId];
        const total = data.inventory + data.delivery + data.transfer;
        const remaining = Math.max(0, data.requested - total);

        html += `<tr style="font-size: 11px;">`;
        html += `<td><small class="text-primary font-weight-bold">ITM-${itemId}</small></td>`;
        html += `<td class="text-center">${data.inventory > 0 ? Math.round(data.inventory) : '-'}</td>`;
        html += `<td class="text-center">${data.delivery > 0 ? Math.round(data.delivery) : '-'}</td>`;
        html += `<td class="text-center">${data.transfer > 0 ? Math.round(data.transfer) : '-'}</td>`;

        if (remaining > 0) {
            html += `<td class="text-center text-danger font-weight-bold">${Math.round(remaining)}</td>`;
        } else if (total > data.requested) {
            html += `<td class="text-center text-warning font-weight-bold">+${Math.round(total - data.requested)}</td>`;
        } else {
            html += `<td class="text-center text-success font-weight-bold">✓</td>`;
        }
        html += `</tr>`;
    }

    html += '</tbody></table></div>';
    content.innerHTML = html;
}

// Show allocation warning for over-allocated items
function showAllocationWarning(itemId, overAllocation, remainingQty) {
    // Remove existing warning first
    clearAllocationWarning(itemId);

    // Find the item row in any of the tables
    const itemRow = document.querySelector(`[data-item-id="${itemId}"]`)?.closest('tr');
    if (!itemRow) return;

    // Highlight the input fields with warning styling
    const inputs = document.querySelectorAll(`[data-item-id="${itemId}"]`);
    inputs.forEach(input => {
        if (input.value && parseFloat(input.value) > 0) {
            input.classList.add('quantity-input-warning');
        }
    });

    // Create warning element
    const warningDiv = document.createElement('div');
    warningDiv.className = 'allocation-warning alert alert-danger mt-2 py-2 px-3';
    warningDiv.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="material-icons text-danger mr-2" style="font-size: 18px;">warning</i>
            <div>
                <strong>⚠️ Over-Allocation Warning!</strong><br>
                <small>You've allocated <strong>${overAllocation}</strong> more than the remaining quantity of <strong>${remainingQty}</strong>.<br>
                Please reduce the quantities before submitting.</small>
            </div>
        </div>
    `;

    // Insert warning after the row
    itemRow.parentNode.insertBefore(warningDiv, itemRow.nextSibling);
}

// Clear allocation warning
function clearAllocationWarning(itemId) {
    const warnings = document.querySelectorAll('.allocation-warning');
    warnings.forEach(warning => {
        const row = warning.previousElementSibling;
        if (row && row.querySelector(`[data-item-id="${itemId}"]`)) {
            warning.remove();
        }
    });

    // Remove warning styling from input fields
    const inputs = document.querySelectorAll(`[data-item-id="${itemId}"]`);
    inputs.forEach(input => {
        input.classList.remove('quantity-input-warning');
    });
}

// Enhanced form validation to prevent over-allocation
function validateAllocation() {
    let hasErrors = false;
    const errors = [];

    console.log('=== VALIDATION DEBUG ===');

    for (let itemId in itemData) {
        const data = itemData[itemId];
        const total = data.inventory + data.delivery + data.transfer;
        const remainingQty = data.requested; // This is the remaining quantity to be issued

        console.log(`Item ${itemId}:`, {
            inventory: data.inventory,
            delivery: data.delivery,
            transfer: data.transfer,
            totalAllocating: total,
            remainingToIssue: remainingQty,
            difference: total - remainingQty,
            isOverAllocated: total > remainingQty + 0.01
        });

        // Allow reasonable decimal tolerance (0.1) to account for rounding
        if (total > remainingQty + 0.1) {
            hasErrors = true;
            const overAllocation = (total - remainingQty).toFixed(2);
            errors.push(`Item ${itemId}: Over-allocated by ${overAllocation} units (Allocating: ${total}, Remaining to issue: ${remainingQty})`);
        } else if (total > 0 && total <= remainingQty + 0.1) {
            console.log(`✅ Item ${itemId} allocation is valid`);
        }
    }

    if (hasErrors) {
        alert('❌ Allocation Error!\n\nYou cannot allocate more than the required quantity:\n\n' + errors.join('\n\n') + '\n\nPlease adjust the quantities before submitting.');
        return false;
    }

    return true;
}

// Debug function to check form state
function debugFormState() {
    console.log('=== FORM DEBUG INFO ===');

    const inventoryEnabled = document.getElementById('enable_inventory').checked;
    const deliveryEnabled = document.getElementById('enable_delivery').checked;
    const transferEnabled = document.getElementById('enable_transfer').checked;
    const submitBtn = document.getElementById('submitBtn');

    console.log('Sections enabled:', {
        inventory: inventoryEnabled,
        delivery: deliveryEnabled,
        transfer: transferEnabled
    });

    console.log('Item data:', itemData);

    let hasAllocations = false;
    for (let itemId in itemData) {
        const total = itemData[itemId].inventory + itemData[itemId].delivery + itemData[itemId].transfer;
        console.log(`Item ${itemId}:`, {
            requested: itemData[itemId].requested,
            inventory: itemData[itemId].inventory,
            delivery: itemData[itemId].delivery,
            transfer: itemData[itemId].transfer,
            total: total
        });
        if (total > 0) hasAllocations = true;
    }

    console.log('Has allocations:', hasAllocations);
    console.log('Submit button disabled:', submitBtn.disabled);

    // Check form fields
    if (inventoryEnabled) {
        const receiverName = document.querySelector('input[name="inventory_receiver_name"]');
        const dispatchDate = document.querySelector('input[name="inventory_dispatch_date"]');
        console.log('Inventory required fields:', {
            receiverName: receiverName ? receiverName.value : 'not found',
            dispatchDate: dispatchDate ? dispatchDate.value : 'not found'
        });
    }

    alert('Debug info logged to console. Press F12 to view.');
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    updateSummary();

    // Initialize reference panels for all items
    @foreach($transaction->items as $item)
        @if(isset($item->quantity_remaining) && $item->quantity_remaining <= 0)
            // Skip updating status for fully issued items - keep original HTML status
            console.log('Item {{ $item->item_id }} is fully issued, preserving original status');
        @else
            updateReferencePanel({{ $item->item_id }}, 0, 0, 0, {{ $item->quantity_remaining ?? $item->quantity_requested }});
        @endif
    @endforeach

    // Add form submit debug handler
    const form = document.getElementById('multiSourceForm');
    if (form) {
        form.addEventListener('submit', function(e) {
            console.log('🚀 Form submission attempted');

            // First, validate allocation quantities
            const allocationValid = validateAllocation();
            console.log('Allocation validation result:', allocationValid);

            if (!allocationValid) {
                console.log('❌ Form submission blocked - allocation validation failed');
                e.preventDefault();
                return false; // This stops form submission
            }

            console.log('✅ Allocation validation passed, continuing with form submission...');

            // Debug: Log form data
            const formData = new FormData(form);
            console.log('Form Data being sent:');
            for (let [key, value] of formData.entries()) {
                console.log(`${key}: ${value}`);
            }

            // Check if any sections are enabled
            const inventoryEnabled = document.getElementById('enable_inventory').checked;
            const deliveryEnabled = document.getElementById('enable_delivery').checked;
            const transferEnabled = document.getElementById('enable_transfer').checked;

            if (!inventoryEnabled && !deliveryEnabled && !transferEnabled) {
                e.preventDefault();
                alert('Please enable at least one issue section before submitting.');
                return false;
            }

            // Check allocations
            let hasAllocations = false;
            for (let itemId in itemData) {
                const total = itemData[itemId].inventory + itemData[itemId].delivery + itemData[itemId].transfer;
                if (total > 0) {
                    hasAllocations = true;
                    break;
                }
            }

            if (!hasAllocations) {
                e.preventDefault();
                alert('Please allocate quantities for at least one item before submitting.');
                return false;
            }

            // Check required fields for inventory section
            if (inventoryEnabled) {
                const receiverName = document.querySelector('input[name="inventory_receiver_name"]');
                if (!receiverName || !receiverName.value.trim()) {
                    e.preventDefault();
                    alert('Please enter a receiver name for inventory dispatch.');
                    return false;
                }
            }

            // Show loading indicator
            const submitBtn = document.getElementById('submitBtn');
            if (submitBtn) {
                submitBtn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px;">hourglass_empty</i> Processing...';
                submitBtn.disabled = true;
            }

            console.log('Form validation passed, submitting...');
            return true;
        });
    }

    // Initialize Select2 event listeners after Select2 is loaded
    setTimeout(function() {
        if (typeof $ !== 'undefined') {
            // Reinitialize Select2 dropdowns with fixed width settings
            $('.select2-dropdown').each(function() {
                const $select = $(this);
                if ($select.data('select2')) {
                    $select.select2('destroy');
                }

                $select.select2({
                    placeholder: $select.attr('placeholder') || 'Please select...',
                    allowClear: true,
                    width: '100%',
                    dropdownAutoWidth: false, // Prevent auto-width
                    theme: 'default'
                });
            });

            // Initialize reference dropdown specifically
            if ($('#transfer_reference_select').length) {
                $('#transfer_reference_select').select2({
                    placeholder: 'Select Reference Number',
                    allowClear: true,
                    width: '100%',
                    dropdownAutoWidth: false,
                    theme: 'default'
                });

                // Add event listener for reference selection
                $('#transfer_reference_select').on('change', function() {
                    updateReferenceItemAvailability();
                    validateForm(); // Also validate form when reference changes
                });
            }

            // Event listeners are handled by the Select2 component via onchange attribute
        }
    }, 500); // Wait for Select2 to initialize

    // Add real-time validation to quantity inputs
    document.querySelectorAll('.inventory-qty, .delivery-qty, .transfer-qty').forEach(input => {
        input.addEventListener('input', function() {
            const itemId = this.dataset.itemId;
            if (itemId) {
                // Small delay to allow updateAllocations to process
                setTimeout(() => updateAllocations(itemId), 100);
            }
        });

        // Also validate on blur (when user leaves the field)
        input.addEventListener('blur', function() {
            const itemId = this.dataset.itemId;
            if (itemId) {
                updateAllocations(itemId);
            }
        });

        // Set max attribute to prevent typing more than remaining quantity
        const itemId = this.dataset.itemId;
        if (itemId && itemData[itemId]) {
            const remainingQty = itemData[itemId].requested;
            this.setAttribute('max', remainingQty);
        }
    });

    // Add real-time validation to required fields
    const requiredFields = [
        'input[name="inventory_receiver_name"]',
        'input[name="inventory_dispatch_date"]',
        'input[name="delivery_note_number"]',
        'select[name="transfer_source_project_id"]',
        'select[name="transfer_reference_id"]',
        'input[name="transfer_date"]'
    ];

    requiredFields.forEach(selector => {
        const field = document.querySelector(selector);
        if (field) {
            field.addEventListener('input', validateForm);
            field.addEventListener('change', validateForm);
        }
    });

    // Initial validation
    validateForm();
});
</script>
@endpush