@extends('layouts.admin-simple')

@section('title', 'Create Material Request')

@section('styles')
<style>
    .pulse-error {
        animation: pulse-red 1s ease-in-out 3;
        border-color: #dc3545 !important;
    }

    @keyframes pulse-red {
        0% {
            box-shadow: 0 0 0 0 rgba(220, 53, 69, 0.7);
        }
        70% {
            box-shadow: 0 0 0 10px rgba(220, 53, 69, 0);
        }
        100% {
            box-shadow: 0 0 0 0 rgba(220, 53, 69, 0);
        }
    }

    .select2-container .select2-selection.pulse-error {
        animation: pulse-red 1s ease-in-out 3;
        border-color: #dc3545 !important;
    }

    /* Force Select2 dropdown to appear below */
    .select2-container--open .select2-dropdown {
        top: 100% !important;
        bottom: auto !important;
        margin-top: 1px !important;
    }

    /* Force dropdown direction to be below */
    .select2-container--open .select2-dropdown--below {
        top: 100% !important;
        border-top: none !important;
        border-top-left-radius: 0 !important;
        border-top-right-radius: 0 !important;
    }

    /* Hide dropdown above option */
    .select2-container--open .select2-dropdown--above {
        display: none !important;
    }

    /* Custom width for item selection dropdowns */
    #items-table .select2-container {
        width: 80% !important;
        max-width: 400px !important;
        min-width: 250px !important;
    }

    /* Custom width for requested by dropdown */
    #requested_by + .select2-container {
        width: 60% !important;
        max-width: 300px !important;
        min-width: 200px !important;
    }

    /* Custom width for project dropdown */
    #project_id + .select2-container {
        width: 70% !important;
        max-width: 450px !important;
        min-width: 280px !important;
    }

    /* General Select2 width fix for form dropdowns */
    .form-group .select2-container {
        width: 60% !important;
        max-width: 300px !important;
        min-width: 200px !important;
    }

    /* Exception for project dropdown to be wider */
    .form-group #project_id + .select2-container {
        width: 70% !important;
        max-width: 450px !important;
        min-width: 280px !important;
    }

    /* Ensure dropdown doesn't get cut off */
    .table-responsive {
        overflow: visible !important;
    }

    #items-table {
        overflow: visible !important;
    }

    /* Specific positioning for item dropdowns in table */
    #items-tbody .select2-container {
        position: relative !important;
    }

    #items-tbody .select2-dropdown {
        position: absolute !important;
        top: 100% !important;
        z-index: 9999 !important;
        width: 100% !important;
    }
</style>
@endsection

@section('content')
<div class="container-fluid px-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1 class="h3 mb-2 text-gray-800">
                        <i class="material-icons align-middle mr-2">assignment</i>
                        Create Material Request
                    </h1>
                    <p class="text-muted mb-0">Request materials from inventory for project use</p>
                </div>
                <div>
                    <a href="{{ route('warehouse.outgoing.index') }}" class="btn btn-secondary">
                        <i class="material-icons mr-1">arrow_back</i>
                        Back to List
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Main Container -->
    <div class="card shadow">
        <div class="card-header">
            <h6 class="m-0 font-weight-bold text-primary">
                <i class="material-icons align-middle mr-1">assignment</i>
                Material Request Form
            </h6>
        </div>
        <div class="card-body">
            <form action="{{ route('warehouse.outgoing.store') }}" method="POST" id="materialRequestForm">
                @csrf

                <!-- Request Information -->
                <div class="row mb-4">
                    <div class="col-12">
                        <h6 class="font-weight-bold text-secondary mb-3">
                            <i class="material-icons align-middle mr-1">info</i>
                            Request Information
                        </h6>
                        <div class="row">
                            <!-- Date -->
                            <div class="col-md-4 mb-3">
                                <label for="request_date" class="form-label">Request Date <span class="text-danger">*</span></label>
                                <input type="date"
                                       class="form-control @error('request_date') is-invalid @enderror"
                                       id="request_date"
                                       name="request_date"
                                       value="{{ old('request_date', date('Y-m-d')) }}"
                                       required>
                                @error('request_date')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Material Request Number -->
                            <div class="col-md-4 mb-3">
                                <label for="material_request_number" class="form-label">Material Request Number <span class="text-danger">*</span></label>
                                <input type="text"
                                       class="form-control @error('material_request_number') is-invalid @enderror"
                                       id="material_request_number"
                                       name="material_request_number"
                                       value="{{ old('material_request_number', 'MR-' . date('Ymd') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT)) }}"
                                       placeholder="Auto-generated request number"
                                       required>
                                @error('material_request_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Requested By -->
                            <div class="col-md-4 mb-3">
                                <x-warehouse.select2-dropdown
                                    name="requested_by"
                                    id="requested_by"
                                    label="Requested By"
                                    placeholder="Select Requester"
                                    :options="isset($users) ? $users->map(function($user) {
                                        return ['value' => $user->id, 'text' => $user->name];
                                    }) : []"
                                    :selected="old('requested_by')"
                                    :required="true"
                                    allowClear="true"
                                />
                            </div>
                        </div>

                        <div class="row">
                            <!-- Project -->
                            <div class="col-md-12 mb-3">
                                <x-warehouse.select2-dropdown
                                    name="project_id"
                                    id="project_id"
                                    label="Project Number & Name"
                                    placeholder="Select Project"
                                    :options="isset($projects) ? $projects->map(function($project) {
                                        return [
                                            'value' => $project->id,
                                            'text' => $project->project_number . ' - ' . $project->project_name,
                                            'data' => [
                                                'division' => $project->division_id ?? '',
                                                'division-name' => $project->division->name ?? ''
                                            ]
                                        ];
                                    }) : []"
                                    :selected="old('project_id')"
                                    :required="true"
                                    allowClear="true"
                                />

                                <!-- Division Display Box -->
                                <div id="division_display" class="mt-2" style="display: none;">
                                    <div class="alert alert-info py-2 mb-0">
                                        <strong>Division:</strong> <span id="division_name_display"></span>
                                    </div>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

                <hr class="my-4">

                <!-- Items Section -->
                <div class="row">
                    <div class="col-12">
                        <h6 class="font-weight-bold text-secondary mb-3">
                            <i class="material-icons align-middle mr-1">shopping_cart</i>
                            Items Selection & Management
                        </h6>

                        <!-- Items Table -->
                        <div class="table-responsive">
                            <table class="table table-bordered" id="items-table">
                                <thead class="thead-light">
                                    <tr>
                                        <th width="50%">Item</th>
                                        <th width="25%">Requested Qty</th>
                                        <th width="20%">Unit</th>
                                        <th width="5%">Action</th>
                                    </tr>
                                </thead>
                                <tbody id="items-tbody">
                                    <tr>
                                        <td>
                                            <select name="items[0][item_id]" id="item_select_0" class="form-control item-select select2-dropdown" onchange="updateItemRow(this, 0)" required>
                                                <option value="">Select Item</option>
                                                @foreach($items as $item)
                                                <option value="{{ $item->id }}"
                                                        data-code="{{ $item->item_code }}"
                                                        data-description="{{ $item->item_description }}"
                                                        data-unit="{{ $item->unit_of_measure }}"
                                                        data-stock="{{ $item->current_stock ?? 0 }}">
                                                    {{ $item->item_code }} - {{ $item->item_description }}
                                                </option>
                                                @endforeach
                                            </select>
                                        </td>
                                        <td>
                                            <input type="number" name="items[0][quantity_requested]" class="form-control" step="0.01" min="0.01" required>
                                        </td>
                                        <td>
                                            <input type="text" name="items[0][unit_of_measure]" class="form-control" readonly>
                                        </td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeItemRow(this)">
                                                <i class="material-icons">delete</i>
                                            </button>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>

                        <!-- Add Row Button -->
                        <div class="mb-3">
                            <button type="button" class="btn btn-outline-primary" onclick="addItemRow()">
                                <i class="material-icons mr-1">add</i>
                                Add Another Item
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Form Actions -->
                <hr class="my-4">
                <div class="row">
                    <div class="col-12">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="text-muted">
                                <i class="material-icons align-middle mr-1">info</i>
                                <strong>Summary:</strong> <span id="total_items_display">0 items</span>
                            </div>
                            <div>
                                <a href="{{ route('warehouse.outgoing.index') }}" class="btn btn-outline-secondary mr-2">
                                    <i class="material-icons mr-1">arrow_back</i>
                                    Cancel
                                </a>
                                <button type="submit" class="btn btn-success" id="submit_request_btn" disabled>
                                    <i class="material-icons mr-1">send</i>
                                    Submit Material Request
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Information Card -->
    <div class="row mt-4">
        <div class="col-12">
            <div class="card border-left-success">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                <i class="material-icons align-middle mr-1">info</i>
                                Material Request Process
                            </div>
                            <div class="text-gray-800">
                                <ul class="mb-0 pl-3">
                                    <li>Material requests are processed directly upon approval</li>
                                    <li>Once approved, inventory will be automatically updated</li>
                                    <li>Requested items will be deducted from available stock immediately</li>
                                    <li>You can track all material requests through the inventory section</li>
                                </ul>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="material-icons fa-2x text-gray-300">check_circle</i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
let rowCounter = 1;

$(document).ready(function() {
    console.log('🚀 Material Request Form: Initializing...');

    // Project selection change handler
    $('#project_id').off('change.materialRequest').on('change.materialRequest', function() {
        const selectedOption = $(this).find('option:selected');
        const divisionDisplay = $('#division_display');
        const divisionNameDisplay = $('#division_name_display');

        if (selectedOption.val() && selectedOption.data('division-name')) {
            divisionNameDisplay.text(selectedOption.data('division-name'));
            divisionDisplay.show();
        } else {
            divisionDisplay.hide();
        }
    });

    // Form submission validation
    $('#materialRequestForm').off('submit.materialRequest').on('submit.materialRequest', function(e) {
        console.log('📋 Form submission triggered');

        // Validate required fields
        let isValid = true;
        let missingFields = [];

        // Check request date
        if (!$('#request_date').val()) {
            isValid = false;
            missingFields.push('Request Date');
            $('#request_date').addClass('is-invalid');
        } else {
            $('#request_date').removeClass('is-invalid');
        }

        // Check material request number
        if (!$('#material_request_number').val()) {
            isValid = false;
            missingFields.push('Material Request Number');
            $('#material_request_number').addClass('is-invalid');
        } else {
            $('#material_request_number').removeClass('is-invalid');
        }

        // Check project selection
        if (!$('#project_id').val()) {
            isValid = false;
            missingFields.push('Project');
            $('#project_id').next('.select2-container').find('.select2-selection').addClass('is-invalid');
        } else {
            $('#project_id').next('.select2-container').find('.select2-selection').removeClass('is-invalid');
        }

        // Check requested by
        if (!$('#requested_by').val()) {
            isValid = false;
            missingFields.push('Requested By');
            $('#requested_by').next('.select2-container').find('.select2-selection').addClass('is-invalid');
        } else {
            $('#requested_by').next('.select2-container').find('.select2-selection').removeClass('is-invalid');
        }

        // Check if there are items
        const hasItems = $('#items-tbody tr').length > 0;
        if (!hasItems) {
            isValid = false;
            missingFields.push('Items');
        } else {
            // Check for duplicate items
            const selectedItems = [];
            let hasDuplicates = false;

            $('#items-tbody tr').each(function(index) {
                const itemSelect = $(this).find('select[name*="item_id"]');
                const itemId = itemSelect.val();

                if (itemId && selectedItems.includes(itemId)) {
                    hasDuplicates = true;
                    isValid = false;
                    itemSelect.addClass('pulse-error');
                    setTimeout(() => {
                        itemSelect.removeClass('pulse-error');
                    }, 3000);
                } else if (itemId) {
                    selectedItems.push(itemId);
                }
            });

            if (hasDuplicates) {
                missingFields.push('Duplicate items detected');
            }

            // Validate each item row
            $('#items-tbody tr').each(function(index) {
                const itemSelect = $(this).find('select[name*="item_id"]');
                const quantityInput = $(this).find('input[name*="quantity_requested"]');

                if (!itemSelect.val()) {
                    isValid = false;
                    missingFields.push(`Item selection in row ${index + 1}`);
                    itemSelect.addClass('is-invalid');
                    itemSelect.next('.select2-container').find('.select2-selection').addClass('is-invalid');
                } else {
                    itemSelect.removeClass('is-invalid');
                    itemSelect.next('.select2-container').find('.select2-selection').removeClass('is-invalid');
                }

                if (!quantityInput.val() || quantityInput.val() <= 0) {
                    isValid = false;
                    missingFields.push(`Quantity in row ${index + 1}`);
                    quantityInput.addClass('is-invalid');
                } else {
                    quantityInput.removeClass('is-invalid');
                }
            });
        }

        if (!isValid) {
            e.preventDefault();
            alert('Please fill in the following required fields:\n\n• ' + missingFields.join('\n• '));

            // Focus on the first missing field and scroll to it
            setTimeout(function() {
                if (!$('#request_date').val()) {
                    $('#request_date').addClass('pulse-error').focus();
                    $('html, body').animate({scrollTop: $('#request_date').offset().top - 100}, 500);
                    setTimeout(() => $('#request_date').removeClass('pulse-error'), 3000);
                } else if (!$('#material_request_number').val()) {
                    $('#material_request_number').addClass('pulse-error').focus();
                    $('html, body').animate({scrollTop: $('#material_request_number').offset().top - 100}, 500);
                    setTimeout(() => $('#material_request_number').removeClass('pulse-error'), 3000);
                } else if (!$('#project_id').val()) {
                    $('#project_id').next('.select2-container').find('.select2-selection').addClass('pulse-error');
                    $('#project_id').select2('open');
                    $('html, body').animate({scrollTop: $('#project_id').offset().top - 100}, 500);
                    setTimeout(() => $('#project_id').next('.select2-container').find('.select2-selection').removeClass('pulse-error'), 3000);
                } else if (!$('#requested_by').val()) {
                    $('#requested_by').next('.select2-container').find('.select2-selection').addClass('pulse-error');
                    $('#requested_by').select2('open');
                    $('html, body').animate({scrollTop: $('#requested_by').offset().top - 100}, 500);
                    setTimeout(() => $('#requested_by').next('.select2-container').find('.select2-selection').removeClass('pulse-error'), 3000);
                } else {
                    // Focus on first invalid item field
                    const firstInvalidItem = $('#items-tbody tr').first().find('select[name*="item_id"]');
                    if (firstInvalidItem.length) {
                        firstInvalidItem.next('.select2-container').find('.select2-selection').addClass('pulse-error');
                        firstInvalidItem.select2('open');
                        $('html, body').animate({scrollTop: firstInvalidItem.offset().top - 100}, 500);
                        setTimeout(() => firstInvalidItem.next('.select2-container').find('.select2-selection').removeClass('pulse-error'), 3000);
                    }
                }
            }, 100);

            return false;
        }

        console.log('✅ Form validation passed, submitting...');
    });

    // Initialize Select2 for the first item row
    initializeSelect2ForRow(0);

    // Enable submit button initially since we have one row
    updateSubmitButton();

    // Add click handler to submit button for debugging
    $('#submit_request_btn').off('click.debug').on('click.debug', function(e) {
        console.log('🖱️ Submit button clicked');
        const isDisabled = $(this).prop('disabled');
        if (isDisabled) {
            console.log('⚠️ Button is disabled');
            e.preventDefault();
            return false;
        }
        console.log('✅ Button click proceeding');
    });

    // Add event listeners to clear validation errors when user interacts with fields
    $('#request_date').on('change', function() {
        $(this).removeClass('is-invalid');
    });

    $('#material_request_number').on('input', function() {
        if ($(this).val()) {
            $(this).removeClass('is-invalid');
        }
    });

    $('#project_id').on('select2:select', function() {
        $(this).next('.select2-container').find('.select2-selection').removeClass('is-invalid');
    });

    $('#requested_by').on('select2:select', function() {
        $(this).next('.select2-container').find('.select2-selection').removeClass('is-invalid');
    });

    // Add event listener for quantity inputs (using event delegation)
    $(document).on('input', 'input[name*="quantity_requested"]', function() {
        if ($(this).val() && $(this).val() > 0) {
            $(this).removeClass('is-invalid');
        }
    });

    // Add event listener for item select changes (using event delegation)
    $(document).on('select2:select', 'select[name*="item_id"]', function() {
        $(this).removeClass('is-invalid');
        $(this).next('.select2-container').find('.select2-selection').removeClass('is-invalid');
    });

    console.log('✅ Material Request Form: Initialization complete');
});


// Update submit button state
function updateSubmitButton() {
    const hasRows = $('#items-tbody tr').length > 0;
    const submitBtn = $('#submit_request_btn');

    console.log('🔘 Updating submit button - Rows:', hasRows, 'Button found:', submitBtn.length > 0);

    if (hasRows) {
        submitBtn.prop('disabled', false);
        console.log('✅ Submit button enabled');
    } else {
        submitBtn.prop('disabled', true);
        console.log('❌ Submit button disabled');
    }
}

// Update item row when item is selected
function updateItemRow(selectElement, rowIndex) {
    const selectedOption = selectElement.options[selectElement.selectedIndex];
    const selectedItemId = selectedOption ? selectedOption.value : selectElement.value;

    if (selectedItemId) {
        // Check for duplicate items
        const allItemSelects = document.querySelectorAll('.item-select');
        let isDuplicate = false;

        allItemSelects.forEach(function(select) {
            if (select !== selectElement && select.value === selectedItemId) {
                isDuplicate = true;
            }
        });

        if (isDuplicate) {
            alert('This item has already been added to the request. Please select a different item.');
            $(selectElement).val('').trigger('change');
            const unitInput = selectElement.closest('tr').querySelector('input[name*="unit_of_measure"]');
            if (unitInput) {
                unitInput.value = '';
            }
            $(selectElement).next('.select2-container').find('.select2-selection').addClass('pulse-error');
            setTimeout(() => {
                $(selectElement).next('.select2-container').find('.select2-selection').removeClass('pulse-error');
            }, 3000);
            return;
        }

        const unit = selectedOption ? (selectedOption.dataset.unit || '') : '';

        // Update unit
        const unitInput = selectElement.closest('tr').querySelector('input[name*="unit_of_measure"]');
        if (unitInput) {
            unitInput.value = unit;
        }

        // Remove error styling if item is selected
        $(selectElement).removeClass('is-invalid');
        $(selectElement).next('.select2-container').find('.select2-selection').removeClass('is-invalid');
    } else {
        // Clear fields when no item selected
        const unitInput = selectElement.closest('tr').querySelector('input[name*="unit_of_measure"]');
        if (unitInput) {
            unitInput.value = '';
        }
    }
}


// Add a new item row
function addItemRow() {
    const tbody = document.getElementById('items-tbody');
    const newRow = document.createElement('tr');

    newRow.innerHTML = `
        <td>
            <select name="items[${rowCounter}][item_id]" id="item_select_${rowCounter}" class="form-control item-select select2-dropdown" onchange="updateItemRow(this, ${rowCounter})" required>
                <option value="">Select Item</option>
                @foreach($items as $item)
                <option value="{{ $item->id }}"
                        data-code="{{ $item->item_code }}"
                        data-description="{{ $item->item_description }}"
                        data-unit="{{ $item->unit_of_measure }}"
                        data-stock="{{ $item->current_stock ?? 0 }}">
                    {{ $item->item_code }} - {{ $item->item_description }}
                </option>
                @endforeach
            </select>
        </td>
        <td>
            <input type="number" name="items[${rowCounter}][quantity_requested]" class="form-control" step="0.01" min="0.01" required>
        </td>
        <td>
            <input type="text" name="items[${rowCounter}][unit_of_measure]" class="form-control" readonly>
        </td>
        <td>
            <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeItemRow(this)">
                <i class="material-icons">delete</i>
            </button>
        </td>
    `;

    tbody.appendChild(newRow);

    // Initialize Select2 for the new dropdown
    initializeSelect2ForRow(rowCounter);

    rowCounter++;

    // Update submit button state
    updateSubmitButton();
}

// Initialize Select2 for a specific row
function initializeSelect2ForRow(rowIndex) {
    const selectId = `#item_select_${rowIndex}`;

    // Initialize Select2
    $(selectId).select2({
        placeholder: 'Select Item',
        allowClear: true,
        theme: 'default',
        width: '80%',
        dropdownAutoWidth: false,
        language: {
            noResults: function() {
                return 'No results found';
            },
            searching: function() {
                return 'Searching...';
            }
        },
        dropdownCssClass: 'select2-dropdown-below',
        adaptDropdownCssClass: function(cssClass) {
            return 'select2-dropdown-below';
        }
    });

    // Force dropdown to open below
    $(selectId).on('select2:open', function() {
        const dropdown = $('.select2-dropdown');
        dropdown.removeClass('select2-dropdown--above').addClass('select2-dropdown--below');
        dropdown.css({
            'top': '100%',
            'bottom': 'auto'
        });
    });

    // Prevent Select2 events from interfering with Bootstrap dropdowns
    $(selectId).on('select2:open select2:close', function(e) {
        e.stopPropagation();
    });

    // Handle dropdown clicks to prevent Bootstrap dropdown interference
    $(selectId).next('.select2-container').on('click', function(e) {
        e.stopPropagation();
    });
}

// Remove an item row
function removeItemRow(button) {
    const tbody = document.getElementById('items-tbody');
    const rowToRemove = button.closest('tr');

    // Don't allow removing the last row
    if (tbody.children.length > 1) {
        // Find and destroy Select2 instance before removing the row
        const selectElement = rowToRemove.querySelector('select.select2-dropdown');
        if (selectElement && $(selectElement).hasClass('select2-hidden-accessible')) {
            $(selectElement).select2('destroy');
        }
        rowToRemove.remove();
    } else {
        alert('At least one item row is required');
    }

    // Update submit button state
    updateSubmitButton();
}
</script>
@endpush