<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Warehouse\DashboardController;
use App\Http\Controllers\Warehouse\ItemController;
use App\Http\Controllers\Warehouse\InventoryController;
use App\Http\Controllers\Warehouse\ItemCategoryController;
use App\Http\Controllers\Warehouse\IncomingOperationController;
use App\Http\Controllers\Warehouse\OutgoingTransactionController;
use App\Http\Controllers\Warehouse\ReturnTransactionController;
use App\Http\Controllers\Warehouse\ProjectController;
use App\Http\Controllers\Warehouse\ProjectDivisionController;
use App\Http\Controllers\Warehouse\SupplierController;
use App\Http\Controllers\Warehouse\LocationController;
use App\Http\Controllers\Warehouse\ReportController;
use App\Http\Controllers\Warehouse\AnalyticsController;
use App\Http\Controllers\Warehouse\SettingController;
use App\Http\Controllers\Warehouse\UserController;
use App\Http\Controllers\Warehouse\ActivityLogController;
use App\Http\Controllers\Warehouse\MaterialTransferRequestController;
use App\Http\Controllers\Warehouse\DirectDeliveryController;
use App\Http\Controllers\Warehouse\MaterialTransferIssueController;
use App\Http\Controllers\Warehouse\MaterialIssueController;

/*
|--------------------------------------------------------------------------
| Warehouse Routes
|--------------------------------------------------------------------------
|
| Here are the routes for the warehouse management system. All routes
| are protected by the warehouse.access middleware and use role-based
| permissions for fine-grained access control.
|
*/

Route::prefix('warehouse')->name('warehouse.')->middleware(['auth', 'warehouse.access'])->group(function () {

    // Dashboard
    Route::get('/', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/api/quick-stats', [DashboardController::class, 'quickStats'])->name('api.quick-stats');

    // Items Management
    Route::prefix('items')->name('items.')->group(function () {
        Route::get('/', [ItemController::class, 'index'])->name('index');
        Route::get('/create', [ItemController::class, 'create'])->name('create');
        Route::post('/', [ItemController::class, 'store'])->name('store');
        Route::get('/units-dropdown', [ItemController::class, 'getUnitsDropdown'])->name('units-dropdown');
        Route::get('/dropdown', [ItemController::class, 'getItemsDropdown'])->name('dropdown');

        // Import functionality (specific routes BEFORE parameterized routes)
        Route::get('/import-template', [ItemController::class, 'downloadTemplate'])->name('template');
        Route::post('/import', [ItemController::class, 'import'])->name('import');
        Route::get('/export', [ItemController::class, 'export'])->name('export');

        // Item specific actions (specific routes)
        Route::post('/bulk-import', [ItemController::class, 'bulkImport'])->name('bulk-import');
        Route::get('/low-stock/list', [ItemController::class, 'lowStock'])->name('low-stock');

        // Parameterized routes (AFTER specific routes)
        Route::get('/{id}', [ItemController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ItemController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ItemController::class, 'update'])->name('update');
        Route::delete('/{id}', [ItemController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/generate-barcode', [ItemController::class, 'generateBarcode'])->name('generate-barcode');
        Route::post('/{id}/generate-qr-code', [ItemController::class, 'generateQrCode'])->name('generate-qr-code');

        // Categories
        Route::prefix('categories')->name('categories.')->group(function () {
            Route::get('/', [ItemCategoryController::class, 'index'])->name('index');
            Route::get('/create', [ItemCategoryController::class, 'create'])->name('create');
            Route::post('/', [ItemCategoryController::class, 'store'])->name('store');
            Route::get('/{id}', [ItemCategoryController::class, 'show'])->name('show');
            Route::get('/{id}/edit', [ItemCategoryController::class, 'edit'])->name('edit');
            Route::put('/{id}', [ItemCategoryController::class, 'update'])->name('update');
            Route::delete('/{id}', [ItemCategoryController::class, 'destroy'])->name('destroy');
        });
    });

    // Item Categories (standalone)
    Route::prefix('item-categories')->name('item-categories.')->group(function () {
        Route::get('/', [ItemCategoryController::class, 'index'])->name('index');
        Route::get('/create', [ItemCategoryController::class, 'create'])->name('create');
        Route::post('/', [ItemCategoryController::class, 'store'])->name('store');

        // Import functionality (specific routes BEFORE parameterized routes)
        Route::get('/import-template', [ItemCategoryController::class, 'downloadTemplate'])->name('template');
        Route::post('/import', [ItemCategoryController::class, 'import'])->name('import');
        Route::get('/export', [ItemCategoryController::class, 'export'])->name('export');

        // Parameterized routes (AFTER specific routes)
        Route::get('/{id}', [ItemCategoryController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ItemCategoryController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ItemCategoryController::class, 'update'])->name('update');
        Route::delete('/{id}', [ItemCategoryController::class, 'destroy'])->name('destroy');
    });

    // Item Types
    Route::prefix('item-types')->name('item-types.')->group(function () {
        Route::get('/', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'index'])->name('index');
        Route::get('/create', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'create'])->name('create');
        Route::post('/', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'store'])->name('store');
        Route::get('/dropdown', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'dropdown'])->name('dropdown');
        Route::get('/{id}', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'edit'])->name('edit');
        Route::put('/{id}', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'update'])->name('update');
        Route::delete('/{id}', [App\Http\Controllers\Warehouse\ItemTypeController::class, 'destroy'])->name('destroy');
    });

    // Inventory Management
    Route::prefix('inventory')->name('inventory.')->group(function () {
        Route::get('/', [InventoryController::class, 'index'])->name('index');
        Route::get('/location/{location}', [InventoryController::class, 'byLocation'])->name('by-location');
        Route::get('/low-stock', [InventoryController::class, 'lowStock'])->name('low-stock');
        Route::get('/aging-report', [InventoryController::class, 'agingReport'])->name('aging-report');
        Route::get('/valuation', [InventoryController::class, 'valuation'])->name('valuation');

        // Inventory adjustments
        Route::get('/adjustments', [InventoryController::class, 'adjustments'])->name('adjustments');
        Route::post('/adjustments', [InventoryController::class, 'storeAdjustment'])->name('store-adjustment');
        Route::get('/adjustments/{id}/details', [InventoryController::class, 'getAdjustmentDetails'])->name('adjustment-details');
        Route::delete('/adjustments/clear', [InventoryController::class, 'clearAdjustments'])->name('clear-adjustments');
        Route::get('/{itemId}/adjust', [InventoryController::class, 'adjust'])->name('adjust');
        Route::post('/{itemId}/adjust', [InventoryController::class, 'performAdjustment'])->name('perform-adjustment');

        // Inventory transfers
        Route::get('/transfers', [InventoryController::class, 'transfers'])->name('transfers');
        Route::get('/{itemId}/transfer', [InventoryController::class, 'transfer'])->name('transfer');
        Route::post('/{itemId}/transfer', [InventoryController::class, 'performTransfer'])->name('perform-transfer');

        // Inventory reservations
        Route::post('/{itemId}/reserve', [InventoryController::class, 'reserve'])->name('reserve');
        Route::post('/{itemId}/release-reserved', [InventoryController::class, 'releaseReserved'])->name('release-reserved');

        // Physical count
        Route::get('/physical-count', [InventoryController::class, 'physicalCount'])->name('physical-count');
        Route::post('/physical-count', [InventoryController::class, 'performPhysicalCount'])->name('perform-physical-count');

        // Stock movements
        Route::get('/{itemId}/stock-movements', [InventoryController::class, 'stockMovements'])->name('stock-movements');

        // Material Requests
        Route::get('/material-requests', [InventoryController::class, 'materialRequests'])->name('material-requests');
        Route::get('/material-requests/{id}', [InventoryController::class, 'viewMaterialRequest'])->name('material-requests.view');
        Route::get('/material-requests/{id}/dispatch', [InventoryController::class, 'dispatchMaterialRequest'])->name('material-requests.dispatch');
        Route::get('/material-requests/{id}/print', [InventoryController::class, 'printMaterialRequest'])->name('material-requests.print');
        Route::delete('/material-requests/{id}', [InventoryController::class, 'deleteMaterialRequest'])->name('material-requests.delete');
        Route::post('/material-requests/{id}/dispatch', [InventoryController::class, 'processDispatch'])->name('material-requests.process-dispatch');

        // Material Issue Decision Center (replaces old material-issued)
        Route::get('/material-issued', [MaterialIssueController::class, 'index'])->name('material-issued');
        Route::get('/material-issued/{id}/print', [InventoryController::class, 'printMaterialIssued'])->name('material-issued.print');

        // Bulk operations
        Route::post('/check-bulk-availability', [InventoryController::class, 'checkBulkAvailability'])->name('check-bulk-availability');
    });

    // Material Issue Decision Center (separate prefix)
    Route::prefix('material-issue')->name('material-issue.')->group(function () {
        Route::get('/', [MaterialIssueController::class, 'index'])->name('index');
        Route::get('/{transaction}', [MaterialIssueController::class, 'show'])->name('show');
        Route::post('/{transaction}/process', [MaterialIssueController::class, 'processDecision'])->name('process-decision');
        Route::post('/{transaction}/process-multi', [MaterialIssueController::class, 'processMultiSource'])->name('process-multi');
    });

    // Enhanced Incoming Operations
    Route::prefix('incoming-operations')->name('incoming-operations.')->group(function () {
        Route::get('/', [IncomingOperationController::class, 'index'])->name('index');
        Route::get('/create', [IncomingOperationController::class, 'create'])->name('create');
        Route::post('/', [IncomingOperationController::class, 'store'])->name('store');
        Route::get('/{operation}', [IncomingOperationController::class, 'show'])->name('show');
        Route::get('/{operation}/edit', [IncomingOperationController::class, 'edit'])->name('edit');
        Route::put('/{operation}', [IncomingOperationController::class, 'update'])->name('update');
        Route::delete('/{operation}', [IncomingOperationController::class, 'destroy'])->name('destroy');

        // Quality Control & Inspection
        Route::post('/{operation}/inspect', [IncomingOperationController::class, 'inspect'])->name('inspect');
        Route::post('/{operation}/complete', [IncomingOperationController::class, 'complete'])->name('complete');

        // Printing & Documents
        Route::get('/{operation}/print', [IncomingOperationController::class, 'print'])->name('print');

        // API routes
        Route::get('/api/items', [IncomingOperationController::class, 'getItems'])->name('api.items');
        Route::get('/api/lpo-details', [IncomingOperationController::class, 'getLpoDetails'])->name('api.lpo-details');
        Route::get('/api/quality-checklists', [IncomingOperationController::class, 'getQualityChecklists'])->name('api.quality-checklists');
    });

    // Supplier Delivery (Dedicated Page)
    Route::prefix('supplier-delivery')->name('supplier-delivery.')->group(function () {
        Route::get('/', [IncomingOperationController::class, 'supplierDeliveryIndex'])->name('index');
        Route::get('/create', [IncomingOperationController::class, 'supplierDeliveryCreate'])->name('create');
        Route::get('/api/data', [IncomingOperationController::class, 'supplierDeliveryData'])->name('api.data');
        Route::get('/export', [IncomingOperationController::class, 'exportSupplierDeliveries'])->name('export');
    });

    // Site Return (Dedicated Page)
    Route::prefix('site-return')->name('site-return.')->group(function () {
        Route::get('/', [IncomingOperationController::class, 'siteReturnIndex'])->name('index');
        Route::get('/create', [IncomingOperationController::class, 'siteReturnCreate'])->name('create');
        Route::get('/project/{projectId}', [IncomingOperationController::class, 'showProjectReturns'])->name('project-returns');
        Route::get('/get-material-request/{id}', [IncomingOperationController::class, 'getMaterialRequest'])->name('get-material-request');
        Route::get('/get-direct-delivery/{id}', [IncomingOperationController::class, 'getDirectDelivery'])->name('get-direct-delivery');
        Route::get('/get-material-transfer/{id}', [IncomingOperationController::class, 'getMaterialTransfer'])->name('get-material-transfer');
        Route::get('/get-material-requests-by-project/{projectId}', [IncomingOperationController::class, 'getMaterialRequestsByProject'])->name('get-material-requests-by-project');
        Route::get('/get-direct-deliveries-by-project/{projectId}', [IncomingOperationController::class, 'getDirectDeliveriesByProject'])->name('get-direct-deliveries-by-project');
        Route::get('/get-combined-references-by-project/{projectId}', [IncomingOperationController::class, 'getCombinedReferencesByProject'])->name('get-combined-references-by-project');
        Route::get('/mr-returns/{materialRequestId}', [IncomingOperationController::class, 'showMaterialRequestReturns'])->name('show-mr-returns');
        Route::get('/mr-returns/{materialRequestId}/print', [IncomingOperationController::class, 'printMaterialRequestReturns'])->name('print-mr-returns');
        Route::get('/{returnId}/print', [IncomingOperationController::class, 'printSiteReturn'])->name('print-site-return');
        Route::get('/project/{projectId}/print-consolidated', [IncomingOperationController::class, 'printProjectConsolidatedReturns'])->name('print-project-consolidated');
        Route::get('/api/data', [IncomingOperationController::class, 'siteReturnData'])->name('api.data');
        Route::get('/export', [IncomingOperationController::class, 'exportSiteReturns'])->name('export');
    });

    // Outgoing Transactions
    Route::prefix('outgoing')->name('outgoing.')->group(function () {
        Route::get('/', [OutgoingTransactionController::class, 'index'])->name('index');
        Route::get('/create', [OutgoingTransactionController::class, 'create'])->name('create');
        Route::post('/', [OutgoingTransactionController::class, 'store'])->name('store');
        Route::get('/{id}', [OutgoingTransactionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [OutgoingTransactionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [OutgoingTransactionController::class, 'update'])->name('update');
        Route::delete('/{id}', [OutgoingTransactionController::class, 'destroy'])->name('destroy');

        Route::get('/pending-approval/list', [OutgoingTransactionController::class, 'pendingApproval'])->name('pending-approval');
        Route::post('/{id}/approve', [OutgoingTransactionController::class, 'approve'])->name('approve');
        Route::post('/{id}/dispatch', [OutgoingTransactionController::class, 'dispatch'])->name('dispatch');
    });

    // Return Transactions
    Route::prefix('returns')->name('returns.')->group(function () {
        Route::get('/', [ReturnTransactionController::class, 'index'])->name('index');
        Route::get('/create', [ReturnTransactionController::class, 'create'])->name('create');
        Route::post('/', [ReturnTransactionController::class, 'store'])->name('store');
        Route::get('/{id}', [ReturnTransactionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ReturnTransactionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ReturnTransactionController::class, 'update'])->name('update');
        Route::delete('/{id}', [ReturnTransactionController::class, 'destroy'])->name('destroy');

        Route::post('/{id}/process', [ReturnTransactionController::class, 'process'])->name('process');
    });

    // Projects
    Route::prefix('projects')->name('projects.')->group(function () {
        Route::get('/', [ProjectController::class, 'index'])->name('index');
        Route::get('/create', [ProjectController::class, 'create'])->name('create');
        Route::post('/', [ProjectController::class, 'store'])->name('store');
        Route::get('/{id}', [ProjectController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ProjectController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ProjectController::class, 'update'])->name('update');
        Route::delete('/{id}', [ProjectController::class, 'destroy'])->name('destroy');
        Route::post('/sync-external', [ProjectController::class, 'syncExternal'])->name('sync-external');
        Route::get('/inspect-external', [ProjectController::class, 'inspectExternal'])->name('inspect-external');
        Route::get('/sync-statistics', [ProjectController::class, 'getSyncStatistics'])->name('sync-statistics');
        Route::post('/backfill-external-ids', [ProjectController::class, 'backfillExternalIds'])->name('backfill-external-ids');
    });

    // Project Divisions
    Route::prefix('project-divisions')->name('project-divisions.')->group(function () {
        Route::get('/', [ProjectDivisionController::class, 'index'])->name('index');
        Route::get('/create', [ProjectDivisionController::class, 'create'])->name('create');
        Route::post('/', [ProjectDivisionController::class, 'store'])->name('store');
        Route::get('/{id}', [ProjectDivisionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ProjectDivisionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ProjectDivisionController::class, 'update'])->name('update');
        Route::delete('/{id}', [ProjectDivisionController::class, 'destroy'])->name('destroy');
    });

    // Suppliers
    Route::prefix('suppliers')->name('suppliers.')->group(function () {
        Route::get('/', [SupplierController::class, 'index'])->name('index');
        Route::get('/data', [SupplierController::class, 'data'])->name('data');
        Route::get('/create', [SupplierController::class, 'create'])->name('create');
        Route::post('/', [SupplierController::class, 'store'])->name('store');

        // Additional supplier functionality (specific routes BEFORE parameterized routes)
        Route::get('/import-template', [SupplierController::class, 'downloadTemplate'])->name('template');
        Route::post('/import', [SupplierController::class, 'import'])->name('import');
        Route::get('/export', [SupplierController::class, 'export'])->name('export');
        Route::get('/filter-options', [SupplierController::class, 'filterOptions'])->name('filterOptions');
        Route::get('/dropdown', [SupplierController::class, 'dropdown'])->name('dropdown');

        // Parameterized routes (AFTER specific routes)
        Route::get('/{id}', [SupplierController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [SupplierController::class, 'edit'])->name('edit');
        Route::put('/{id}', [SupplierController::class, 'update'])->name('update');
        Route::delete('/{id}', [SupplierController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/update-rating', [SupplierController::class, 'updateRating'])->name('updateRating');
        Route::get('/{id}/performance', [SupplierController::class, 'performance'])->name('performance');
    });

    // Locations
    Route::prefix('locations')->name('locations.')->group(function () {
        Route::get('/', [LocationController::class, 'index'])->name('index');
        Route::get('/create', [LocationController::class, 'create'])->name('create');
        Route::post('/', [LocationController::class, 'store'])->name('store');
        Route::get('/{id}', [LocationController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [LocationController::class, 'edit'])->name('edit');
        Route::put('/{id}', [LocationController::class, 'update'])->name('update');
        Route::delete('/{id}', [LocationController::class, 'destroy'])->name('destroy');
    });

    // Reports
    Route::prefix('reports')->name('reports.')->group(function () {
        Route::get('/', [ReportController::class, 'index'])->name('index');
        Route::get('/inventory', [ReportController::class, 'inventory'])->name('inventory');
        Route::get('/inventory/item/{itemId}', [ReportController::class, 'inventoryItemDetails'])->name('inventory.item-details');
        Route::get('/incoming', [ReportController::class, 'incoming'])->name('incoming');
        Route::get('/incoming/item/{itemId}', [ReportController::class, 'incomingItemDetails'])->name('incoming.item-details');
        Route::get('/outgoing', [ReportController::class, 'outgoing'])->name('outgoing');
        Route::get('/outgoing/item/{itemId}', [ReportController::class, 'outgoingItemDetails'])->name('outgoing.item-details');
        Route::get('/outgoing/request/{requestId}', [ReportController::class, 'outgoingRequestDetails'])->name('outgoing.request-details');
        Route::get('/returns', [ReportController::class, 'returns'])->name('returns');
        Route::get('/returns/item/{itemId}', [ReportController::class, 'returnItemDetails'])->name('returns.item-details');
        Route::get('/stock-movements', [ReportController::class, 'stockMovements'])->name('stock-movements');
        Route::get('/transactions', [ReportController::class, 'transactions'])->name('transactions');
        Route::get('/aging', [ReportController::class, 'aging'])->name('aging');
        Route::get('/low-stock', [ReportController::class, 'lowStock'])->name('low-stock');
        Route::get('/supplier-performance', [ReportController::class, 'supplierPerformance'])->name('supplier-performance');
        Route::get('/project-transfers', [ReportController::class, 'projectTransfers'])->name('project-transfers');
        Route::get('/direct-deliveries', [ReportController::class, 'directDeliveries'])->name('direct-deliveries');
        Route::get('/stock-on-project', [ReportController::class, 'stockOnProject'])->name('stock-on-project');
        Route::get('/stock-on-project/{projectId}/details', [ReportController::class, 'stockOnProjectDetails'])->name('stock-on-project.details');
        Route::get('/stock-on-project/{projectId}/item/{itemId}/allocations', [ReportController::class, 'itemReferenceAllocations'])->name('item-reference-allocations');
    });

    // Analytics
    Route::prefix('analytics')->name('analytics.')->group(function () {
        Route::get('/', [AnalyticsController::class, 'index'])->name('index');
        Route::get('/inventory-turnover', [AnalyticsController::class, 'inventoryTurnover'])->name('inventory-turnover');
        Route::get('/stock-forecast', [AnalyticsController::class, 'stockForecast'])->name('stock-forecast');
        Route::get('/performance-metrics', [AnalyticsController::class, 'performanceMetrics'])->name('performance-metrics');
    });

    // Settings
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/', [SettingController::class, 'index'])->name('index');
        Route::put('/general', [SettingController::class, 'updateGeneral'])->name('update-general');
        Route::put('/notifications', [SettingController::class, 'updateNotifications'])->name('update-notifications');
        Route::put('/workflows', [SettingController::class, 'updateWorkflows'])->name('update-workflows');
    });

    // User Management
    Route::prefix('users')->name('users.')->group(function () {
        Route::get('/', [UserController::class, 'index'])->name('index');
        Route::get('/stats', [UserController::class, 'stats'])->name('stats');
        Route::post('/', [UserController::class, 'store'])->name('store');
        Route::get('/{user}', [UserController::class, 'show'])->name('show');
        Route::put('/{user}', [UserController::class, 'update'])->name('update');
        Route::delete('/{user}', [UserController::class, 'destroy'])->name('destroy');
        Route::put('/{user}/role', [UserController::class, 'updateRole'])->name('update-role');
    });

    // Material Transfer Requests
    Route::prefix('transfer-requests')->name('transfer-requests.')->group(function () {
        Route::get('/', [MaterialTransferRequestController::class, 'index'])->name('index');
        Route::get('/create', [MaterialTransferRequestController::class, 'create'])->name('create');
        Route::post('/', [MaterialTransferRequestController::class, 'store'])->name('store');
        Route::get('/{transferRequest}', [MaterialTransferRequestController::class, 'show'])->name('show');
        Route::get('/{transferRequest}/edit', [MaterialTransferRequestController::class, 'edit'])->name('edit');
        Route::put('/{transferRequest}', [MaterialTransferRequestController::class, 'update'])->name('update');
        Route::get('/{transferRequest}/approve', [MaterialTransferRequestController::class, 'showApproval'])->name('approve');
        Route::get('/{transferRequest}/print', [MaterialTransferRequestController::class, 'print'])->name('print');

        // Workflow actions
        Route::post('/{transferRequest}/submit', [MaterialTransferRequestController::class, 'submit'])->name('submit');
        Route::post('/{transferRequest}/approve-transferrer', [MaterialTransferRequestController::class, 'approveByTransferrer'])->name('approve-transferrer');
        Route::post('/{transferRequest}/reject', [MaterialTransferRequestController::class, 'reject'])->name('reject');
        Route::post('/{transferRequest}/cancel', [MaterialTransferRequestController::class, 'cancel'])->name('cancel');

        // AJAX endpoints
        Route::get('/project/{project}/inventory', [MaterialTransferRequestController::class, 'getProjectInventory'])->name('project-inventory');
        Route::get('/direct-delivery/{directDelivery}/items', [MaterialTransferRequestController::class, 'getDirectDeliveryItems'])->name('get-direct-delivery-items');
        Route::get('/project/{project}/combined-references', [MaterialTransferRequestController::class, 'getCombinedReferencesByProject'])->name('get-combined-references-by-project-transfer');
        Route::get('/material-transfer/{materialTransfer}', [MaterialTransferRequestController::class, 'getMaterialTransfer'])->name('get-material-transfer');
        Route::get('/material-request/{materialRequest}/items', [MaterialTransferRequestController::class, 'getMaterialRequestItems'])->name('get-material-request-items');
    });

    // Material Transfer Issues
    Route::prefix('transfer-issues')->name('transfer-issues.')->group(function () {
        Route::get('/', [MaterialTransferIssueController::class, 'index'])->name('index');
        Route::get('/create/{transferRequest}', [MaterialTransferIssueController::class, 'create'])->name('create');
        Route::post('/{transferRequest}', [MaterialTransferIssueController::class, 'store'])->name('store');
        Route::get('/{transferIssue}', [MaterialTransferIssueController::class, 'show'])->name('show');

        // Workflow actions
        Route::post('/{transferIssue}/dispatch', [MaterialTransferIssueController::class, 'dispatch'])->name('dispatch');
        Route::post('/{transferIssue}/mark-in-transit', [MaterialTransferIssueController::class, 'markInTransit'])->name('mark-in-transit');
        Route::post('/{transferIssue}/receive', [MaterialTransferIssueController::class, 'receive'])->name('receive');
        Route::post('/{transferIssue}/complete-inspection', [MaterialTransferIssueController::class, 'completeInspection'])->name('complete-inspection');
        Route::post('/{transferIssue}/upload-documents', [MaterialTransferIssueController::class, 'uploadDocuments'])->name('upload-documents');
        Route::post('/{transferIssue}/cancel', [MaterialTransferIssueController::class, 'cancel'])->name('cancel');

        // Print/Export
        Route::get('/{transferIssue}/print', [MaterialTransferIssueController::class, 'print'])->name('print');
    });

    // Direct Deliveries (Supplier to Project)
    Route::prefix('direct-deliveries')->name('direct-deliveries.')->group(function () {
        Route::get('/', [DirectDeliveryController::class, 'index'])->name('index');
        Route::get('/create', [DirectDeliveryController::class, 'create'])->name('create');
        Route::post('/', [DirectDeliveryController::class, 'store'])->name('store');
        Route::get('/{directDelivery}', [DirectDeliveryController::class, 'show'])->name('show');
        Route::get('/{directDelivery}/edit', [DirectDeliveryController::class, 'edit'])->name('edit');
        Route::put('/{directDelivery}', [DirectDeliveryController::class, 'update'])->name('update');
        Route::delete('/{directDelivery}', [DirectDeliveryController::class, 'destroy'])->name('destroy');
        Route::get('/{directDelivery}/print', [DirectDeliveryController::class, 'print'])->name('print');

        // Receipt processing
        Route::get('/{directDelivery}/receipt', [DirectDeliveryController::class, 'receipt'])->name('receipt');
        Route::post('/{directDelivery}/receipt', [DirectDeliveryController::class, 'processReceipt'])->name('process-receipt');

        // Approval workflow
        Route::post('/{directDelivery}/approve', [DirectDeliveryController::class, 'approve'])->name('approve');

        // Site return processing
        Route::get('/{directDelivery}/site-return', [DirectDeliveryController::class, 'siteReturn'])->name('site-return');
        Route::post('/{directDelivery}/site-return', [DirectDeliveryController::class, 'processSiteReturn'])->name('process-site-return');

        // AJAX endpoints
        Route::get('/project/{projectId}/divisions', [DirectDeliveryController::class, 'getProjectDivisions'])->name('project-divisions');
    });

    // Activity Log
    Route::prefix('activity-log')->name('activity-log.')->group(function () {
        Route::get('/', [ActivityLogController::class, 'index'])->name('index');
        Route::get('/{id}', [ActivityLogController::class, 'show'])->name('show');
        Route::delete('/cleanup', [ActivityLogController::class, 'cleanup'])->name('cleanup');
    });
});